#!/usr/bin/env perl
#
# Generator of MySQL configs and privileges
#
# $Id$

use strict;
use warnings;

use Data::Dumper;
use File::Basename;
use Getopt::Long;
use Sys::Hostname;

use constant ON		=> 1;
use constant OFF 	=> 0;

#-- Variables ----------------------------------------------------------

my $default_host = hostname;
my $default_instance = 'master';
my $default_outdir = '/var/tmp';

my $thiscmd = basename($0);
my $basedir = dirname($0);
my $tpldir = "$basedir/tpl";

my $out_map = {

	'wmc_hostdb'	=> {
		'cnf_main'	=> {
			'tpl'	=> 'wmc_my.cnf.tpl',
			'out'	=> 'my.cnf',
		},
		'cnf_local'	=> {
			'tpl'	=> 'wmc_my_local.cnf.tpl',
			'out'	=> 'my_local.cnf',
		},
		'priv'		=> {
			'tpl'	=> 'wmc_hostdb_priv.sql.tpl',
			'out'	=> 'my_priv.sql',
		},
	},

	'wmc_userdb'	=> {
		'cnf_main'	=> {
			'tpl'	=> 'wmc_my.cnf.tpl',
			'out'	=> 'my.cnf',
		},
		'cnf_local'	=> {
			'tpl'	=> 'wmc_my_local.cnf.tpl',
			'out'	=> 'my_local.cnf',
		},
		'priv'		=> {
			'tpl'	=> 'wmc_userdb_priv.sql.tpl',
			'out'	=> 'my_priv.sql',
		},
	},

	'socialnews_db'	=> {
		'cnf_main'	=> {
			'tpl'	=> 'socialnews_my.cnf.tpl',
			'out'	=> 'my.cnf',
		},
		'cnf_local'	=> {
			'tpl'	=> 'socialnews_my_local.cnf.tpl',
			'out'	=> 'my_local.cnf',
		},
		'priv'		=> {
			'tpl'	=> 'socialnews_priv.sql.tpl',
			'out'	=> 'my_priv.sql',
		},
	},

	'news_mf_db'	=> {
		'cnf_main'	=> {
			'tpl'	=> 'news_mf_my.cnf.tpl',
			'out'	=> 'my.cnf',
		},
		'cnf_local'	=> {
			'tpl'	=> 'news_mf_my_local.cnf.tpl',
			'out'	=> 'my_local.cnf',
		},
		'priv'		=> {
			'tpl'	=> 'news_mf_priv.sql.tpl',
			'out'	=> 'my_priv.sql',
		},
	},

	'sandbox_db'	=> {
		'cnf_main'	=> {
			'tpl'	=> 'sandbox_my.cnf.tpl',
			'out'	=> 'my.cnf',
		},
		'cnf_local'	=> {
			'tpl'	=> 'sandbox_my_local.cnf.tpl',
			'out'	=> 'my_local.cnf',
		},
		'priv'		=> {
			'tpl'	=> 'sandbox_priv.sql.tpl',
			'out'	=> 'my_priv.sql',
		},
	},

	'sepecharts_db'	=> {
		'cnf_main'	=> {
			'tpl'	=> 'sepecharts_my.cnf.tpl',
			'out'	=> 'my.cnf',
		},
		'cnf_local'	=> {
			'tpl'	=> 'sepecharts_my_local.cnf.tpl',
			'out'	=> 'my_local.cnf',
		},
		'priv'		=> {
			'tpl'	=> 'sepecharts_priv.sql.tpl',
			'out'	=> 'my_priv.sql',
		},
	},

	'subscription_db'	=> {
		'cnf_main'	=> {
			'tpl'	=> 'subscription_my.cnf.tpl',
			'out'	=> 'my.cnf',
		},
		'cnf_local'	=> {
			'tpl'	=> 'subscription_my_local.cnf.tpl',
			'out'	=> 'my_local.cnf',
		},
		'priv'		=> {
			'tpl'	=> 'subscription_priv.sql.tpl',
			'out'	=> 'my_priv.sql',
		},
	},

       'roboevents_db' => {
               'cnf_main'      => {
                       'tpl'   => 'roboevents_my.cnf.tpl',
                       'out'   => 'my.cnf',
               },
               'cnf_local'     => {
                       'tpl'   => 'roboevents_my_local.cnf.tpl',
                       'out'   => 'my_local.cnf',
               },
               'priv'          => {
                       'tpl'   => 'roboevents_priv.sql.tpl',
                       'out'   => 'my_priv.sql',
               },
       },

       'am_db' => {
               'cnf_main'      => {
                       'tpl'   => 'am_my.cnf.tpl',
                       'out'   => 'my.cnf',
               },
               'cnf_local'     => {
                       'tpl'   => 'am_my_local.cnf.tpl',
                       'out'   => 'my_local.cnf',
               },
               'priv'          => {
                       'tpl'   => 'am_priv.sql.tpl',
                       'out'   => 'my_priv.sql',
               },
       },

       'bm_db' => {
               'cnf_main'      => {
                       'tpl'   => 'bm_my.cnf.tpl',
                       'out'   => 'my.cnf',
               },
               'cnf_local'     => {
                       'tpl'   => 'bm_my_local.cnf.tpl',
                       'out'   => 'my_local.cnf',
               },
               'priv'          => {
                       'tpl'   => 'bm_priv.sql.tpl',
                       'out'   => 'my_priv.sql',
               },
       },

};

my $generic_hosts = {

	'wmc_hostdb'	=> {
		'master'	=> {
			'port'				=> 3306,
			'read_only'			=> OFF,
			'log_bin'			=> ON,
			'binlog_do_db'			=> ON,
			'replicate_do_db'		=> OFF,
			'innodb_buffer_pool_size'	=> '12G',
			'priv_host_rw'			=> ON,
			'priv_monitor'			=> ON,
			'priv_clean_rw'			=> ON,
			'priv_wmc_backup'		=> OFF,
		},
		'slave'	=> {
			'port'				=> 3307,
			'read_only'			=> ON,
			'log_bin'			=> OFF,
			'binlog_do_db'			=> OFF,
			'replicate_do_db'		=> ON,
			'innodb_buffer_pool_size'	=> '4G',
			'priv_host_rw'			=> OFF,
			'priv_monitor'			=> ON,
			'priv_clean_rw'			=> OFF,
			'priv_wmc_backup'		=> ON,
		},
	},

	'wmc_userdb'	=> {
		'master'	=> {
			'port'				=> 3306,
			'read_only'			=> OFF,
			'log_bin'			=> ON,
			'binlog_do_db'			=> ON,
			'replicate_do_db'		=> OFF,
			'innodb_buffer_pool_size'	=> '16G',
			'priv_user_rw'			=> ON,
			'priv_monitor'			=> ON,
			'priv_wmc_backup'		=> OFF,
		},
		'slave'	=> {
			'port'				=> 3306,
			'read_only'			=> ON,
			'log_bin'			=> OFF,
			'binlog_do_db'			=> OFF,
			'replicate_do_db'		=> ON,
			'innodb_buffer_pool_size'	=> '16G',
			'priv_user_rw'			=> OFF,
			'priv_monitor'			=> ON,
			'priv_wmc_backup'		=> ON,
		},
	},

	'socialnews_db'	=> {
		'master'	=> {
			'port'				=> 3307,
			'read_only'			=> OFF,
			'log_bin'			=> ON,
			'binlog_do_db'			=> ON,
			'replicate_do_db'		=> OFF,
			'priv_socialnews_rw'		=> ON,
			'priv_monitor'			=> ON,
		},
		'slave'	=> {
			'port'				=> 3307,
			'read_only'			=> ON,
			'log_bin'			=> OFF,
			'binlog_do_db'			=> OFF,
			'replicate_do_db'		=> ON,
			'priv_socialnews_rw'		=> OFF,
			'priv_monitor'			=> ON,
		},
	},

	'news_mf_db'	=> {
		'master'	=> {
			'port'				=> 3313,
			'read_only'			=> OFF,
			'log_bin'			=> ON,
			'binlog_do_db'			=> ON,
			'replicate_do_db'		=> OFF,
			'priv_news_mf_rw'		=> ON,
			'priv_monitor'			=> ON,
		},
		'slave'	=> {
			'port'				=> 3313,
			'read_only'			=> ON,
			'log_bin'			=> OFF,
			'binlog_do_db'			=> OFF,
			'replicate_do_db'		=> ON,
			'priv_news_mf_rw'		=> OFF,
			'priv_monitor'			=> ON,
		},
	},

	'sandbox_db'	=> {
		'master'	=> {
			'port'				=> 3310,
			'read_only'			=> OFF,
			'log_bin'			=> ON,
			'binlog_do_db'			=> ON,
			'replicate_do_db'		=> OFF,
			'priv_sandbox_rw'		=> ON,
			'priv_monitor'			=> ON,
			'priv_backup'			=> ON,
		},
		'slave'	=> {
			'port'				=> 3310,
			'read_only'			=> ON,
			'log_bin'			=> ON,
			'binlog_do_db'			=> ON,
			'replicate_do_db'		=> ON,
			'priv_sandbox_rw'		=> ON,
			'priv_monitor'			=> ON,
			'priv_backup'			=> ON,
		},
	},

	'sepecharts_db'	=> {
		'master'	=> {
			'port'				=> 3311,
			'read_only'			=> OFF,
			'log_bin'			=> ON,
			'binlog_do_db'			=> ON,
			'replicate_do_db'		=> OFF,
			'priv_sepecharts_rw'		=> ON,
			'priv_monitor'			=> ON,
			'priv_backup'			=> ON,
		},
		'slave'	=> {
			'port'				=> 3311,
			'read_only'			=> ON,
			'log_bin'			=> OFF,
			'binlog_do_db'			=> OFF,
			'replicate_do_db'		=> ON,
			'priv_sepecharts_rw'		=> OFF,
			'priv_monitor'			=> ON,
			'priv_backup'			=> ON,
		},
	},

	'subscription_db'	=> {
		'master'	=> {
			'port'				=> 3309,
			'read_only'			=> OFF,
			'log_bin'			=> ON,
			'binlog_do_db'			=> ON,
			'replicate_do_db'		=> OFF,
			'priv_subscription'		=> ON,
			'priv_monitor'			=> ON,
			'priv_backup'			=> ON,
		},
	},

	'roboevents_db' => {
		'master'	=> {
			'port'				=> 3312,
			'read_only'			=> OFF,
			'log_bin'			=> ON,
			'binlog_do_db'			=> ON,
			'replicate_do_db'		=> OFF,
			'priv_roboevents_rw'		=> ON,
			'priv_monitor'			=> ON,
		},
		'slave' => {
			'port'				=> 3312,
			'read_only'			=> ON,
			'log_bin'			=> OFF,
			'binlog_do_db'			=> OFF,
			'replicate_do_db'		=> ON,
			'priv_roboevents_rw'		=> OFF,
			'priv_monitor'			=> ON,
		},
	},

	'am_db' => {
		'master'	=> {
			'port'				=> 3305,
			'read_only'			=> OFF,
			'log_bin'			=> ON,
			'binlog_do_db'			=> ON,
			'replicate_do_db'		=> OFF,
			'priv_am_rw'			=> ON,
			'priv_monitor'			=> ON,
		},
	},

	'bm_db' => {
		'master'	=> {
			'port'				=> 3306,
			'read_only'			=> OFF,
			'log_bin'			=> ON,
			'binlog_do_db'			=> ON,
			'replicate_do_db'		=> OFF,
			'priv_bm_rw'			=> ON,
			'priv_monitor'			=> ON,
		},
	},

};

my $hosts = {

	#-- Project: WMC -----------------------------------------------

	'wmc-01.yandex.ru'	=> {
		'type'		=> 'wmc_userdb',
		'instances'	=> {
			'slave'		=> {
				'db'		=> 'robot',
				'server_id'	=> 348,
			},
		},
	},

	'wmc-02.yandex.ru'	=> {
		'type'		=> 'wmc_userdb',
		'instances'	=> {
			'master'	=> {
				'db'		=> 'robot',
				'server_id'	=> 388,
			},
		},
	},

	'wmc-03.yandex.ru'	=> {
		'type'		=> 'wmc_hostdb',
		'instances'	=> {
			'master'	=> {
				'db'		=> 'host_03',
				'server_id'	=> 393,
				'innodb_buffer_pool_size'	=> '16G',
			},
			'slave'		=> {
				'db'		=> 'host_05',
				'server_id'	=> 391,
			},
		},
	},

	'wmc-04.yandex.ru'	=> {
		'type'		=> 'wmc_hostdb',
		'instances'	=> {
			'master'	=> {
				'db'		=> 'host_04',
				'server_id'	=> 394,
			},
			'slave'		=> {
				'db'		=> 'host_06',
				'server_id'	=> 397,
			},
		},
	},

	'wmc-05.yandex.ru'	=> {
		'type'		=> 'wmc_hostdb',
		'instances'	=> {
			'master'	=> {
				'db'		=> 'host_05',
				'server_id'	=> 395,
			},
			'slave'		=> {
				'db'		=> 'host_03',
				'server_id'	=> 398,
			},
		},
	},

	'wmc-06.yandex.ru'	=> {
		'type'		=> 'wmc_hostdb',
		'instances'	=> {
			'master'	=> {
				'db'		=> 'host_06',
				'server_id'	=> 396,
			},
			'slave'		=> {
				'db'		=> 'host_04',
				'server_id'	=> 399,
			},
		},
	},

	'wmc-07.yandex.ru'	=> {
		'type'		=> 'wmc_hostdb',
		'instances'	=> {
			'master'	=> {
				'db'		=> 'host_07',
				'server_id'	=> 1038,
			},
			'slave'		=> {
				'db'		=> 'host_09',
				'server_id'	=> 1039,
			},
		},
	},

	'wmc-08.yandex.ru'	=> {
		'type'		=> 'wmc_hostdb',
		'instances'	=> {
			'master'	=> {
				'db'		=> 'host_08',
				'server_id'	=> 1040,
			},
			'slave'		=> {
				'db'		=> 'host_10',
				'server_id'	=> 1041,
			},
		},
	},

	'wmc-09.yandex.ru'	=> {
		'type'		=> 'wmc_hostdb',
		'instances'	=> {
			'master'	=> {
				'db'		=> 'host_09',
				'server_id'	=> 1042,
			},
			'slave'		=> {
				'db'		=> 'host_07',
				'server_id'	=> 1043,
			},
		},
	},

	'wmc-10.yandex.ru'	=> {
		'type'		=> 'wmc_hostdb',
		'instances'	=> {
			'master'	=> {
				'db'		=> 'host_10',
				'server_id'	=> 1044,
				'innodb_buffer_pool_size'	=> '10G',
			},
			'slave'		=> {
				'db'		=> 'host_08',
				'server_id'	=> 1045,
			},
		},
	},

	#-- Project: Social News ---------------------------------------

	'socialnews.dbs-00.yandex.ru'	=> {
		'type'		=> 'socialnews_db',
		'instances'	=> {
			'master'	=> {
				'db'		=> 'socialnews',
				'server_id'	=> 515,
			},
		},
	},

	'socialnews.dbs-01.yandex.ru'	=> {
		'type'		=> 'socialnews_db',
		'instances'	=> {
			'slave'	=> {
				'db'		=> 'socialnews',
				'server_id'	=> 359,
			},
		},
	},

	#-- Project: News MetaFetcher ----------------------------------

	'news_mf.dbs-00.yandex.ru'	=> {
		'type'		=> 'news_mf_db',
		'instances'	=> {
			'master'	=> {
				'db'		=> 'news_mf',
				'server_id'	=> 1487,
			},
		},
	},

	'news_mf.dbs-01.yandex.ru'	=> {
		'type'		=> 'news_mf_db',
		'instances'	=> {
			'slave'	=> {
				'db'		=> 'news_mf',
				'server_id'	=> 1488,
			},
		},
	},

	#-- Project: Sandbox -------------------------------------------

	'sandbox.dbs-00.yandex.ru'	=> {
		'type'		=> 'sandbox_db',
		'instances'	=> {
			'slave'	=> {
				'db'		=> 'sandbox',
				'server_id'	=> 1432,
			},
		},
	},

	'sandbox.dbs-01.yandex.ru'	=> {
		'type'		=> 'sandbox_db',
		'instances'	=> {
			'master'	=> {
				'db'		=> 'sandbox',
				'server_id'	=> 1433,
			},
		},
	},

	#-- Project: SEPE Charts ---------------------------------------

	'sepecharts.dbs-00.yandex.ru'	=> {
		'type'		=> 'sepecharts_db',
		'instances'	=> {
			'master'	=> {
				'db'		=> 'sepecharts',
				'server_id'	=> 1434,
			},
		},
	},

	'sepecharts.dbs-01.yandex.ru'	=> {
		'type'		=> 'sepecharts_db',
		'instances'	=> {
			'slave'	=> {
				'db'		=> 'sepecharts',
				'server_id'	=> 1435,
			},
		},
	},

	#-- Project: Subscription --------------------------------------

	'subscription.dbs-00.yandex.ru'	=> {
		'type'		=> 'subscription_db',
		'instances'	=> {
			'master'	=> {
				'db'		=> 'subscription',
				'server_id'	=> 512,
			},
		},
	},

	#-- Project: Robot events --------------------------------------

	'roboevents.dbs-00.yandex.ru'	=> {
		'type'		=> 'roboevents_db',
		'instances'	=> {
			'master'	=> {
				'db'		=> 'roboevents',
				'server_id'	=> 1462,
			},
		},
	},

	'roboevents.dbs-01.yandex.ru'	=> {
		'type'		=> 'roboevents_db',
		'instances'	=> {
			'slave'		=> {
				'db'		=> 'roboevents',
				'server_id'	=> 1463,
			},
		},
	},

	#-- Project: Authority Metrics (Mango) --------------------------

	'authority-metrics.mango001.yandex.ru'	=> {
		'type'		=> 'am_db',
		'instances'	=> {
			'master'	=> {
				'db'		=> 'authority_viewer',
				'server_id'	=> 1489,
			},
		},
	},

	#-- Project: Blending Metrics (Mango) --------------------------

	'blending-metrics.mango001.yandex.ru'	=> {
		'type'		=> 'bm_db',
		'instances'	=> {
			'master'	=> {
				'db'		=> 'BlendingMetrics',
				'server_id'	=> 1490,
			},
		},
	},

};


#-- Main ---------------------------------------------------------------

my ($host, $instance, $outdir, $get_master, $get_slaves, $list, $help);

GetOptions (
	'host|h=s'	=> \$host,
	'instance|i=s'	=> \$instance,
	'outdir|o=s'	=> \$outdir,
	'get-master|m'	=> \$get_master,
	'get-slaves|s=s'  => \$get_slaves,
	'list|l'	=> \$list,
	'help'		=> \$help,
) or usage();

if ($help) {
	usage();
}

if ($list) {
	list();
}

if ($get_slaves) {
	get_slaves();
}

check_h_options();

if ($get_master) {
	get_master();
}
else {
	gen_cnf();
}


#-- Subroutines --------------------------------------------------------

sub usage {
	printf STDERR "Usage: $thiscmd [-h [<project>.]<FQDN>] [-i <instance>] [-o <outdir>]\n";
	printf STDERR "       $thiscmd -m [-h [<project>.]<FQDN>] [-i <instance>]\n";
	printf STDERR "       $thiscmd [-s ARG] -l\n";
	printf STDERR "Options:\n";
	printf STDERR "  -h, --host           FQDN (and optionally project name) of target host\n";
	printf STDERR "  -i, --instance       MySQL instance (see /etc/rc.conf)\n";
	printf STDERR "  -o, --outdir         target dir for generation of configs\n";
	printf STDERR "  -l, --list           list available hosts and instances\n";
	printf STDERR "  -m, --get-master     get master for specified slave\n";
	printf STDERR "  -s, --get-slaves ARG get all slave servers for specified instance\n";
	printf STDERR "  -h, --help           show this help\n";
	exit 1;
}

sub error {
	printf STDERR join(' ', @_);
	usage();
}

sub list {
	my $_tpl = "%-36s%s\n";

	printf $_tpl, "HOST", "INSTANCES";

	foreach my $_host (sort keys %$hosts) {
		my $_instances = join " ", (sort keys %{$hosts->{$_host}->{instances}});
		printf $_tpl, $_host, $_instances;
	}

	exit 0;
}

sub check_h_options {
	$host = $host ? $host : $default_host;
	$instance = $instance ? $instance : $default_instance;
	$outdir = $outdir ? $outdir : $default_outdir;

	if (! exists $hosts->{$host}) {
		error "Unknown host $host !\n";
	}

	if (! exists $hosts->{$host}->{instances}->{$instance}) {
		error "Unknown instance $instance at $host !\n";
	}
}

sub gen_cnf {
	my $_type = $hosts->{$host}->{type};

	foreach my $_tpl_name (keys %{$out_map->{$_type}}) {
		my $_tpl = qq[$tpldir/$out_map->{$_type}->{$_tpl_name}->{tpl}];
		my $_out = qq[$outdir/$out_map->{$_type}->{$_tpl_name}->{out}];

		my $_replace = $generic_hosts->{$_type}->{$instance};

		foreach my $_key (keys %{$hosts->{$host}->{instances}->{$instance}}) {
			$_replace->{$_key} = $hosts->{$host}->{instances}->{$instance}->{$_key};
		}

		$_replace->{instance} = $instance;

		local (*TPL, *OUT);

		open(TPL, '<', $_tpl) or die "Can't open template $_tpl: $!\n"; 
		open(OUT, '>', $_out) or die "Can't open output file $_out: $!\n";

		while (my $_str = <TPL>) {
			# Handle boolean params
			if ($_str =~ /\$\{#(\w+)\}/) {
				my $_found_key = lc($1);
				if (! exists $_replace->{$_found_key}) {
					die "Not found replacement for \$\{#$_found_key\} in $_tpl !\n";
				}
				if ($_replace->{$_found_key} eq ON) {
					$_str =~ s/\$\{#(\w+)\}//;
				}
				else {
					next;
				}
			}

			# Substitute for other params
			while ($_str =~ /\$\{(\w+)\}/i) {
				my $_found_key = lc($1);
				if (! exists $_replace->{$_found_key}) {
					die "Not found replacement for \$\{$_found_key\} in $_tpl !\n";
				}
				my $_replace_str = $_replace->{$_found_key};
				$_str =~ s/\$\{\w+\}/$_replace_str/;
			}

			print OUT $_str;
		}

		close(OUT) or die "Can't close $_out: $!\n"; 
		close(TPL) or die "Can't close $_tpl: $!\n"; 
	}
}

sub get_master {
	if ($instance ne "slave") {
		error "There isn't master for instance $instance at $host !\n";
	}

	my $_replicated_db = $hosts->{$host}->{instances}->{$instance}->{db};

	foreach my $_host (sort keys %$hosts) {
		if (! exists $hosts->{$_host}->{instances}->{master}) {
			next;
		}
		my $_db = $hosts->{$_host}->{instances}->{master}->{db};
		if ($_db eq $_replicated_db) {
			my $_master_host = $_host;
			my $_master_type = $hosts->{$_host}->{type};
			my $_master_port = $hosts->{$_host}->{master}->{port} ||
				$generic_hosts->{$_master_type}->{master}->{port};
			my $_master_db = $_db;
			printf qq[master_host="%s"\n], $_master_host;
			printf qq[master_port=%d\n], $_master_port;
			printf qq[master_db="%s"\n], $_master_db
		}
	}
}

# Get all MySQL slaves for specified type
sub get_slaves {
	foreach my $_host (sort keys %$hosts) {
		if ($hosts->{$_host}->{type} eq $get_slaves) {
		# slave server exists
			if (exists $hosts->{$_host}->{instances}->{slave}) {
				print $_host . ":" . $generic_hosts->{$get_slaves}->{slave}->{port} . "/" . $hosts->{$_host}->{instances}->{slave}->{db} . "\n";
			}
		}
	}

	exit 0;
}
