#!/bin/sh -e
#
# Script for optimization of tables which have been created with
# innodb_file_per_table.
#
# $Id$

PATH=/bin:/usr/bin:/usr/local/bin
export PATH

#-- Subroutines --------------------------------------------------------

err()
{       
	local _exitval

	_exitval=$1
	shift

	echo 1>&2 "ERROR: $*"
	exit $_exitval
}

usage()
{
	echo 1>&2 "Usage: $0 [-e exclude_tbls_re] [-i instance] [databases]"
	echo 1>&2 "Options:"
	echo 1>&2 "  -e exclude_tbls_re    grep(1) regexp to exclude tables" \
		"from optimizing"
	echo 1>&2 "  -i instance           MySQL instance (the name of" \
		"rc-script in"
	echo 1>&2 "                        /usr/local/etc/rc.d)"
	exit 1
}

get_opts()
{
	local _opt

	while getopts "e:i:" _opt; do
		case "$_opt" in
			i) instance="$OPTARG" ;;
			e) exclude_tbls_re="$OPTARG" ;;
			*) usage ;;
		esac
	done

	shift $(($OPTIND - 1))

	dbs="$*"
}

handle_opts()
{
	local _db

	: ${instance:="$default_instance"}

	mysql_rc_script="${local_startup}/${instance}"
	if [ ! -x $mysql_rc_script ]; then
		err 1 "Executable RC script for instance ${instance} (${mysql_rc_script}) not found !"
	fi

	mysql_socket=$(${mysql_rc_script} get_socket)
	if [ ! -S "${mysql_socket}" ]; then
		err 1 "Socket for instance ${instance} (${mysql_socket}) not found !"
	fi

	mysql_defaults_file=$(${mysql_rc_script} get_defaults_file)
	if [ ! -r $mysql_defaults_file ]; then
		err 1 "Defaults file for instance ${instance} (${mysql_defaults_file}) not found !"
	fi

	mysql_run="mysql --defaults-file=${mysql_defaults_file} --socket=${mysql_socket} -NBAe"

	# Check DBs
	if [ -z "${dbs}" ]; then
		# If DBs not defined use all non excluded DBs
		dbs=$(${mysql_run} "SHOW DATABASES" | \
			grep -vwE "${exclude_dbs_re}")
	else
		for _db in $dbs; do
			if ! $mysql_run "SHOW DATABASES LIKE '${_db}'" | \
					grep -vwE "${exclude_dbs_re}" | \
					grep -wq "$_db"
			then
				err 1 "Database ${_db} not found !"
			fi
		done
	fi
}

check_depends()
{
	if [ $($mysql_run "SELECT @@${file_per_table_var}") -ne $file_per_table_req_val ]
	then
		err 1 "Value of \"${file_per_table_var}\" is not \"${file_per_table_req_val}\" !"
	fi
}

get_table_size()
{
	local _db _tbl _size

	_db="$1"
	_tbl="$2"

	_size=$($mysql_run "SHOW TABLE STATUS LIKE '${_tbl}'" $_db | \
		awk '{printf "%d", $7 + $9}' || \
		err 1 "Can't get status of table ${_tbl} !")

	echo $_size
}

optimize_tables()
{
	local _db _tbl _dt _tbl_orig_size _tbl_opt_size _tbl_diff_size

	_db="$1"

	printf ">>> DATABASE: %s\n" $_db

	for _tbl in $($mysql_run "SELECT table_name FROM information_schema.tables \
		WHERE table_schema='${_db}' AND engine = 'InnoDB'" || \
		err 1 "Can't get tables for database ${_db} !")
	do
		if echo $_tbl | grep -qE "${exclude_tbls_re}"; then
			printf ">>> %s skipped by exclude pattern (%s)\n" \
				$_tbl "${exclude_tbls_re}"
			continue
		fi
		_dt="[$(date "+%F %T")]"
		_tbl_orig_size=$(get_table_size ${_db} ${_tbl})
		printf "%s %-48s%-15d" "${_dt}" "${_tbl}" $_tbl_orig_size

		if ! $mysql_run "OPTIMIZE TABLE ${_tbl}" $_db >/dev/null; then
			err 1 "Can't optimize table ${_tbl} !"
		fi

		_tbl_opt_size=$(get_table_size ${_db} ${_tbl})
		_tbl_diff_size=$(($_tbl_opt_size - $_tbl_orig_size))
		printf "%d\n" $_tbl_diff_size
	done
}


#-- Variables ----------------------------------------------------------

local_startup="/usr/local/etc/rc.d"
default_instance="mysql"

date=$(date +%F)
log_dir="/place/mysql/optimize"

file_per_table_var="innodb_file_per_table"
file_per_table_req_val=1

exclude_dbs_re="information_schema|mysql|performance_schema|tmp"


#-- Main ---------------------------------------------------------------

trap 'echo "Cancel .."; exit 1' 2

get_opts $@
handle_opts

check_depends

log="${log_dir}/${instance}.${date}.log"
exec > $log

for db in $dbs; do
	optimize_tables $db
done

