#!/bin/sh

# This is a CLI for golem API's net_query function (interface for getting information about net).
# API function desctiption can be found at http://golem.yandex-team.ru/api/net_query.sbml
# Author: Korenevsky Denis (denkoren@yandex-team.ru)
#
# You are free to change this script if you're think, it will help other SEARCH admins.
# Please, don't forget to notify searchadm@ for changes.

set -u

##########################
#### GLOBAL VARIABLES ####
##########################



#########################
#### LOCAL VARIABLES ####
#########################

script_name="$(basename $0)"
script_dir="$(dirname $0)"
# Number of required parameters, specified after all options.
required_params=1

# Golem API availability flag
golem_available=false
# Search query text
search_pattern=""
# List of columns to show. ","-separated
columns_list="net_with_mask,vlan"
# Defines type of query (look golem/api/netquery.sbml for description).
search_type="q"
# Versions of IP to request information for. Specifes address families to show.
ip_ver="4,6"

#########################
#### LOCAL FUNCTIONS ####
#########################

help () {
	cat <<EndOfHelp | ${PAGER:-less}

NAME
	$script_name - CLI for net_query Golem API function

SYNOPSIS
	$script_name [-c columns_list] [-d|-v] [-4|-6] address
	$script_name [-c columns_list] [-d|-v] [-4|-6] -D fqdn
	$script_name [-c columns_list] [-d|-v] [-4|-6] -V vlan
	$script_name [-c columns_list] -a
	$script_name -h

DESCRIPTION
	Script trying to get information about networks using Golem API.
        Detailed information about API function can be found at
        http://golem.yandex-team.ru/api/net_query.sbml

OPTIONS
	-4	request only for IPv4 networks. Forces script to send Golem IPv4
		network information request only.
		Both of IPv4 and IPv6 are requested by default.

	-6	request only for IPv6 networks. Forces script to send Golem IPv6
		network information request only.
		Both of IPv4 and IPv6 are requested by default.

	-a	all networks. Print information about all networks.

	-c columns_list
		define columns list to show. Comma-separated.
                Default value: "net_with_mask,vlan"
		available values:
		    For IPv4 and IPv6 addresses:
			short_dc - DC short name
			id - network Golem ID
			name - network description
			ip - network IP address (integer)
			mask - netmask (integer)
			net_with_mask - ip_dd/mask for IPv4,
					ip_ch/mask for IPv6
			router_ip - network router IP address (integer)
			vlan - network VLAN number

		    For IPv4 addresses only:
			broadcast_ip - network broadcast address
			    (integer)
			broadcast_ip_dd - network broadcast address
			    (dot-decimal)
			ip_dd - network IP address (dot-decimal)
			mask_dd - netmask (dot-decimal)
			router_ip_dd - network router IP (dot-decimal)
			tags - network tags (comma-separated)

		    For IPv6 addresses only:
			ip_ch - network IP address (colon-hexadecimal)
			mask_ch - netmask (colon-hexadecimal)
			router_ip_ch - router IP address
			    (colon-hexadecimal)

		For exapmple:
		    -c net_with_mask,vlan
		will print:
		<dot-decimal address/netmask>	<vlan number>
		(tab-separated)

	-d	debug mode. Enables printing of debug messages.

	-D	all direct connected networks. Print information about all
		networks, directly connected to host, specified in
		search_pattern.

	-h	show this help and exit.

	-v	verbose mode. Enables printing of workprocess reports.

	-V	all networks, with specified VLAN number. Print information
		about all networks with VLAN number, specified in parameter.

PARAMETERS
	address
		IP address (dot-decimal for IPv4 or сolon-hexadecimal for IPv6)
		or FQDN, to search network.
		    Examples:
			178.154.220.81
			ws5-000.yandex.ru
			2a02:6b8:0:a02::5d9e:9421

	fqdn	host FQDN to search all networks, directly connected to.
		    Example:
			ws30-200.yandex.ru

	vlan	vlan number to search networks, belonging to.
		    Example:
			761

EXAMPLES
	Get ws5-000.yandex.ru's native vlan, based on it's IP-address.
	Get IP from DNS:
	    ${script_name} -c vlan ws5-000.yandex.ru

	Get ws5-000.yandex.ru's all vlans
	    ${script_name} -c vlan -D ws5-000.yandex.ru
	    ${script_name} -c vlan -D ws5-000.fb.yandex.ru

EXIT CODES
	0 on success

	not 0 on error (exit code conform to wget exit code)

BSD				March 15, 2013				    BSD

EndOfHelp
}

# Print a short script usage description
reference ()
{
	cat <<EndOfReference

	$script_name requires ${required_params} parameters
	    $script_name search_pattern

	For verbose help use -h flag:
	    $script_name -h

EndOfReference
}

. "${script_dir}/messages.sh"
. "${script_dir}/functions.sh"

#################################################
#### TESTING OPTIONS, CONFIGURING CURREN RUN ####
#################################################

while getopts "46ac:dDhvV" opt; do
	case $opt in
	"4")
		ip_ver="4"
		;;
	"6")
		ip_ver="6"
		;;
	a)
		search_type="all"
		required_params="0";
		search_pattern="y"
		;;
	c)
		columns_list=${OPTARG}
		;;
	d)
		msg_verbose_flag=true
		msg_debug_flag=true
		;;
	D)
		search_type="all-direct-nets=y&q"
		;;
	h)
		help
		exit 0
		;;
	v)
		msg_verbose_flag=true
		;;
	V)
		search_type="vlan"
		;;
	\:)
		critical 64 "Option -${OPTARG} requires an argument"
		;;
	\?)
		reference
		exit 64
		;;
	esac
done
shift $(($OPTIND - 1))

if [ "$#" -lt "$required_params" ]; then
	reference
	exit 64
fi

#######################
#### SCRIPT'S BODY ####
#######################

if [ -z "${search_pattern}" ]; then
	search_pattern="${1}"
fi

debug "Checking Golem API availability..."

fn_check_availability "ro.admin.yandex-team.ru" "80" \
    && golem_available=true

if ${golem_available}; then
	debug "Golem API is available."
else
	critical 74 "Golem API is not available."
fi

verbose "Requesting Golem for ${search_pattern}'s information." \
    "Following fields will be requested: ${columns_list}."
wget -q -O - "http://ro.admin.yandex-team.ru/api/net_query.sbml?ip_versions=${ip_ver}&${search_type}=${search_pattern}&columns=${columns_list}"
exitcode="$?"

exit "${exitcode}"
