#!/usr/bin/env bash

DER_FILE="yandex.tld"
OCSP_directory_path="/usr/local/etc/ocsp/"
timestamp=`date '+%s'`

isFile() {
    if [ ! -f $1 ]; then
        echo "Error: File $1 not found"
	exit 1
    fi
}

reloadBalancer() {
    for PORT in ${PORTS[@]}; do
        curl -so /dev/null "http://127.0.0.1:${PORT}/admin/events/call/reload_ocsp"
    done
}

reloadNginx() {
    for PID in ${PIDS[@]}; do
        kill -HUP `cat ${PID}`
    done
}

PIDS=()
PORT=()
while getopts "t:p:b:c:P:s:h" opt; do
    case $opt in
        t)  TYPE=${OPTARG}
        ;;
        p)  PIDS+=( "${OPTARG}" )
        ;;
        b)  server_cert=${OPTARG}
        ;;
        c)  CA_cert=${OPTARG}
        ;;
        P)  PORTS+=( "${OPTARG}" )
        ;;
        s)  SUFFIX=${OPTARG}
        ;;
        h)  echo "Not implemented (as usual)"
        ;;
        *)  echo "Unknown key. Run '$0 -h' for help"
            exit 1
    esac
done

OPENSSL=$(command -v openssl)
if [ "$?" -ne 0 ]; then
    echo "Error: openssl binary file not found"
    exit 1
fi

if [ -z ${CA_cert} ]; then
    CA_cert="/usr/local/etc/CER/CA-yandex.pem"
fi

if [ -z ${server_cert} ]; then
    server_cert="/usr/local/etc/CER/CAbundle.pem"
fi

# sanity checks
if [ "${TYPE}" != "nginx" ] && [ "${TYPE}" != "balancer" ]; then
    echo "Error: unknown webserver type"
    exit 1
fi

if [ -z ${PORTS} ] && [ "${TYPE}" == "balancer" ]; then
     echo "Error: unknown port for required instance (-P)"
     exit 1
fi

if [ -n ${SUFFIX} ]; then
     DER_FILE="${DER_FILE}_${SUFFIX}"
fi

for file in ${PIDS[@]} ${CA_cert} ${server_cert}; do
    isFile ${file}
done

OCSP_responder_URL=$(openssl x509 -noout -in ${server_cert} -ocsp_uri)

if [ -z ${OCSP_responder_URL} ]; then
    echo "Error: couldn't determine OCSP responder URL"
    exit 1
fi

[ -d ${OCSP_directory_path} ] || mkdir -p ${OCSP_directory_path}

# get response from CA responder
$OPENSSL ocsp -CAfile ${server_cert} -issuer ${CA_cert} -cert ${server_cert} -url ${OCSP_responder_URL} -respout ${OCSP_directory_path}/${DER_FILE}-${timestamp}.der 2>&1 | grep -q "OK" 2>/dev/null
if [ "$?" -ne "0" ]; then
    echo "Error: unexpected error from openssl"
    exit 1
fi

# set new ticket as active
ln -sf ${OCSP_directory_path}/${DER_FILE}-${timestamp}.der ${OCSP_directory_path}/${DER_FILE}.der

# rotate old ssl tickets except active one
basename_resolved_symlink_response=$( basename `readlink -f ${OCSP_directory_path}/${DER_FILE}.der` )
find ${OCSP_directory_path} -type f -name "${DER_FILE}*" -and -not -name "${basename_resolved_symlink_response}" -mtime +4 -delete

# reread ssl ticket file in webservers
if [ "${TYPE}" = "nginx" ]; then
    reloadNginx
else
    reloadBalancer
fi

