#!/bin/sh

# This is script for opening server's console using javaws.
# It can open KVM and IPMI console using host's FQDN or KVM console by
# Author: Denis Korenevsky <denkoren@yandex-team.ru>
# http://wiki.yandex-team.ru/SEEK/Instruments/scripts/openkvm
#
# You are free to change this script if you're think, it will help other SEARCH
# admins.

##########################
#### GLOBAL VARIABLES ####
##########################

set -u

# Golem cookie file path. Used to get .jnlp from Golem.
: ${GOLEM_COOKIE_FILE:="${HOME}/.golem_cookie"}

# Folder to store KVM console .jnlp files. $TMPDIR/kvms by default.
# If TMPDIR not set, then /tmp/kvms will be used.
: ${JNLP_FILES_FOLDER:="${TMPDIR:-"/tmp/"}/kvms"}

# Default domain for specified host.
: ${DEFAULT_DOMAIN:="yandex.ru"}

#########################
#### LOCAL VARIABLES ####
#########################

script_name="$(basename $0)"
script_dir="$(dirname $0)"
# Protection boolean for "dry run" activation.
real_run=true
# Number of required parameters, specified after all options.
required_params=1

# KVM address to open: XXX-N/NN-NN
kvm_target=""
# File to save .jnlp given by Golem.
jnlp_kvm_file=""
# Requet type. URI with parameter name.
request_type="launch-kvm.sbml?kvm"
# Flag to inform, whether KVM format checks needed.
skip_checks=false
# Java Web Start binary name.
java_web_start="javaws"
# Browser
browser="open"

# In MacOS X we have nice utility "open"
uname | grep -qi darwin && java_web_start="open"

#########################
#### LOCAL FUNCTIONS ####
#########################

help () {
	cat <<EndOfHelp | ${PAGER:-less}

NAME
	$script_name - shell script for opening KVM consoles.

SYNOPSIS
	$script_name [-d] [-v] [-n] kvm_address
	$script_name -g
	$script_name -h

DESCRIPTION
	This is a /bin/sh script, used to start KVM java consoles using
	Java Web Start application. It requires access to Golem through HTTPs.
	It requests https://golem.yandex-team.ru/launch-kvm.sbml or
	https://golem.yandex-team.ru/launch-ipmi.sbml and launches Java Web
	Start application on derived .jnlp file.
	Description also avalible at wiki:
	http://wiki.yandex-team.ru/SEEK/Instruments/scripts/openkvm

OPTIONS
	-h	show this help and exit.

	-d	debug mode. Enables verbose and debug messages.

	-g	generate new Golem authorization cookie. Save it to file,
		specified in GOLEM_COOKIE_FILE environment variable.
		Default: \${HOME}/.golem_cookie

	-n	dry run. Just get .jnlp and save it to disk.

	-q	quiet mode. Suppress all messages except ERROR ones.

	-v	verbose mode. Enables verbose messages.

PARAMETERS
	kvm_address
		address of KVM to open or host's FQDN for IPMI servers.
		Address of KVM may contain any punctuation and space symbols.
		There are just 2 requirements:
		    * it must contain 3 alpabet symbols at the begin.
		    * it must contain 4 or 5 numbers.
		Script transforms kvm_address and tries to bring it to canonical
		shape (XXX-N/NN-NN). So next parameter values will be tranfromed
		to the following shape:
		    ugr2\ 02  12	->	UGR-2/02-12
		    myt 3 113		->	MYT-3/11-3
		    Iva-21111		->	IVA-2/11-11
		If kvm_address was given in canonical form it will not
		be transformed.
		So ugr-2/2-1 is correct KVM console address.

VARIABLES
	Script uses following variables to control it's behaviour

	GOLEM_COOKIE_FILE
		Golem authorization cookie file path. Script accepts
		'wget' cookie file format.
		Default: \${HOME}/.golem_cookie

	JNLP_FILES_FOLDER
		Path to folder to store .jnlp files.
		Default: /tmp/kvms/

	DEFAULT_DOMAIN
		Default domain for hosts. Is appended to host name only if no
		'.' found.
		Default: yandex.ru

EXAMPLES
	$script_name ugr-4/10 07
	will open console with address UGR-4/10-07

	$script_name ws36-000.yandex.ru
	will open console for server ws36-000.yandex.ru using IPMI.

BSD				February 15, 2013			    BSD

EndOfHelp
}

# Print a short script usage description
reference ()
{
	cat <<EndOfReference

	$script_name requires at least ${required_params} parameters
	    $script_name kvm_address

	For verbose help use -h flag:
	    $script_name -h

EndOfReference
}

. "${script_dir}/messages.sh"
. "${script_dir}/functions.sh"

get_new_cookie() {
	local golem_login
	local golem_password

	warning "Getting Golem authorization cookie."
	printf "Golem login name [${USER}]: "
	read golem_login
	if [ -z "${golem_login}" ]; then
		golem_login="${USER}"
	fi
	printf "Password: "
	stty -echo && read -r golem_password; stty echo
	printf "\n"

	: > "${GOLEM_COOKIE_FILE}"
	chmod 0600 "${GOLEM_COOKIE_FILE}"

	curl -s -k -o /dev/null -L \
	    --connect-timeout 3 \
	    -b ${GOLEM_COOKIE_FILE} \
	    -c ${GOLEM_COOKIE_FILE} \
	    -d "retpath=https://golem.yandex-team.ru/" \
	    -d "login=${golem_login}" \
	    -d "passwd=${golem_password}" \
	    "https://passport.yandex-team.ru/auth"

	# On successful cookie generation strings in cookie file will contain
	# <user>:<hash> data. So we look for user name in file to check the cookie
	# is correct.
	grep -q "${golem_login}" "${GOLEM_COOKIE_FILE}" \
	    && return 0

	return 1
}

kvm_check() {

	local _kvm_address

	_kvm_address="${1}"

	debug "Checking target KVM address for correctness."
	if printf "${_kvm_address}" \
	    | egrep -q \
	    "[A-Z]{3}-[[:digit:]]/[[:digit:]]{1,2}-[[:digit:]]{1,2}"; then
		debug "KVM address '${_kvm_address}' is correct."
		return 0
	else
		debug "KVM address '${_kvm_address}' is not correct."
		return 1
	fi
}

# Transform kvm to canonical form ( LLL-N/NN-NN )
kvm_transform() {

	local _kvm_address

	_kvm_address="${1}"

	if printf "${_kvm_address}" \
	    | grep -q 'UGR2\.'; then
		printf "${_kvm_address}" \
		    | sed 's/UGR2\.-[1-2]/UGR-2/'

	else
		printf "${_kvm_address}" \
		    | tr -d "[:punct:]" \
		    | sed -E 's|^([A-Z]{3})([0-9])([0-9]{2})|\1-\2/\3-|'
	fi


}

# Encode KVM address to 'urlencoded' form: ( LLL-N%20NN-NN )
kvm_encode() {

	local _kvm_address

	_kvm_address="${1}"

	printf "${_kvm_address}" | sed 's#/#%2F#'

}

##################################################
#### TESTING OPTIONS, CONFIGURING CURRENT RUN ####
##################################################

while getopts "dghqvn" opt; do
	case $opt in
	d)
		msg_verbose_flag=true
		msg_debug_flag=true
		;;
	g)
		while ! get_new_cookie; do
			error "Uncorrect login or password given." \
			    "Please, try again."
		done
		exit 0
		;;
	h)
		help
		exit
		;;
	q)
		msg_warning_flag=false
		msg_verbose_flag=false
		msg_debug_flag=false
		;;
	v)
		msg_verbose_flag=true
		;;
	n)
		real_run=false
		;;
	\:)
		critical 1 "Option -${OPTARG} requires an argument"
		;;
	\?)
		reference
		exit 1
		;;
	esac
done
shift $(($OPTIND - 1))

if [ "$#" -lt "$required_params" ]; then
	reference
	exit 1
fi

#######################
#### SCRIPT'S BODY ####
#######################

# Check Golem cookie file for readablility. Nothing to do without it.
if [ ! -r ${GOLEM_COOKIE_FILE} ]; then
	critical 1 "Golem authorization cookie not found at" \
	    "'${GOLEM_COOKIE_FILE}' or is not readable." \
	    "To generate it use ${script_name} -g." \
	    "To point another file use GOLEM_COOKIE_FILE environment variable."
fi
# Create folder to store .jnlp files if none exists yet.
test -d "${JNLP_FILES_FOLDER}" || mkdir -p "${JNLP_FILES_FOLDER}"

# Inform user about dry run.
! $real_run \
    && printf "${_msg_LRed}Dry run. " \
    && printf "Script will just save .jnlp file.${_msg_Normal}\n"

# All parameters specified are parts of target KVM address.
# All literas at address must be uppercase.
kvm_target="$( printf "${*}" \
    | tr "[:lower:]" "[:upper:]" \
    | tr -d "[:space:]" )"
# Or we have just one parameter with host's name or FQDN.
host="${1}"

if ! echo "${host}" | grep -q '\.'; then
	host="${host}.${DEFAULT_DOMAIN}"
fi

debug "Checking parameters: KVM address or hostname?"
if fn_is_real_host "${host}"; then
	# If it is real host, get it IP address
	verbose "${host} seems to be a real host"

	debug "Trying to get KVM address using Golem API."
	kvm_target="$( "${script_dir}/host_query.sh" -q -c kvm "${host}" )"
        exitcode="$?"

	if [ "${exitcode}" -ne "0" ]; then
		warning "Golem API request error. Can't define host's KVM" \
		    "address"
		kvm_target=""
	fi

	if [ -z "${kvm_target}" ]; then
		verbose "Host migth have IPMI." \
		    "Trying to open IPMI using Golem API."
		kvm_target="${host}"
		jnlp_kvm_file="${JNLP_FILES_FOLDER}/${kvm_target}.jnlp"
		request_type="launch-ipmi.sbml?h"
		skip_checks=true
	fi

fi

if ! ${skip_checks}; then

	verbose "${kvm_target} seems to be KVM port address."
	if ! kvm_check "${kvm_target}"; then
		debug "Address format incorrect. Trying to transform."
		kvm_target=$( kvm_transform "${kvm_target}" )
		debug "Address after transformation: ${kvm_target}"
	fi

	if kvm_check "${kvm_target}"; then
		# '/' symbol at KVM port will points to fake folder.
		# So, we replace it with '_' symbol
		jnlp_kvm_file="${JNLP_FILES_FOLDER}/$(
		    printf "${kvm_target}" | sed "s#/#_#g" ).jnlp"
		# Golem API requires urlencoded string in request.
		kvm_target="$( kvm_encode "${kvm_target}" )"
	else
		critical 1 "KVM address is wrong. It should contain 3-letters" \
		    "abbreviation of DC and at least 4 digits of console" \
		    "address. Delimiters has no matter. Case insensitive." \
		    "For example both of 'urg-2/02-01' and 'ugr 2 \ 02 1'" \
		    "are correct."
	fi

fi

debug "Getting .jnlp file from Golem."

curl -k -s -o "${jnlp_kvm_file}" \
    --connect-timeout 2 \
    -b "${GOLEM_COOKIE_FILE}" \
    -c "${GOLEM_COOKIE_FILE}" \
    "https://golem.yandex-team.ru/${request_type}=${kvm_target}" 

debug "File saved to ${jnlp_kvm_file}."

debug "Performing simple .jnlp file correctness check."
if grep 'Невозможно авторизоваться на DSView<br' ${jnlp_kvm_file} >&2; then
	critical 2 "Golem_session error." \
	"If the error occurred for the first time - try again." \
	"If this error occurs constantly - something is not right."
fi
if grep -e "<div \|pass.yandex-team.ru" ${jnlp_kvm_file} >&2; then
	critical 2 "Golem request error. Look for details above." \
	"Possibly your Golem authorization cookie is spoiled." \
	"To renew it use '${script_name} -g' command."
fi

if ${real_run}; then
	# bin/echo is used here because jnlp_kvm_file can contain %N symbol
	# sequence, so printf interprets it as formatter.
	if grep -q "<html>" "${jnlp_kvm_file}"; then
		mv "${jnlp_kvm_file}" "${jnlp_kvm_file}.html"
		jnlp_kvm_file="${jnlp_kvm_file}.html"
		open_command="${browser}"
		printf "Starting ${jnlp_kvm_file} using browser.\n"
	else
		printf "Starting ${jnlp_kvm_file} using java web start. "
		printf "Please, be patient, JavaWS can start very slowly :(.\n"
		open_command="${java_web_start}"
	fi
	${open_command} "${jnlp_kvm_file}"
fi

