#!/bin/sh -e
#
# Script for deploying Orange releases.
#
# $Header$

PATH=/bin:/sbin:/usr/bin:/usr/sbin:/usr/local/bin:/usr/local/sbin:/Berkanavt/bin/scripts
export PATH

#-- Subroutines --------------------------------------------------------

err()
{
	local _exitval

	_exitval=$1
	shift

	echo 1>&2 "ERROR: $*"
	exit $_exitval
}

verbose()
{
	local _d

	if [ -n "${verbose}" ]; then
		_d=$(date "+%F %T")
		echo "[${_d}] VERBOSE: $*"
	fi
}

usage()
{
	echo 1>&2 "Usage: ${thiscmd} -i <instance> [-u <url>] [-sv] <target>"
	echo 1>&2 "       ${thiscmd} -l"
	echo 1>&2 "Options are:"
	echo 1>&2 "  -i <instance>         instance of orange (ru, us)"
	echo 1>&2 "  -l                    list available instances"
	echo 1>&2 "  -s                    use stdrestart instead of restart"
	echo 1>&2 "  -u <url>              release archive URL" \
		"(rsync://<host>/<path>)"
	echo 1>&2 "  -v                    verbose output"
	echo 1>&2 "Targets are:"
	echo 1>&2 "  get_rel               get archive from <url> onto a" \
		"copy host"
	echo 1>&2 "  cp_to_all             copy archive from the copy host" \
		"to whole cluster"
	echo 1>&2 "  extract_rel           extract release on whole cluster"
	echo 1>&2 "  deploy                meta for get_rel, cp_to_all," \
		"extract_rel"
	echo 1>&2 "  switch                switch to a new release"
	echo 1>&2 "  check                 check cluster"
	echo 1>&2 "  check_links           check release symlink on a cluster"
	echo 1>&2 "  full                  meta for deploy, switch, check"

	exit 1
}

add_to_cleanup()
{
	cleanup_files="${cleanup_files}${cleanup_files:+ }$*"
}

mk_tmp_file()
{
	local _var _tmp_file
	_var="$1"
	_tmp_file=$(mktemp -t $thiscmd) || err 1 "Can't make temporaty file !"
	add_to_cleanup $_tmp_file
	eval $_var=\"\$_tmp_file\"
}

get_opts()
{
	local _opt

	while getopts "i:u:lsv" _opt; do
		case "$_opt" in
			i) instance="${OPTARG}" ;;
			l) mode="list" ;;
			u) rel_url="${OPTARG}" ;;
			s) stdrestart="stdrestart" ;;
			v) verbose="yes" ;;
			*) usage ;;
		esac
	done

	shift $(($OPTIND - 1))

	if [ -n "$1" ]; then
		target="$1"; shift
	fi

	if [ $# -ne 0 ]; then
		usage
	fi
}

check_instance()
{
	if [ -z "${instance}" -a "${mode}" = "${default_mode}" ]; then
		usage
	fi
}

check_rel_url()
{
	if [ -z "${rel_url}" ]; then
		usage
	fi
}

check_opts()
{
	: ${mode:=${default_mode}}
	check_instance
	case "${target}" in
		get_rel|cp_to_all|extract_rel|deploy|switch) check_rel_url ;;
	esac
}

set_yr_vars()
{
	local _yr_var _ctl_var

	for _yr_var in gr_list yr_gr_all gr_cp; do
		eval $_yr_var=\"\$${instance}_${_yr_var}\"
		eval _ctl_var=\"\$${_yr_var}\"
		if [ -z "${_ctl_var}" ]; then
			err 1 "Can't get \$${instance}_${_yr_var} !" \
				"Probably instance is incorrect."
		fi
	done

	verbose "Instance: ${instance}."

}

set_cp_host()
{
	eval yr_gr_cp=\"\$${instance}_yr_gr_${gr_cp}\"
	h_cp=$(yr $yr_gr_cp LIST | sort | head -1)

	if [ -z "${h_cp}" ]; then
		err 1 "Can't get host for copy operations !"
	fi

	verbose "Host for copy operations: ${h_cp}."
}

set_rel_vars()
{
	rel_arch="${rel_url##*/}"
	rel="${rel_arch%${arch_sfx}}"
	rel_path="${rel_tmp_dir}/${rel_arch}"

}

set_orange_vars()
{
	eval orange_dir=\"\$${instance}_orange_dir\"
	: ${orange_dir:=${default_orange_dir}}
	eval orange_user=\"\$${instance}_orange_user\"
	: ${orange_user:=${default_orange_user}}
	eval orange_group=\"\$${instance}_orange_group\"
	: ${orange_group:=${default_orange_group}}
	eval orange_rel_link=\"\$${instance}_orange_rel_link\"
	: ${orange_rel_link:=${default_orange_rel_link}}
	eval orange_ok_re=\"\$${instance}_orange_ok_re\"
	: ${orange_ok_re:=${default_orange_ok_re}}
}

set_vars()
{
	set_yr_vars
	set_cp_host
	set_rel_vars
	set_orange_vars
}

recognize_rel_url()
{
	# Easy heuristic to recognize what kind of URL we have
	case "${rel_url}" in
		rsync://*|*::*)
			get_rel_cmd="rsync -t ${rel_url} ."
			;;
		http://*)
			get_rel_cmd="_get_rel_cmd=\$(${get_http_client}) && \
				\$_get_rel_cmd ${rel_url}"
			;;
		*:*)
			get_rel_cmd="scp ${rel_url} ."
			;;
		*)
			err 1 "Can't recognize scheme of ${rel_url} !"
	esac
	verbose "Use \"${get_rel_cmd}\" to get release."
}

get_rel()
{
	recognize_rel_url
	verbose "Getting release from ${rel_url} .."
	if ! ssh -l root $h_cp "cd ${rel_tmp_dir} && \
		${get_rel_cmd} && md5=\$(${get_md5}) && \
		\${md5} ${rel_arch}"
	then
		err 1 "Can't get release on ${h_cp} !"
	fi
	verbose "${rel_arch} is got successfully."
}

cp_to_part()
{
	local _dsts _srcs _srcs_num _get_src_cmd _yr_log

	_dsts="$*"

	if [ -z "${wo_list}" ]; then
		_get_src_cmd="src=\"${h_cp}\""
	else
		_srcs=$(echo $wo_list | sed -Ee 's/^\-//; s/ \-/ /g')
		_srcs_num=$(echo $_srcs | tr ' ' '\n' | grep -c "")
		_get_src_cmd="rnd=\$(jot -r 1 1 $_srcs_num) && \
			src=\$(echo $_srcs | tr ' ' '\\n' | \
			sed -ne \"\${rnd}p\")"
	fi

	mk_tmp_file _yr_log

	verbose "Copying release to ${_dsts}."

	if ! yr $_dsts / \=\= "${_get_src_cmd} && \
		rsync -ae ssh \${src}${cp_domain}:${rel_path} ${rel_path} && \
		echo \"copied\"" | \
		tee $_yr_log
	then
		err 1 "Can't copy release to ${dst} !"
	fi

	if grep -q "${yr_err_re}" $_yr_log; then
		err 1 "Copying failed on some hosts !"
	fi
}

reduce_cp()
{
	local _list _num _cur_factor

	_list="$*"
	_num=$(yr $_list LIST | grep -c "")
	_cur_factor=$((${_cur_factor:-1} * $cp_factor))

	if [ $_num -le $_cur_factor ]; then
		cp_to_part $_list
	else
		w_list=$(yr $_list LIST | sort | head -$_cur_factor \
			| sed -e 's/^/+/' | tr '\n' ' ')
		cp_to_part $w_list $wo_list
		wo_list=$(echo $w_list | sed -e 's/\+/-/g')
		reduce_cp $_list $wo_list
	fi
}

cp_to_all()
{
	reduce_cp $yr_gr_all
}

extract_rel()
{
	local _yr_log

	mk_tmp_file _yr_log

	verbose "Extracting release on whole cluster."

	if ! yr $yr_gr_all /90 \=\= "cd ${orange_dir} && tar zxf ${rel_path} && \
		chown -R ${orange_user}:${orange_group} ${rel} && \
		chmod -R ug+w,o-w ${rel} && rm -f ${rel_path} && \
		echo \"extracted\"" | \
		tee $_yr_log
	then
		err 1 "Can't extract release !"
	fi

	if grep -q "${yr_err_re}" $_yr_log; then
		err 1 "Extracting failed on some hosts !"
	fi
}

deploy()
{
	get_rel
	cp_to_all
	extract_rel
}

switch()
{
	local _gr _yr_gr _yr_log _orange_rst_int _orange_rst_by
	local _orange_ctl_freebsd _orange_ctl_linux _get_orange_ctl

	for _gr in $gr_list; do
		eval _yr_gr=\"\$${instance}_yr_gr_${_gr}\"
		if [ -z "${_yr_gr}" ]; then
			err 1 "Can't get yr group for ${_gr} group !" \
				"Probably \$${instance}_yr_gr_${_gr} is" \
				"not set."
		fi

		eval _orange_rst_int=\"\$${instance}_orange_rst_int_${_gr}\"
		: ${_orange_rst_int:=${default_orange_rst_int}}
		eval _orange_rst_by=\"\$${instance}_orange_rst_by_${_gr}\"
		: ${_orange_rst_by:=${default_orange_rst_by}}
		eval _orange_rst_stime=\"\$${instance}_orange_rst_stime_${_gr}\"
		: ${_orange_rst_stime:=${default_orange_rst_stime}}
		eval _orange_ctl_freebsd=\"\$${instance}_orange_ctl_freebsd_${_gr}\"
		: ${_orange_ctl_freebsd:=${default_orange_ctl_freebsd}}
		eval _orange_ctl_linux=\"\$${instance}_orange_ctl_linux_${_gr}\"
		: ${_orange_ctl_linux:=${default_orange_ctl_linux}}

		_get_orange_ctl="uname -s | sed \
			-e 's,freebsd,${_orange_ctl_freebsd},I' \
			-e 's,linux,${_orange_ctl_linux},I'"

		mk_tmp_file _yr_log

		verbose "Switching ${_yr_gr} .."

		if ! yr $_yr_gr $_orange_rst_by \=\= "\
			orange_ctl=\$(${_get_orange_ctl}) && \
			cd ${orange_dir} && \
			if [ -z \"${stdrestart}\" ]; then \
				\$orange_ctl stop; \
				sleep ${_orange_rst_stime}; \
			fi && \
			rm ${orange_rel_link} && \
			ln -sv ${rel} ${orange_rel_link} && \
			if [ -z \"${stdrestart}\" ]; then \
				\$orange_ctl start; \
			else \
				\$orange_ctl ${stdrestart}; \
			fi && \
			echo \"switched\" && \
			sleep ${_orange_rst_int}" | \
			tee $_yr_log
		then
			err 1 "Can't switch release on ${_yr_gr} !"
		fi

		if grep -q "${yr_err_re}" $_yr_log; then
			err 1 "Switching failed on some hosts !"
		fi

		verbose "Switching ${_yr_gr} is done."
	done
}

check()
{
	verbose "Checking ${yr_gr_all} .."
	if ! yr $yr_gr_all / \=\= "\
		sed -nEe '/${orange_ok_re}/,$ p' ${orange_dir}/start.out | \
		egrep . || { echo failed; exit 1; }"
	then
		err 1 "Can't check orange on ${yr_gr_all} !"
	fi
	verbose "Checking ${yr_gr_all} is done."
}

check_links()
{
	verbose "Checking links on ${yr_gr_all} .."
	if ! yr $yr_gr_all / \=\= "readlink \
		${orange_dir}/${orange_rel_link} | grep -q ${rel} || \
		{ echo failed; exit 1; }"
	then
		err 1 "Can't check links on ${yr_gr_all} !"
	fi
	verbose "Checking links on ${yr_gr_all} is done."
}

full()
{
	deploy
	switch
	check
	check_links
}

list()
{
	local _instance _orange_dir _yr_gr_all

	printf "${list_tpl}" "INSTANCE" "ORANGE DIR" "YR GROUPS"

	for _instance in $instances; do
		eval _orange_dir=\"\$${_instance}_orange_dir\"
		: ${_orange_dir:=${default_orange_dir}}
		eval _yr_gr_all=\"\$${_instance}_yr_gr_all\"
		if [ -z "${_orange_dir}" -o -z "${_yr_gr_all}" ]; then
			err 1 "Internal error in the \"${_instance}\"" \
				"configuration !"
		fi
		printf "${list_tpl}" "${_instance}" "${_orange_dir}" \
			"${_yr_gr_all}"
	done | sort -b -k3

	exit 0
}

cleanup()
{
	if [ -n "${cleanup_files}" ]; then
		rm -f $cleanup_files || true
	fi
}


#-- Variables ----------------------------------------------------------

thiscmd=$(basename $0)

# Default params
default_mode="run"

default_orange_dir="/Berkanavt/orange"
default_orange_user="crawler"
default_orange_group="wheel"
default_orange_rel_link="release"
default_orange_rst_int=0		# seconds between restarting nodes
default_orange_rst_by="/40"		# in yr terms
default_orange_rst_stime=3		# safe time between stopping and
					# starting
default_orange_ctl_freebsd="/usr/local/etc/rc.d/orange"
default_orange_ctl_linux="service orange"
default_orange_ok_re="$(date +%F).*[0-9]+ applications started successfully"

get_md5="uname -s | sed -e 's/freebsd/md5/I; s/linux/md5sum/I'"
get_http_client="uname -s | sed -e 's/freebsd/fetch/I; s/linux/wget -nv/I'"

rel_tmp_dir="/var/tmp"
arch_sfx=".tar.gz"
cp_factor=10
: ${cp_domain=".fb.yandex.ru"}

yr_err_re="^--- ERROR "

# List of available instances, used only in list()
instances="mpa mr mt ru ruft ruidx sl ss ssft sstt tm trft trtw us usft usidx yta"

list_tpl="%10s  %-24s %s\n"

# RU Orange cluster
ru_orange_rst_int_realspam=45
ru_orange_rst_by_realspam=" "
ru_orange_rst_int_realtime=45
ru_orange_rst_by_realtime=" "
ru_gr_list="realspam realtime feeder orange"
ru_gr_cp="realspam"
ru_yr_gr_feeder="+TURBOSPIDER"
ru_yr_gr_realspam="+REALSPAM"
ru_yr_gr_realtime="+REALTIME"
ru_yr_gr_orange="+ORANGE"
ru_yr_gr_all="+ORANGE_RU"

# US Grape cluster
us_orange_rst_int_feeder=45
us_orange_rst_by_feeder=" "
us_orange_rst_int_raisin=15
us_orange_rst_by_raisin=" "
us_orange_rst_int_realstore=30
us_orange_rst_by_realstore=" "
us_gr_list="raisin realstore feeder grape"
us_gr_cp="raisin"
us_yr_gr_feeder="+FEEDER_US"
us_yr_gr_raisin="+RAISIN"
us_yr_gr_realstore="+REALSTORE_US"
us_yr_gr_grape="+GRAPE"
us_yr_gr_all="+ORANGE_US"

# RU RTIndexer cluster
ruidx_orange_dir="/Berkanavt/realtime"
ruidx_orange_rst_int_realidx=45
ruidx_orange_rst_by_realidx=" "
ruidx_gr_list="realidx"
ruidx_gr_cp="realidx"
ruidx_yr_gr_realidx="+REALIDX"
ruidx_yr_gr_all="${ruidx_yr_gr_realidx}"

# US RTIndexer cluster
usidx_orange_dir="/Berkanavt/realtime"
usidx_orange_ctl_linux_realstore="service realtime"
usidx_orange_rst_int_realstore=45
usidx_orange_rst_by_realstore=" "
usidx_gr_list="realstore"
usidx_gr_cp="realstore"
usidx_yr_gr_realstore="+REALSTORE_US"
usidx_yr_gr_all="${usidx_yr_gr_realstore}"

# RU FastTier cluster
ruft_orange_dir="/Berkanavt/fasttier"
ruft_orange_rst_int_fasttier_ru=20
ruft_orange_rst_by_fasttier_ru=" "
ruft_gr_list="fasttier_ru"
ruft_gr_cp="fasttier_ru"
ruft_yr_gr_fasttier_ru="+FASTTIERRU"
ruft_yr_gr_all="${ruft_yr_gr_fasttier_ru}"

# TR FastTier cluster
trft_orange_dir="/Berkanavt/fasttier"
trft_orange_rst_int_fasttier_tr=20
trft_orange_rst_by_fasttier_tr=" "
trft_gr_list="fasttier_tr"
trft_gr_cp="fasttier_tr"
trft_yr_gr_fasttier_tr="+FASTTIERTR"
trft_yr_gr_all="${trft_yr_gr_fasttier_tr}"

# TR TwitTier cluster
trtw_orange_dir="/Berkanavt/twittier"
trtw_orange_ctl_linux_fasttier_tr="service twittier"
trtw_orange_rst_int_fasttier_tr=20
trtw_orange_rst_by_fasttier_tr=" "
trtw_gr_list="fasttier_tr"
trtw_gr_cp="fasttier_tr"
trtw_yr_gr_fasttier_tr="+FASTTIERTR"
trtw_yr_gr_all="${trtw_yr_gr_fasttier_tr}"

# TRENDERMR cluster
tm_orange_dir="/Berkanavt/orange"
tm_orange_rst_int_trendermr=20
tm_orange_rst_by_trendermr=" "
tm_gr_list="trendermr"
tm_gr_cp="trendermr"
tm_yr_gr_trendermr="+TRENDERMR"
tm_yr_gr_all="${tm_yr_gr_trendermr}"

# SiteSearch cluster
ss_orange_dir="/Berkanavt/sitetier"
ss_orange_rst_int_sitesearch=20
ss_orange_rst_by_sitesearch=" "
ss_gr_list="sitesearch"
ss_gr_cp="sitesearch"
ss_yr_gr_sitesearch="+SITESEARCH"
ss_yr_gr_all="${ss_yr_gr_sitesearch}"

# SiteSearch FastTier cluster
ssft_orange_dir="/Berkanavt/fasttier"
ssft_orange_ctl_linux_fasttier_tr="service fasttier"
ssft_orange_rst_int_sitesearch=20
ssft_orange_rst_by_sitesearch=" "
ssft_gr_list="sitesearch"
ssft_gr_cp="sitesearch"
ssft_yr_gr_sitesearch="+SITESEARCH"
ssft_yr_gr_all="${ssft_yr_gr_sitesearch}"

# SiteSearch TwitTier cluster
sstt_orange_dir="/Berkanavt/twittier"
sstt_orange_ctl_linux_fasttier_tr="service twittier"
sstt_orange_rst_int_sitesearch=20
sstt_orange_rst_by_sitesearch=" "
sstt_gr_list="sitesearch"
sstt_gr_cp="sitesearch"
sstt_yr_gr_sitesearch="+SITESEARCH"
sstt_yr_gr_all="${sstt_yr_gr_sitesearch}"

# US FastTier cluster
usft_orange_dir="/Berkanavt/fasttier"
usft_orange_rst_int_fasttier_us=45
usft_orange_rst_by_fasttier_us=" "
usft_gr_list="fasttier_us"
usft_gr_cp="fasttier_us"
usft_yr_gr_fasttier_us="+FASTTIERUS"
usft_yr_gr_all="${usft_yr_gr_fasttier_us}"

# MR Agent
mr_orange_dir="/Berkanavt/mragent"
mr_orange_ctl_freebsd_orange="/usr/local/etc/rc.d/mragent"
mr_gr_list="orange"
mr_gr_cp="orange"
mr_yr_gr_orange="+ORANGE"
mr_yr_gr_all="${mr_yr_gr_orange}"

# URL Trender
sl_orange_rst_int_saloon=60
sl_orange_rst_by_saloon=" "
sl_gr_list="saloon"
sl_gr_cp="saloon"
sl_yr_gr_saloon="+SALOON"
sl_yr_gr_all="${sl_yr_gr_saloon}"

# MR Proxy -> MR Agent
mpa_orange_dir="/Berkanavt/mragent"
mpa_orange_rst_int_mrproxy=20
mpa_orange_rst_by_mrproxy=" "
mpa_gr_list="mrproxy"
mpa_gr_cp="mrproxy"
mpa_yr_gr_mrproxy="+MRPROXY"
mpa_yr_gr_all="${mpa_yr_gr_mrproxy}"

# MR Proxy -> YT Agent
yta_orange_dir="/Berkanavt/ytagent"
yta_orange_ctl_linux_mrproxy="service ytagent"
yta_orange_rst_int_mrproxy=15
yta_orange_rst_by_mrproxy=" "
yta_gr_list="mrproxy"
yta_gr_cp="mrproxy"
yta_yr_gr_mrproxy="+MRPROXY"
yta_yr_gr_all="${yta_yr_gr_mrproxy}"

# Mango Trender
mt_gr_list="mango_feeder"
mt_gr_cp="mango_feeder"
mt_yr_gr_mango_feeder="+MANGO_FEEDER"
mt_yr_gr_all="${mt_yr_gr_mango_feeder}"


#-- Main ---------------------------------------------------------------

get_opts $@
check_opts

case "${mode}" in
	${default_mode}) ;;
	list) $mode ;;
	*) usage ;;
esac

set_vars

trap cleanup EXIT

case "${target}" in
	get_rel|cp_to_all|extract_rel|deploy|switch|check|check_links|full)
		$target
		;;
	*)
		usage
		;;
esac

