#!/usr/bin/env python

import json
import os
import re
import sys
import time

import otrs
import requests


def get_all_tickets_info():
    """
    Generator yielding tuples (tid, ticket_info)
    """
    next_page = 1
    params = otrs.load_defaults({
        'action': 'search',
        'ansformat': 'json',
        'ansdetail': 'full',
        'status': 'open',
    })
    url_params, data_params = otrs.split_options(params)
    while next_page:
        url_params['current_page'] = next_page
        next_page, page_data = get_page(url_params, data_params)
        for tid in page_data['result']:
            yield tid, get_ticket_info(tid)


def get_page(url_params, data_params):
    result = json.loads(requests.send_request(url_params, data_params))
    next_page = result['navigation']['next_page']
    return next_page, result


def get_ticket_info(tid):
    params = otrs.load_defaults({
        'action': 'getmessages',
        'ansformat': 'json',
        'ansdetail': 'full',
        'tid': tid,
    })
    url_params, data_params = otrs.split_options(params)
    return json.loads(requests.send_request(url_params, data_params))


def close_tickets(tickets):
    "Close selected tickets"
    params = otrs.load_defaults({
        'action': 'close',
        'ansformat': 'json',
        'ansdetail': 'full',
    })
    url_params, data_params = otrs.split_options(params)
    for tid, subject in tickets:
        url_params['tid'] = tid
        sys.stdout.write(('Closing ticket #%d: %s\n' % (tid, subject)).encode("utf-8"))
        requests.send_request(url_params, data_params)


def print_titles():
    for tid, info in get_all_tickets_info():
        title = info['result']['Subject']
        owner = info['result']['OwnerID']
        sys.stdout.write("%s\n" % "\t".join(map(str, (title.strip().encode("utf-8"), tid, owner))))


class Filter:
    def __init__(self, data):
        self.__data = []

        for line in data.decode("utf-8").split("\n"):
            line = line.strip()
            if not line:
                continue

            self.__data.append(re.compile(line))

    def match(self, data):
        if not data:
            return False

        for el in self.__data:
            if el.match(data):
                return True

        return False

    def search(self, data):
        if not data:
            return False

        for el in self.__data:
            if el.search(data):
                return True

        return False


def filter_titles(config_path):
    config = eval(file(config_path).read())
    age = config.get("age")
    skip_owners = set(config.get("free_id", []))

    subject_filter = Filter(config.get("subject_re", ""))
    from_filter = Filter(config.get("from_re", ""))
    body_filter = Filter(config.get("body_re", ""))

    res = []
    for tid, info in get_all_tickets_info():
        if info['result']['Priority'] != 3:
            continue

        title = info['result']['Subject']
        owner = info['result']['OwnerID']
        fromm = info['result']["Article"][0]["From"][0]["Email"]
        body = info['result']["Article"][0]["Body"]["Body"].strip().split("\n")[0]

        if owner in skip_owners:
            continue

        if from_filter.match(fromm) or body_filter.search(body.lower()):
            res.append((tid, title))
            continue

        update_time = info['result']['DateUpdateUnix']
        if age and (time.time() - update_time) / 3600 / 24 > age:
            res.append((tid, title + " %s" % info['result']['DateUpdate']))
            continue

        title = info['result']['Subject']

        if subject_filter.search(title):
            res.append((tid, title))

    return res


def print_filtered_titles(filter_path):
    for tid, title in filter_titles(filter_path):
        print title, tid


def cleanup(filter_path):
    close_tickets(filter_titles(filter_path))


if __name__ == '__main__':
    from optparse import OptionParser

    parser = OptionParser()
    parser.add_option('--titles', action='store_true')
    parser.add_option('--filter', action='store_true')
    parser.add_option('--clear', action='store_true')

    (options, args) = parser.parse_args()

    config_path = os.path.join(os.path.dirname(sys.argv[0]), "cleanup.cfg")

    if options.titles:
        print_titles()
    elif options.clear:
        cleanup(config_path)
    elif options.filter:
        print_filtered_titles(config_path)
