import urllib2

import urlutils

API_URL = 'http://bot.yandex-team.ru/api/otrs_api.php'

NECESSARY = 1
OPTIONAL = 0


class RequesterError(Exception):
    pass


class BotOTRSRequester(object):
    '''
    Base class for all requests to new otrs api in bot.yandex-team.ru.
    (http://bot.yandex-team.ru/api/otrs_api.php)

    Docs for api can be found at http://wiki.yandex-team.ru/bot/api/otrs.
    '''

    def __init__(self):
        raise NotImplemented('Use one of child classes instead')

    @classmethod
    def _get_required_params(cls):
        url_params = {'action': NECESSARY}
        data_params = {}
        return url_params, data_params

    def _check_missed_params(self, got, required):
        '''
        Check if we got all necessary parameters.
        '''
        missed = []
        necessary = [param for param in required
                     if required[param] == NECESSARY]
        for param in necessary:
            if not param in got:
                missed.append(param)
        return missed

    def _filter_unused_params(self, got, required):
        return dict([(k, v) for (k, v) in got.items() if k in required and v])

    def _check_params(self, url_params, data_params):
        url_params_r, data_params_r = self._get_required_params()
        missed = self._check_missed_params(url_params, url_params_r)
        missed += self._check_missed_params(data_params, data_params_r)
        if missed:
            raise RequesterError(
                'Not enough parameters. Missing: %s' %
                (', '.join(missed))
            )
        url_params_result = self._filter_unused_params(url_params,
                                                       url_params_r)
        data_params_result = self._filter_unused_params(data_params,
                                                        data_params_r)
        return url_params_result, data_params_result

    def make_request(self, url_params, data_params):
        url_params, data_params = self._check_params(url_params, data_params)
        url = urlutils.get_full_url(API_URL, url_params)
        data = urlutils.urlencode(data_params)
        request = urllib2.Request(url)
        request.add_data(data)
        return request

    def send_request(self, url_params, data_params):
        request = self.make_request(url_params, data_params)
        try:
            opened = urllib2.urlopen(request)
            result = opened.read()
        except urllib2.URLError as err:
            raise RequesterError('Error occured when connecting api: %s' % err)
        return result

class TicketRequester(BotOTRSRequester):
    '''
    Base class for all ticket requests
    '''

    def __init__(self):
        pass

    @classmethod
    def _get_required_params(cls):
        required = BotOTRSRequester._get_required_params()
        url_params, data_params = required
        url_params.update({
            'initiator': NECESSARY,
            'ansformat': OPTIONAL,
            'ansdetail': OPTIONAL,
        })
        data_params.update({
            'token': NECESSARY,
        })
        return url_params, data_params


class CreateTicketRequester(TicketRequester):
    '''
    Requester for creating tickets
    '''

    def __init__(self):
        pass

    @classmethod
    def _get_required_params(cls):
        required = TicketRequester._get_required_params()
        url_params, data_params = required
        url_params.update({
            'replyto': NECESSARY,
            'cc': OPTIONAL,
            'queueid': NECESSARY,
            'inv': OPTIONAL,
            'fqdn': OPTIONAL,
            'logic': OPTIONAL,
        })
        data_params.update({
            'subject': NECESSARY,
            'body': NECESSARY,
        })
        return url_params, data_params


class AnswerTicketRequester(TicketRequester):
    '''
    Requester for answering tickets
    '''

    def __init__(self):
        pass

    @classmethod
    def _get_required_params(cls):
        required = TicketRequester._get_required_params()
        url_params, data_params = required
        url_params.update({
            'replyto': NECESSARY,
            'cc': OPTIONAL,
            'tn': NECESSARY,
        })
        data_params.update({
            'subject': NECESSARY,
            'body': NECESSARY,
        })
        return url_params, data_params


class CommentTicketRequester(TicketRequester):
    '''
    Requester for commenting tickets
    '''

    def __init__(self):
        pass

    @classmethod
    def _get_required_params(cls):
        required = TicketRequester._get_required_params()
        url_params, data_params = required
        url_params.update({
            'tn': NECESSARY,
        })
        data_params.update({
            'body': NECESSARY,
        })
        return url_params, data_params


class AddTagTicketRequester(TicketRequester):
    '''
    Requester for adding tags to tickets
    '''

    def __init__(self):
        pass

    @classmethod
    def _get_required_params(cls):
        required = TicketRequester._get_required_params()
        url_params, data_params = required
        url_params.update({
            'tn': NECESSARY,
            'articleid': OPTIONAL,
            'tag': NECESSARY,
        })
        return url_params, data_params


class DelTagTicketRequester(TicketRequester):
    '''
    Requester for deleting tags from tickets
    '''

    def __init__(self):
        pass

    @classmethod
    def _get_required_params(cls):
        required = TicketRequester._get_required_params()
        url_params, data_params = required
        url_params.update({
            'tn': NECESSARY,
            'tag': NECESSARY,
        })
        return url_params, data_params


class CloseTicketRequester(TicketRequester):
    '''
    Requester for closing tickets
    '''

    def __init__(self):
        pass

    @classmethod
    def _get_required_params(cls):
        required = TicketRequester._get_required_params()
        url_params, data_params = required
        url_params.update({
            'tn': OPTIONAL,
            'tid': OPTIONAL,
        })
        data_params.update({
            'body': OPTIONAL,
        })
        return url_params, data_params


class GetTagRequester(BotOTRSRequester):
    '''
    This request resolves tag names by id and vice versa
    '''

    def __init__(self):
        pass

    @classmethod
    def _get_required_params(cls):
        required = BotOTRSRequester._get_required_params()
        url_params, data_params = required
        url_params.update({
            'ansformat': OPTIONAL,
            'ansdetail': OPTIONAL,
            'name': OPTIONAL,
            'id': OPTIONAL,
        })
        data_params.update({
            'token': NECESSARY,
        })
        return url_params, data_params


class SearchRequester(BotOTRSRequester):
    '''
    This request resolves tag names by id and vice versa
    '''

    def __init__(self):
        pass

    @classmethod
    def _get_required_params(cls):
        required = BotOTRSRequester._get_required_params()
        url_params, data_params = required
        url_params.update({
            'queueid': NECESSARY,
            'status': OPTIONAL,
            'tags': OPTIONAL,
            'from': OPTIONAL,
            'ansformat': OPTIONAL,
            'ansdetail': OPTIONAL,
            'current_page': OPTIONAL,
            'entries_per_page': OPTIONAL,
        })
        data_params.update({
            'token': NECESSARY,
        })
        return url_params, data_params


class GetMessagesRequester(BotOTRSRequester):
    def __init__(self):
        pass

    @classmethod
    def _get_required_params(cls):
        required = BotOTRSRequester._get_required_params()
        url_params, data_params = required
        url_params.update({
            'queueid': NECESSARY,
            'ansformat': OPTIONAL,
            'ansdetail': OPTIONAL,
            'tn': OPTIONAL,
            'tid': OPTIONAL,
        })
        data_params.update({
            'token': NECESSARY,
        })
        return url_params, data_params

KNOWN_REQUESTS = {
    'create': CreateTicketRequester,
    'close': CloseTicketRequester,
    'comment': CommentTicketRequester,
    'answer': AnswerTicketRequester,
    'addtag': AddTagTicketRequester,
    'deltag': DelTagTicketRequester,
    'gettag': GetTagRequester,
    'search': SearchRequester,
    'getmessages': GetMessagesRequester,
}


def send_request(url_params, data_params):
    action = url_params.get('action')
    if not action:
        raise RequesterError('Action required.')
    Requester = KNOWN_REQUESTS.get(action)
    if not Requester:
        raise RequesterError('Unknow action %s' % action)
    requester = Requester()
    return requester.send_request(url_params, data_params)
