#!/bin/sh -e
#
# Script for deploying new robot stables.
#
# $Id$

PATH=/bin:/sbin:/usr/bin:/usr/sbin:/usr/local/bin:/usr/local/sbin:/Berkanavt/bin/scripts
export PATH

#-- Subroutines --------------------------------------------------------

err()
{
	local _exitval

	_exitval=$1
	shift

	echo 1>&2 "ERROR: $*"
	exit $_exitval
}

usage()
{
	echo 1>&2 "Usage: ${thiscmd} -i <instance> [-c <control_host>]" \
		"[-r <robot_stable>] [-d] <target>"
	echo 1>&2 "Options are:"
	echo 1>&2 "  -i <instance>         instance of robot (ru, eng)"
	echo 1>&2 "  -r <robot_stable>     format: robot-stable-YYYY-MM-DD"
	echo 1>&2 "  -c <control_host>     control host"
	echo 1>&2 "  -d                    dry-run (only for certain targets)"
	echo 1>&2 "Targets are:"
	echo 1>&2 "  svn_cp_to_new         copy svn dirs to ${sfx_new}-ones"
	echo 1>&2 "  svn_sw_new            switch ${sfx_new}-dirs to" \
		"<robot_stable>"
	echo 1>&2 "  svn_sync              sync ${sfx_new}-dirs from" \
		"<control_host>"
	echo 1>&2 "  svn_pre_install       meta-target for svn" \
		"pre-install-targets"
	echo 1>&2 "  svn_install           install ${sfx_new}-dirs"
	echo 1>&2 "  svn_install_all       install ${sfx_new}-dirs on" \
		"all hosts"
	exit 1
}

get_opts()
{
	local _opt

	while getopts "c:di:r:" _opt; do
		case "$_opt" in
			c) ctl_host="$OPTARG" ;;
			d) dry_run="echo" ;;
			i) instance="${OPTARG}" ;;
			r) robot_stable="$OPTARG" ;;
			*) usage ;;
		esac
	done

	shift $(($OPTIND - 1))

	if [ -n "$1" ]; then
		target="$1"; shift
	fi

	if [ $# -ne 0 ]; then
		usage
	fi
}

get_group()
{
	local _host

	if [ -n "$1" ]; then
		_host="$1"
	else
		_host="${hostname_s}"
	fi

	echo $_host | sed -Ee 's/[0-9]+$//; s/-/_/g'
}

set_svn_dirs()
{
	group=$(get_group)

	eval svn_dirs=\"\$svn_dirs_${group}\"
	: ${svn_dirs:=${svn_dirs_default}}
}

set_yr_vars()
{
	local _yr_grp _ctl_host _group

	eval yr_groups=\"\$yr_groups_${instance}\"
	if [ -z "${yr_groups}" ]; then
		err 1 "Can't set \$yr_groups ! Probably instance is incorrect !"
	fi

	for _yr_grp in $yr_groups; do
		_ctl_host=$(yr $_yr_grp LIST | sort | head -1)
		if [ -z "${_ctl_host}" ]; then
			err 1 "Can't get control host for ${_yr_grp} !"
		fi
		yr_ctl_hosts="${yr_ctl_hosts}${yr_ctl_hosts:+ }+${_ctl_host}"
		if echo $_yr_grp | grep -q '[A-Z]'; then
			_group=$(get_group $_ctl_host)
			eval ctl_host_${_group}=\"${_ctl_host}\"
			eval yr_group_${_group}=\"${_yr_grp}\"
			yr_big_groups="${yr_big_groups}${yr_big_groups:+ }${_group}"
		fi
	done
}

set_vars()
{
	set_svn_dirs
	set_yr_vars
}

am_i_root()
{
	if [ $(id -u) -ne 0 ]; then
		return 1
	else
		return 0
	fi
}

only_root()
{
	if ! am_i_root; then
		err 1 "Only root can launch target ${target} !"
	fi
}

no_root()
{
	if am_i_root; then
		err 1 "root can't launch target ${target} !"
	fi
}

check_vars()
{
	for _var in $*; do
		eval _val=\"\$${_var}\"
		if [ -z "${_val}" ]; then
			err 1 "Variable \$${_var} must be defined for" \
				"target ${target} !"
		fi
	done
}

svn_cp_to_new()
{
	only_root

	local _dir _dir_new

	for _dir in $svn_dirs; do
		_dir_new="${_dir}${sfx_new}"
		if ! rsync -a --delete --force ${_dir}/ ${_dir_new}/; then
			err 1 "Can't sync ${_dir} to ${_dir_new} !"
		fi
	done
}

svn_sw_new()
{
	no_root
	check_vars robot_stable

	local _dir _dir_new _svn_url _svn_url_new

	for _dir in $svn_dirs; do
		_dir_new="${_dir}${sfx_new}"
		_svn_url=$(svn info $_dir_new | \
			awk '$1 == "URL:" {print $2}')
		_svn_url_new=$(echo $_svn_url | \
			sed -Ee "s,\/${robot_stable_re},\/${robot_stable}\/,")
		check_vars _svn_url _svn_url_new
		if ! echo "$_svn_url_new" | grep -q "${robot_stable}"; then
			err 1 "URL for switch (${_svn_url_new}) doesn't" \
				"contain name of new stable" \
				"(${robot_stable}) !"
		fi
		if ! cd $_dir_new; then
			err 1 "Can't change dir to ${_dir_new} !"
		fi
		if ! svn sw --accept theirs-full --force $_svn_url_new; then
			err 1 "Can't switch ${_dir_new} to ${_svn_url_new} !"
		fi
		if ! svn revert -R .; then
			err 1 "Can't revert all local changes in ${_dir_new} !"
		fi
		if ! find . -user $(id -un) -exec chmod ug+w {} \;; then
			err 1 "Can't fix permissions for ${_dir_new} !"
		fi
		if ! svn st -q; then
			err 1 "Can't show status !"
		fi
	done
}

svn_sync()
{
	only_root
	check_vars ctl_host

	local _dir _dir_new

	for _dir in $svn_dirs; do
		_dir_new="${_dir}${sfx_new}"
		if ! rsync -ae ssh --delete --force \
			${ctl_host}${domain_for_cp}:${_dir_new}/ ${_dir_new}/
		then
			err 1 "Can't sync ${_dir_new} from ${ctl_host} !"
		fi
	done
}

svn_install()
{
	only_root

	local _dir _dir_new _dir_old

	for _dir in $svn_dirs; do
		_dir_new="${_dir}${sfx_new}"
		_dir_old="${_dir}${sfx_old}"
		if ! [ -d "${_dir}" -a -d "${_dir_new}" ]; then
			err 1 "Both ${_dir} and ${_dir_new} must exist !"
		fi
		if [ -d "${_dir_old}" ]; then
			rm -rf $_dir_old || true
		fi
		if ! mv $_dir $_dir_old; then
			err 1 "Can't save ${_dir} to ${_dir_old} !"
		fi
		if ! mv $_dir_new $_dir; then
			err 1 "Can't install ${_dir_new} to ${_dir} !"
		fi
	done
}

svn_pre_install()
{
	local _target _target_pre _grp _ctl_host _yr_group

	# TODO dry-run mode

	_target="svn_cp_to_new"
	if ! yr $yr_ctl_hosts / \=\= "${self} -i ${instance} ${_target} && \
		echo ${_target} ${msg_ok}"
	then
		err 1 "Can't run ${_target} !"
	fi

	_target="svn_sw_new"
	if ! yr $yr_ctl_hosts / "${self} -i ${instance} -r ${robot_stable} \
		${_target} && echo ${_target} ${msg_ok}"
	then
		err 1 "Can't run ${_target} !"
	fi

	_target_pre="svn_cp_to_new"
	_target="svn_sync"
	for _grp in $yr_big_groups; do
		eval _ctl_host=\"\$ctl_host_${_grp}\"
		eval _yr_group=\"\$yr_group_${_grp}\"
		if ! yr $_yr_group -$_ctl_host / \=\= \
			"${self} -i ${instance} ${_target_pre} && \
			echo ${_target_pre}:${_grp} ${msg_ok} && \
			${self} -i ${instance} -c ${_ctl_host} ${_target} && \
			echo ${_target}:${_grp} ${msg_ok}"
		then
			# TODO Request for going on
			err 1 "Can't run ${_target}:${_grp} !"
		fi
	done
}

svn_install_all()
{
	local _target

	_target="svn_install"
	if ! yr $yr_groups / \=\= "${self} -i ${instance} ${_target} && \
		echo ${_target} ${msg_ok}"
	then
		err 1 "Can't run ${_target} !"
	fi
}


#-- Variables ----------------------------------------------------------

thiscmd=$(basename $0)
hostname_s=$(hostname -s)

self="/Berkanavt/webscripts/admscripts/scripts/${thiscmd}"

yr_groups_ru="+WALRUS +PRIMUS +AHTARC +ya +regionus +fastcontent00
	+hors +classific +robotmon-ii +walrus +primus +htarc"

yr_groups_eng="+WALRUSENG +PRIMUSENG +HTARCENG +yaeng"

sfx_new=".NEW"
sfx_old=".PREV"

msg_ok="OK"

robot_stable_re="robot-stable-[0-9]{4}(-[0-9]{2}){2}"

svn_dirs_everywhere="/Berkanavt/config"
svn_dirs_default="${svn_dirs_everywhere}"

svn_dirs_ya="
	${svn_dirs_everywhere}
	/Berkanavt/dbspider/scripts
	/Berkanavt/pornofilter
	/Berkanavt/webscripts/bsconfig-search-merge-scripts
	/Berkanavt/webscripts/merge
	/Berkanavt/webscripts/monitoring
	/Berkanavt/webscripts/pagerank
	/Berkanavt/webscripts/robot-checks
	/Berkanavt/webscripts/robot-common
	/Berkanavt/webscripts/selectgeo
	/Berkanavt/webscripts/yscripts
	"

svn_dirs_yaeng="${svn_dirs_ya}"

svn_dirs_walrus="
	${svn_dirs_everywhere}
	/Berkanavt/dbspider/scripts
	/Berkanavt/webscripts/bsconfig-search-merge-scripts
	/Berkanavt/webscripts/merge
	/Berkanavt/webscripts/pagerank
	/Berkanavt/webscripts/robot-checks
	/Berkanavt/webscripts/robot-common
	/Berkanavt/webscripts/yscripts
	"

svn_dirs_walruseng="${svn_dirs_walrus}"

svn_dirs_primus="
	${svn_dirs_everywhere}
	/Berkanavt/webscripts/bsconfig-search-merge-scripts
	/Berkanavt/webscripts/merge
	/Berkanavt/webscripts/pagerank
	/Berkanavt/webscripts/robot-common
	/Berkanavt/webscripts/yscripts
	"

svn_dirs_primuseng="${svn_dirs_primus}"

svn_dirs_regionus="${svn_dirs_primus}"

svn_dirs_htarc="
	${svn_dirs_everywhere}
	/Berkanavt/webscripts/merge
	"
svn_dirs_htarceng="${svn_dirs_htarc}"

svn_dirs_fastcontent="
	${svn_dirs_everywhere}
	/Berkanavt/webscripts/robot-common
	"

domain_for_cp=".fb.yandex.ru"


#-- Main ---------------------------------------------------------------

get_opts $@
set_vars

case "${target}" in
	# Targets
	svn_cp_to_new|svn_sw_new|svn_sync|svn_install) $target ;;
	# Meta-targets
	svn_pre_install|svn_install_all) $target ;;
	*) usage ;;
esac

