#!/bin/sh

#Script for add template for cron job in directory /etc/cron.d/
#
#Author: Igor Miheenko <affe@yandex-team.ru>

set -u

#####################
##### VARIABLES #####
#####################

script_name="$(basename $0)"

#####################
##### FUNCTIONS #####
#####################

usage()
{
	printf "
Requires at 1 parameter.
Usage: 
  sudo ${script_name} <name>

Use -h for additional help.
"
	exit 1
}

help()
{
	printf "
NAME
	${script_name} - adds template for cron job

SYNTAX
	sudo ${script_name} <name>
	${script_name} -h

DESCRIPTION
	This script generates template for cron job and put it in directory
	/etc/cron.d/
	Script needs a root access for write in directory /etc/cron.d/

OPTIONS
	-h		show this help and exit

PARAMETERS
	name		for name it is best to use number of task
			(OTRS ticket or other ITS)

EXAMPLES
	Add template job with ticket number as name:
		sudo ${script_name} 14012215937598748

	Cron scheduling definitions:
	* * * * *  command to execute
	┬ ┬ ┬ ┬ ┬	
	│ │ │ │ │
	│ │ │ │ │
	│ │ │ │ └───── day of week (0 - 7) (7 is Sunday, the same as 0)
	│ │ │ └─────── month (1 - 12)
	│ │ └───────── day of month (1 - 31)
	│ └─────────── hour (0 - 23)
	└───────────── min (0 - 59)
	For more information about cron see http://en.wikipedia.org/wiki/Cron

Dec 25 2013								  BSD
"
	exit 1
}

error()
{
	printf "%s: %s\n" "$script_name" "${1:-"Unknown Error"}" 1>&2
	exit 1
}

#####################
###### OPTIONS ######
#####################

while getopts "h" opt; do
	case ${opt} in
		h)
			help
			;;
		*)
			usage
			;;
	esac
done

#####################
###### CHECKS #######
#####################

#if run without any arguments
if [ -z "$*" ] || [ "$#" -gt 1 ]; then
	usage
elif [ "$#" -eq 1 ]; then
	name="$1"
fi

#need to run as root
if [ "$(whoami)" != root ]; then
	error "Need to run as root"
fi

#####################
####### BODY ########
#####################

if [ "$name" ] && [ -d /etc/cron.d ] && [ ! -f /etc/cron.d/"$name" ]; then
	if (>/etc/cron.d/"$name") >/dev/null 2>&1; then
		printf "File created\n"
	else
		error "Cannot create file /etc/cron.d/${name}"
	fi
	printf "\
SHELL=/bin/sh
PATH=/usr/local/sbin:/usr/local/bin:/sbin:/bin:/usr/sbin:/usr/bin:/Berkanavt/bin
TMPDIR=/var/tmp
#MAILTO=
#minute\thour\tmday\tmonth\twday\twho\tcommand\n
" >>/etc/cron.d/${name}
	printf "\t%s /etc/cron.d/%s\n" ${EDITOR:-vim} ${name}
elif [ -f /etc/cron.d/"$name" ]; then
	error "File with the same name already exists: /etc/cron.d/${name}"
else
	error "Not set name for file or not such directory /etc/cron.d/"
fi