#!/usr/bin/env bash

# This is a script for changing host's switchport vlan.
# Author: Korenevskiy Denis (denkoren@)
# Based on mzelenkov@ wiki page prednalivka (http://wiki.yandex-team.ru/MixailZelenkov/prednalivka)
#
# You are free to change this script if you're think, it will help other SEARCH admins.
# Please, don't forget to notify searchadm@ for changes.

set -u

##########################
#### GLOBAL VARIABLES ####
##########################

# If YR IPMI variables were defined, they will be used.
# Otherwise use default values.
: ${IPMI_DOMAIN:=${YR_IPMI_DOMAIN:-"ipmi.yandex-team.ru"}}
: ${IPMI_USER:=${YR_IPMI_USER:-"ADMIN"}}
: ${IPMI_PASS_FILE:=${YR_IPMI_PASS_FILE:-"${HOME}/.ipmi_pass"}}

#########################
#### LOCAL VARIABLES ####
#########################

script_name="$(basename $0)"
script_dir="$(dirname $0)"
# Long option trigger. Defines whether print treating host's FQDN or not.
long_output=false
# Protection boolean for "dry run" activation.
real_run=true
# Number of required parameters, specified after all options.
required_params=1

# FQDN of treating host
host_fqdn=""
# Host's IP address
host_ip=""
# Host's switch port "address": switch name and port number, got from IPMI
host_switch_ident_ipmi=( )
# If switch ident information from IPMI is not correct, then true.
ipmi_fail=false
# Host's switch port "address": switch name and port number, got from Golem
host_switch_ident_golem=( )
# If switch ident information from Golem is not correct, then true.
golem_fail=false
# Host's switch port "address": switch name and port number
host_switch_ident=( )
# Default IPMI lan channel (ipmitool .... lan print <channel>)
lan_channel=1
# Default native VLAN should be "parking" for sequrity reasons.
native_vlan=999
# Default allowed VLANs
allowed_vlans=( )
# Becomes true on first successful try to access to golem API
golem_available=false
# Force mode flag. Force apply changes on not-trusted data.
force_mode=false

#########################
#### LOCAL FUNCTIONS ####
#########################

help ()
{
	cat <<EndOfHelp | ${PAGER:-less}

NAME
	$script_name - changes host's switchport vlan

SYNOPSIS
	$script_name [-v] [-u user] [-p file|-w password|-W] [-c channel] fqdn
            [vlans]
	$script_name -h

DESCRIPTION
	Changes switchport's vlan settings. Uses racktables.yandex.net API to
	change switch settings. Script is based on information from mzelenkov@
	wiki page. (http://wiki.yandex-team.ru/MixailZelenkov/prednalivka)
	If no VLAN specified script tries to define native swirchport vlan
	using Golem API. If not succeed, it fails. If several VLANs specified,
	then marks all VLANs as allowed and first VLAN as native. Golem API can
	define only native VLAN.

OPTIONS
	-c channel
		get information about specified channel of IPMI lan section
		(ipmitool ... lan print \$channel).
		    1 is default

	-d	debug mode. Enables verbose and debug messages.

	-f	force mode. Forces applying of untrusted data. So, some errors
		becomes just warnings.

	-l	long output. Add FQDN parameter value before status reporting

	-n	dry run. This flag makes script do everything but change
		destination switchport VLAN settings. Using this combined with
		-v or -d flag is useful for debugging.

	-p file
		set path to IPMI password file. When -p and -w|-W are specified
		simalteneously, then -w|-W argument will be used.
		    ~/.ipmi_pass is default
		

	-q	quiet mode. Supresses warning messages.

	-u user
		connect to IPMI as user
		    ADMIN is default

	-v	verbose mode. Enables verbose messages.

	-w password
		use password to connect to IPMI. When -w and -W are specified
		simalteneously, then script will use argument of the last of
		two parameters.

	-W	read the password from STDIN before connect to IPMI. When -w
		and -W are specified simalteneously, then script will use
		argument of the last of two parameters.

PARAMETERS
	fqdn	Fully Qualified Domain Name of host, which switchport settings
		should be changed.

	vlans	numbers of VLANs allowed on switchport delimited by spaces. If
		several VLANs specified, then native vlan is first in list and
		allowed vlans are all of them.


VARIABLES
	Script uses following variables to control it's behaviour

	IPMI_USER,	Username, that will be used to connect to host's IPMI
	YR_IPMI_USER	interface.
	
	IPMI_PASS_FILE,
	YR_IPMI_PASS_FILE
			Filepath, where IPMI user's password stored.

	IPMI_DOMAIN,	Domain suffix, where IPMI host's subdomains are placed.
	YR_IPMI_DOMAIN	"ipmi.yandex-team.ru" is default.

	YR-variables has lower priority, so if both IPMI_USER and YR_IPMI_USER
	are defined, IPMI_USER variable will be used.

EXAMPLES
	Change zverushko.yandex.ru's VLANs to it's native:
	    ${script_name} zverushko.yandex.ru

	Change zverushko.yandex.ru's VLANs to 604, 714 and 761 and mark 604 as
	native:
	    ${script_name} zverushko.yandex.ru 604 714 761

	Change nothing, just get switchport address and check it:
	    ${script_name} -n -v -e folgaine1 zverushko.yandex.ru

EXIT CODES
	0 on successful switchport configuration change (if no -d option
	specified).

	64	command line usage error. Wrong parameter given or script can't
		get additional information required automatically.

	69	Golem API not available and it is required for automatically get
		host's VLANs. You can avoid this error by manually defining VLAN
		numbers to switch to.

	70	possibly wrong host's switchport information. Can't verify
		switchport address or even cant get anything from both of Golem
		and Einstellung.

KNOWN BUGS
	Sometimes Golem holds wrong information about switchport address. If
	requested host have been absent in network fore rather long period of
	time, information about it's location become dissapear from network
	map. During this process address of host's switchport sequentally
	"surfaces" to highest levels of network organization. So, when Golem
	asked for host's location at this time, it answers, that host is
	connected to "highest" switch. So, it can even tell you, that host is
	connected to the DC's root switch directly. Script can't detect such
	situations.  So, IF YOU HAVE ENOUGH PERMISSIONS to change such port's
	configuration, BE CAREFUL to use this script with '-f' option on hosts
	without IPMI interfaces.

CONTACTS
	Author: Denis Korenevsky <denkoren@yandex-team.ru>

BSD				January 30, 2013			    BSD

EndOfHelp
}

reference ()
{
	cat <<EndOfReference
	
	Script requires at least ${required_params} parameter.
		$script_name hostname

	For additional help use -h flag:
		$script_name -h

EndOfReference
}

. "${script_dir}/messages.sh"
. "${script_dir}/functions.sh"

#################################################
#### TESTING OPTIONS, CONFIGURING CURREN RUN ####
#################################################

while getopts "c:dfhlnp:qu:vw:W" opt; do
	case $opt in
	c)
		lan_channel=${OPTARG}
		;;
	d)
		msg_verbose_flag=true
		msg_debug_flag=true
		;;
	f)
		force_mode=true
		;;
	h)
		help
		exit 0
		;;
	l)
		long_output=true
		;;
	n)
		real_run=false
		;;
	p)
		IPMI_PASS_FILE=${OPTARG}
		;;
	q)
		msg_warning_flag=false
		;;
	u)
		IPMI_USER=${OPTARG}
		;;
	v)
		msg_verbose_flag=true
		;;
	w)
		ipmi_password=${OPTARG}
		;;
	W)
		read -s -e -p "IPMI user $IPMI_USER password: " ipmi_password
		;;
	\:)
		critical 64 "Option -${OPTARG} requires an argument"
		;;
	\?)
		reference
		exit 64
		;;
	esac
done
shift $(($OPTIND - 1))

if [ "$#" -lt "$required_params" ]; then
	reference
	exit 64
fi

#######################
#### SCRIPT'S BODY ####
#######################

# Initialize variables
host_fqdn="${1}"

# If -l flag wos given, prin host's FQDN given.
# If verbose mode is active, print \n at the end of the string
$long_output \
    && printf "${host_fqdn}: " \
    && ( $msg_verbose_flag || ! $real_run ) \
    && printf "\n"

# Inform user about dry run.
! $real_run \
    && printf "${_msg_LRed}Dry run. Nothing will be changed.${_msg_Normal}\n"
# Inform user, about force mode.
$force_mode \
    && verbose "${_msg_LRed}Force mode${_msg_Normal} had been activated."

# If password is not set yet, setting.
if [ -z "${ipmi_password=""}" ]; then
	if [ -r $IPMI_PASS_FILE ]; then
		ipmi_password=$(cat $IPMI_PASS_FILE)
	else
		warning "No IPMI password specified."
	fi
fi

debug "Trying to get information about switchport of ${host_fqdn} from several sources..."

#--------------------------------------------------------
#--- GET INFORMATION FROM GOLEM AND TRY TO CHECK IT -----
#--------------------------------------------------------

debug "Checking Golem API availability..."
fn_check_availability ro.admin.yandex-team.ru 80 \
    && golem_available=true

if ${golem_available}; then
	verbose "Golem API is available"
	debug "Testing hostname: is it valid FQDN?"
	if wget -q -O - \
	    "http://ro.admin.yandex-team.ru/api/dns/is_valid_fqdn.sbml"\
	    --post-data "name=${host_fqdn}" \
	    | grep -q "Error"; then
		critical 64 "'${host_fqdn}' is not FQDN."
	fi
else
	warning "Golem API is not available"
fi

# Try to get switchport information using golem API if available.
if ${golem_available}; then
	debug "Trying to get host's switchport information using"\
	    "Golem API..."
	debug "Running host_query.sh to find host's swithcport."
        host_switch_ident_golem=( $( "${script_dir}/host_query.sh" \
	    -c switch_short,switch_port ${host_fqdn} ) )
fi

# Check data, recieved from Golem for correctness
if [ "${#host_switch_ident_golem[@]}" -ne "2" ] \
    || [ -z "${host_switch_ident_golem[0]}" ] \
    || [ -z "${host_switch_ident_golem[1]}" ]; then
	warning "Can't get switchport information from Golem"
	golem_fail=true
else
	verbose "Golem switchport information: ${host_switch_ident_golem[@]}"
fi

#-------------------------------------------------------
#--- GET INFORMATION FROM IPMI AND TRY TO CHECK IT -----
#-------------------------------------------------------

if host ${host_fqdn}.${IPMI_DOMAIN} 2>&1 1>/dev/null; then
	if [ -n "${ipmi_password}" ]; then
		verbose "IPMI interface is available"
		debug "Trying to get host's switchport information using host's IPMI..."
		# Getting SNMP Community string from IPMI, then removing all excess spaces
		# and other symbols, so it contains only network locaton information
		host_switch_ident_ipmi=( $(ipmitool -U "${IPMI_USER}" \
		    -P "${ipmi_password}" \
		    -I lanplus \
		    -H "${host_fqdn}.${IPMI_DOMAIN}" lan print ${lan_channel} \
		    | grep 'SNMP Community' \
		    | cut -d ":" -f 2 \
		    | sed 's/[,]//g;s/^[[:space:]]*//g;s/[[:space:]][[:space:]]//g;s/[[:space:]]*$//g' \
		    | cut -d " " -f 1,2) )
	fi
else
	warning "IPMI interface is not available"
fi

# Check data, recieved from IPMI for correctness
if [ "${#host_switch_ident_ipmi[@]}" -ne "2" ] \
    || [ -z "${host_switch_ident_ipmi[0]}" ] \
    || [ -z "${host_switch_ident_ipmi[1]}" ]; then
	warning "Can't get switchport information from IPMI of"\
	    "${host_fqdn}.${IPMI_DOMAIN}"
	ipmi_fail=true
 else
	verbose "IPMI switchport information: ${host_switch_ident_ipmi[@]}"
 fi

#---------------------------------------------
#--- COMPARE INFORMATION FROM TWO SOURCES ----
#---------------------------------------------

# Checking, that we've got some information
${golem_fail} && ${ipmi_fail} \
    && critical 70 "Can't get switchport information using IPMI or Golem API."

# Checking, that information from Golem and IPMI not differ.
if ! ${golem_fail} && ! ${ipmi_fail}; then
	if [ "${host_switch_ident_golem[*]}" != "${host_switch_ident_ipmi[*]}" ]; then

		critical 70 "IPMI and Golem switchport information is different."\
		    "Golem: ${host_switch_ident_golem[@]},"\
		    "IPMI: ${host_switch_ident_ipmi[@]}."\
		    "Please, check switch for correctness:"\
		    "https://racktables.yandex.net/?page=search&q=${host_switch_ident_golem[0]},"\
		    "https://racktables.yandex.net/?page=search&q=${host_switch_ident_ipmi[0]}"

	else
		host_switch_ident=( ${host_switch_ident_golem[@]} )
	fi
else
	if ${real_run} && ! ${force_mode}; then
		critical 70 "Can't verify ${host_fqdn}'s switchport" \
		    "information. Try -f option to force changes."
	else
		warning "Can't verify ${host_fqdn}'s switchport" \
		    "information."
	fi
	if ${golem_fail}; then
		host_switch_ident=( ${host_switch_ident_ipmi[@]} )
	else
		host_switch_ident=( ${host_switch_ident_golem[@]} )
	fi
fi

#-------------------------------
#--- FINAL CHECK FOR SAFETY ----
#-------------------------------

if [ "${#host_switch_ident[@]}" -ne "2" ] \
    || [ -z "${host_switch_ident[0]}" ] \
    || [ -z "${host_switch_ident[1]}" ]; then
	critical 70 "Can't get switchport information using IPMI or Golem API."
else
	verbose "Switchport information seems to be correct: ${host_switch_ident[@]}."
fi

#---------------------------------------------
#--- GET VLAN LIST TO CHANGE AND CHECK IT ----
#---------------------------------------------

# If no VLAN specified in command-line
if [ "$#" -lt "2" ]; then

	# Define VLANs using Golem.

	if ${golem_available}; then
		debug "Trying to define native VLAN using Golem API."
		native_vlan=$( "${script_dir}/net_query.sh" -D \
		    -c vlan \
		    ${host_fqdn} )
		if [ -z "$native_vlan" ]; then
			native_vlan="999"
			critical 64 "No VLAN's specified in command line. Can't define ${host_fqdn}'s VLANs" \
			    "using Golem API."
		else
			host_fb_fqdn=$( printf "${host_fqdn}\n" | sed "s/\.yandex\./.fb.yandex./" )
			debug "Trying to define FastBone VLAN's for ${host_fb_fqdn}"
			fb_vlans=( $( "${script_dir}/net_query.sh" -D \
			    -c vlan \
			    ${host_fb_fqdn} | sort | tr "\n" " " ) )
			if [ -z ${fb_vlans:-""} ]; then
				warning "Can't get FastBone VLANs for ${host_fb_fqdn}."
			else
				debug "FastBone VLAN's given by Golem are ${fb_vlans[@]}."
				allowed_vlans=( ${native_vlan} ${fb_vlans[@]} )
			fi
		fi
	else
		critical 69 "No VLAN's specified in command line. Can't define ${host_fqdn}'s VLANs" \
		    "using Golem API."
	fi
else
	shift
	native_vlan=${1}
	allowed_vlans=( ${@} )

fi

allowed_vlans_string=$(printf "${allowed_vlans[*]}" | sed "s/^/\&allowed[]=/;s/ /\&allowed[]=/g")

debug "Checking that all of given VLANs are numbers."
for i in ${allowed_vlans[@]}; do
	! test "$i" -eq "$i" 2>/dev/null \
	    && critical 64 "One or more of VLANs given is not a number."
done

#----------------------
#--- APPLY CHANGES ----
#----------------------

verbose "Change ${host_switch_ident[@]} switchport mode to trunk."\
    "Allowed VLANs: ${allowed_vlans[@]}."\
    "Native VLAN is ${native_vlan}"
${real_run} && curl -s -G https://racktables.yandex.net/export/vlanrequest.php \
    -d "switchname=${host_switch_ident[0]}&portname=${host_switch_ident[1]}&native=${native_vlan}&mode=trunk${allowed_vlans_string}"
echo ""
