#!/usr/bin/env python3
# -*- coding: utf-8 -*-
#
# Description:
#   This script monitors UP/DOWN events of specified slb,
#   and sends mail to listed mails.
#
# WARNING! If you want to use it, talk to olegsenin@ first!

import urllib.parse
import urllib.request
import json
import sys
import os.path
import getopt
import argparse
import smtplib
#import daemon
#import lockfile
#import logging
from time import time, sleep
from email.mime.text import MIMEText

DEFAULT_BASE_URL = 'http://slbloghandler.yandex-team.ru/jsrpc.php'
DEFAULT_BALANCER = 'ukrop.search.yandex.net'
DEFAULT_MAILLIST = [ 'olegsenin@yandex-team.ru', 'rdna@yandex-team.ru' ]
DEFAULT_START = int(time()) - 24 * 3600

class TimeKeeper(object):
    ''' Keep and restore time '''

    def __init__(self, balancer):
        ''' Init myself '''
        self.file_path = '/var/tmp/.slb-mon.%s.mark' % (balancer)
        self.time = None

    def store(self, time):
        ''' Store time to file '''
        self.time = time
        with open(self.file_path, 'w') as f:
            f.write(str(self.time))

    def restore(self):
        ''' Restore time from file '''
        ## TODO: Check if date is not corrupted
        if self.time:
            return self.time
        else:
            with open(self.file_path, 'r') as f:
                self.time = f.read()
            return self.time


def get_args():
    '''Get command line arguments'''
    parser = argparse.ArgumentParser()

    parser.add_argument('--url', '-u', action='store', type=str,
                        dest='base_url', default=DEFAULT_BASE_URL,
                        help='URL of slbloghandler')

    parser.add_argument('--balancer', '-b', action='store', type=str,
                        dest='balancer', default=DEFAULT_BALANCER,
                        help='FQDN of balancer')

    parser.add_argument('--maillist', '-m', action='store', type=list,
                        dest='maillist', default=DEFAULT_MAILLIST,
                        help='Watchers maillist')

    parser.add_argument('--start', '-s', action='store', type=int,
                        dest='start', default=DEFAULT_START, help='''If it is a first run,
                        you can specify start time in epoch format
                        (-24 hours from now by default)''')

    parser.add_argument('--once', '-o', action='store_true',
                        dest='once', help='''Run scripts once and exit.
                        Usefull, if you want to run this script as cron job.''')

    parser.add_argument('--rsshow', '-r', action='store_const',
                        dest='rsshow', const=1, default=0,
                        help='Show RS flaps')

    args = parser.parse_args()
    return args


def gen_url(all_args, start_time):
    '''Generate proper url'''

    stop_time = int(time())

    get_params = {
            'mode': 'api',
            'action': 'get_data',
            'vsg': all_args.balancer,
            'from': start_time,
            'to': stop_time,
            'rsshow': all_args.rsshow,
            }

    encoded_get_params = urllib.parse.urlencode(get_params)
    return ('?'.join([all_args.base_url, encoded_get_params]), stop_time)


def get_answer(url):
    '''Get slb flap data'''
    answer = None
    try:
        url_fd = urllib.request.urlopen(url)
        answer = url_fd.read().decode()
        url_fd.close()
    except (urllib.error.HTTPError, urllib.error.URLError) as err:
        print(err)
    return answer


def send_mail(maillist, message, balancer):
    '''Send email with geted data'''
    sender = 'root'
    to = [maillist[0]]
    cc = maillist[1:]
    subj = "SLB change state - %s" % (balancer)

    msg = MIMEText(message, 'plain', 'utf-8')
    msg['Subject'] = subj
    msg['From'] = sender
    msg['To'] = ", ".join(to)
    msg['Cc'] = ", ".join(cc)

    try:
        server = smtplib.SMTP('localhost')
        server.sendmail(sender, to + cc, msg.as_string())
        server.quit()
    except smtplib.SMTPException:
        print("Unable to sent e-mail")


def main():
    '''Main function that combines everything'''
    try:
        args = get_args()
        keeper = TimeKeeper(args.balancer)
        start_time = None

        while True:
            if start_time:
                sleep(60)
            try:
                start_time = keeper.restore()
            except IOError as e:
                print(e)
                start_time = args.start

            url, stop_time = gen_url(args, start_time)

            answer = get_answer(url)

            msg = ""
            if answer and answer != '[]':
                for event in json.loads(answer):
                    msg += ('%(time)s - %(fqdn)s -> %(RS)s %(type)s [%(dcname)s]\n' % event)
                send_mail(args.maillist, msg, args.balancer)
                keeper.store(stop_time)
            if args.once :
                sys.exit(0)

    except KeyboardInterrupt:
        sys.exit(1)


if __name__ == '__main__':
#    with daemon.DaemonContext():
    main()
