#!/usr/bin/env python
# -*- coding: utf-8 -*-

import os
import sys
import time
import subprocess
import re
import string

try:
    import ctypes
except:
    pass

LOG_FILE = '/usr/local/www/logs/tbcontroller.log'
STOP_FILE = '/var/tmp/tbcontroller.stop'
PID_FILE = '/var/run/tbcontroller.pid'

def _doLog(log_type, data):
    print >>_doLog.log_file, '%s %s: %s' % (time.strftime('[%Y-%m-%d %H:%M:%S]', time.localtime()), log_type, data)
_doLog.log_file = open(LOG_FILE, 'a', 1)

def _runCommand(args, fail_nonzero_code = True, shell = False, wait_time = 10.):
    wait_till = time.time() + wait_time
    try:
        p = subprocess.Popen(args, bufsize = 1000000, shell = shell, stdout = subprocess.PIPE, stderr = subprocess.PIPE)
    except Exception, e:
        raise Exception, 'subpocess.Popen for <<%s>> failed: got exception %s' % (args, e)

    stdoutdata, stderrdata = '', ''
    while time.time() < wait_till:
        time.sleep(0.1)
        p.poll()
        stdoutdata += p.stdout.read()
        stderrdata += p.stderr.read()
        if p.returncode == 0:
            return stdoutdata, stderrdata
        if not fail_nonzero_code:
            return stdoutdata, stderrdata
        if p.returncode != None:
            raise Exception, 'Command <<%s>> returned %d' % (args, p.returncode)
    p.kill()
    raise Exception, 'Command <<%s>> timed out (%f seconds)' % (args, wait_time)

def _getFreq(runner):
    if runner.os == 'FreeBSD':
        freq = ctypes.c_uint()
        ln = ctypes.c_size_t(ctypes.sizeof(ctypes.c_uint))
        rc = runner.libc.sysctlbyname('dev.cpu.0.freq', ctypes.byref(freq), ctypes.byref(ln), None, 0)
        if rc == -1:
            raise Exception, 'Sysctl for getting dev.cpu.0.freq failed'
        return freq.value
    elif runner.os == 'Linux':
        freqs = []
        for i in range(runner.ncpu):
            freqs.append(int(open('/sys/devices/system/cpu/cpu%s/cpufreq/scaling_min_freq' % i).read()) / 1000)
        return min(freqs)
    else:
        raise Exception, 'Unknown os %s' % runner.os

def _setFreq(runner, new_freq):
    if runner.os == 'FreeBSD':
        freq = ctypes.c_uint()
        freq.value = new_freq
        ln = ctypes.c_size_t(ctypes.sizeof(ctypes.c_uint))
        rc = runner.libc.sysctlbyname('dev.cpu.0.freq', None, 0, ctypes.byref(freq), ln)
        if rc == -1:
            raise Exception, 'Sysctl for setting dev.cpu.0.freq failed'
        return
    elif runner.os == 'Linux':
        for i in range(runner.ncpu):
            try: # seems we need this
                with open('/sys/devices/system/cpu/cpu%s/cpufreq/scaling_max_freq' % i, 'w') as f:
                    f.write('%s' % (new_freq * 1000))
            except:
                pass

            with open('/sys/devices/system/cpu/cpu%s/cpufreq/scaling_min_freq' % i, 'w') as f:
                f.write('%s' % (new_freq * 1000))
            with open('/sys/devices/system/cpu/cpu%s/cpufreq/scaling_max_freq' % i, 'w') as f:
                f.write('%s' % (new_freq * 1000))
            with open('/sys/devices/system/cpu/cpu%s/cpufreq/scaling_governor' % i, 'w') as f:
                f.write('performance')
        return
    else:
        raise Exception, 'Unknown os %s' % runner.os

def _getRawIdle(runner):
    if runner.os == 'FreeBSD':
        timings = (ctypes.c_size_t * 5)()
        ln = ctypes.c_size_t(ctypes.sizeof(ctypes.c_size_t) * 5)
        rc = runner.libc.sysctlbyname('kern.cp_time', ctypes.byref(timings), ctypes.byref(ln), None, 0)
        if rc == -1:
            raise Exception, 'Sysctl for getting kern.cp_time failed'
        return timings[4] / 133. / runner.ncpu
    elif runner.os == 'Linux':
        return float(open('/proc/uptime').read().split(' ')[1]) / runner.ncpu
    else:
        raise Exception, 'Unknown os %s' % runner.os


class TBaseChecker:
    TIMEOUT = 5

    def __init__(self):
        self.runAt = time.time()

    def enabled(self, runner):
        raise Exception, 'Virtual method called'

    def name(self):
        raise Exception, 'Virtual method called'

    def run(self, runner):
        raise Exception, 'Virtual method called'

''' 
Класс, который занимается установкой максимального C-state-а (он него немного зависит потребление). Поскольку мы не нашли
ситуаци, когда включение максимального C-state-а что-то ухудшает, включаем это везде. В Linux максимальный C-state
устанавливается по умолчанию, поэтому делать ничего не нужно.
'''
class TCStateChecker(TBaseChecker):
    TIMEOUT = 1000

    def __init__(self, runner):
        TBaseChecker.__init__(self)

        if runner.os == 'FreeBSD':
            cx_supported = (ctypes.c_char * 1000)()
            ln = ctypes.c_size_t(1000)
            rc = runner.libc.sysctlbyname('dev.cpu.0.cx_supported', cx_supported, ctypes.byref(ln), None, 0)
            if rc == -1:
                raise Exception, 'Sysctl for dev.cpu.0.cx_supported failed'
            self.cxMax = cx_supported.value.split(' ')[-1].split('/')[0]
        elif runner.os == 'Linux':
            pass

    def enabled(self, runner):
        return True

    def name(self):
        return 'CStateChecker'

    def run(self, runner):
        if runner.os == 'FreeBSD':
            cx_current = (ctypes.c_char * 1000)()
            ln = ctypes.c_size_t(1000)
            rc = runner.libc.sysctlbyname('dev.cpu.0.cx_lowest', cx_current, ctypes.byref(ln), None, 0)
            if cx_current.value != self.cxMax:
                value = ctypes.c_char_p(self.cxMax)
                ln = ctypes.c_size_t(len(self.cxMax))
                rc = runner.libc.sysctlbyname('hw.acpi.cpu.cx_lowest', None, 0, value, ln)
                if rc == -1:
                    raise Exception, 'Sysctl for hw.acpi.cpu.cx_lowest failed'
                _doLog('INFO', '%s: setting cx_lowest: %s -> %s' % (self.name(), cx_current.value, self.cxMax))
        elif runner.os == 'Linux':
            pass # Did not find any problems with C-states on Linux

'''
Класс, следящий за включением/выключением Turbo Boost. Всегда включает на не базовых. На базовых включает/выключает в записимости от нагрузки.
'''
class TTBChecker(TBaseChecker):
    TIMEOUT = 5
    LOW_IDLE_MARK = 0.15
    HI_IDLE_MARK = 0.4
    REACTION_TIME = 30

    MODELS_DATA = {
        'E5-2660' : (2200, 2201),
        'E5-2667' : (2900, 2901),
    }

    def __init__(self, runner):
        TBaseChecker.__init__(self)
        self.data = []

    def enabled(self, runner):
        return runner.model in self.MODELS_DATA

    def name(self):
        return 'TBChecker'

    def _always_at_maximum(self, runner):
        NONMAX_GROUPS = set(['MSK_WEB', 'MSK_WEB_PRIEMKA', 'MSK_WEB_R1', 'MSK_IMGS', 'MSK_IMGS_CBIR', 'MSK_VIDEO', 'MSK_DIVERSITY2',
                             'MSK_IMGS_CBIR_R1_NEW', 'MSK_IMTUB', 'SAS_WEB', 'SAS_IMGS', 'SAS_VIDEO', 'SAS_IMTUB', 'SAS_DIVERSITY2',
                             'AMS_WEB', 'AMS_IMGS', 'AMS_VIDEO', 'AMS_IMTUB', 'AMS_DIVERSITY2'])

        if len(NONMAX_GROUPS & set(runner.groups)):
            return False
        return True

    def run(self, runner):
        if self._always_at_maximum(runner):
            freq = _getFreq(runner)
            hiFreq = self.MODELS_DATA[runner.model][1]
            if freq < hiFreq:
                _setFreq(runner, hiFreq)
                _doLog('INFO', '%s: TB switched on: %s -> %s' % (self.name(), freq, hiFreq))
            return

        now = time.time()
        idle = _getRawIdle(runner)

        self.data.append((now, idle))
        self.data = filter(lambda (x, y): now - x < self.REACTION_TIME, self.data)

        if len(self.data) < int(self.REACTION_TIME / self.TIMEOUT) - 1:
            return

        first_now, first_idle = self.data[0]
        last_now, last_idle = self.data[-1]
        avgIdle = (last_idle - first_idle) / (last_now - first_now)

        freq = _getFreq(runner)
        lowFreq, hiFreq = self.MODELS_DATA[runner.model]
        if avgIdle < self.LOW_IDLE_MARK and freq < hiFreq: # lowidle, but TB off
            _setFreq(runner, hiFreq)
            _doLog('INFO', '%s: TB switched on: %s -> %s' % (self.name(), freq, hiFreq))
        if avgIdle > self.HI_IDLE_MARK and freq > lowFreq:
            _setFreq(runner, lowFreq)
            _doLog('INFO', '%s: TB switched off: %s -> %s' % (self.name(), freq, lowFreq))

'''
Класс, который занимается установкой правильной (максимальной) частоты. Не уменьшает частоту, если она выше номинальной (включен TB).
Влияет на производительность FreeBSD-машин, у которых после ребута установилась слишком маленькая частота. Помогает ли это Linux-у, не понятно. 
'''
class TFreqChecker(TBaseChecker):
    TIMEOUT = 60

    MODELS_DATA = {
        'E5345' : 2333,
        'E5410' : 2333,
        'L5410' : 2333,
        'E5440' : 2826,
        'E5462' : 2803,
        'E5530' : 2400,
        'E5620' : 2400,
        'E5630' : 2533,
        'E5645' : 2400,
        'E5-2660' : 2200,
        'E5-2667' : 2900,
        'AMD6168' : 1900,
        'AMD6172' : 2100,
        'AMD6176' : 2300,
        'AMD6274' : 2200,
        'X5675' : 3066,
    }

    def __init__(self, runner):
        TBaseChecker.__init__(self)

    def enabled(self, runner):
        return runner.model in self.MODELS_DATA

    def name(self):
        return 'FreqChecker'

    def run(self, runner):
        oldfreq = _getFreq(runner)
        if oldfreq < self.MODELS_DATA[runner.model]: # do nothing if frequency higher than required
            _setFreq(runner, self.MODELS_DATA[runner.model])
            _doLog('INFO', '%s: changed freq: %s -> %s' % (self.name(), oldfreq, self.MODELS_DATA[runner.model]))


'''Класс, который перечитыват таги время от времени.'''
class TGroupsInfoUpdater(TBaseChecker):
    TIMEOUT = 600

    def __init__(self, runner):
        TBaseChecker.__init__(self)

    def enabled(self, runner):
        return True

    def name(self):
        return 'TGroupsInfoUpdater'

    def run(self, runner):
        try:
            outdata, errdata = _runCommand(['/db/bin/bsconfig', 'listtags', '--yasm-format'])
            runner.groups = filter(lambda x: re.match('[A-Z0-9_]+$', x), outdata.split())
        except: # do not have bsconfig
            runner.groups = []

        _doLog('INFO', '%s: new groups: %s' % (self.name(), ' '.join(runner.groups)))

class TRunner:
    def __init__(self):
        f = open(PID_FILE, 'w')
        print >>f, os.getpid()
        f.close()

        self.os = os.uname()[0]
        if self.os not in ['FreeBSD', 'Linux']:
            _doLog('ERROR', 'Unsupported os %s, exiting...' % self.os)
            sys.exit(1)

        if self.os == 'FreeBSD':
            self.libc = ctypes.CDLL('libc.so')

        self.model = self._model()
        self.ncpu = self._ncpu()
        self.groups = []

        # fill checkers
        self.checkers = []
        for checker in [TGroupsInfoUpdater(self), TTBChecker(self), TFreqChecker(self), TCStateChecker(self)]:
            if checker.enabled(self):
                _doLog('INFO', '%s: enabled with timeout %s' % (checker.name(), checker.TIMEOUT))
                self.checkers.append(checker)
            else:
                _doLog('INFO', '%s: disabled' % checker.name())


    def _model(self):
        MODELS = {
            'Intel(R) Xeon(R) CPU           E5645  @ 2.40GHz' : 'E5645',
            'Intel(R) Xeon(R) CPU E5-2660 0 @ 2.20GHz'        : 'E5-2660',
            'Intel(R) Xeon(R) CPU           E5630  @ 2.53GHz' : 'E5630',
            'Intel(R) Xeon(R) CPU           E5540  @ 2.53GHz' : 'E5540',
            'AMD Opteron(TM) Processor 6274'                  : 'AMD6274',
            'Intel(R) Xeon(R) CPU           X5450  @ 3.00GHz' : 'X5450',
            'Intel(R) Xeon(R) CPU           E5440  @ 2.83GHz' : 'E5440',
            'AMD Opteron(tm) Processor 6176'                  : 'AMD6176',
            'Intel(R) Xeon(R) CPU           E5530  @ 2.40GHz' : 'E5530',
            'Intel(R) Xeon(R) CPU           X5675  @ 3.07GHz' : 'X5675',
            'Intel(R) Xeon(R) CPU           L5410  @ 2.33GHz' : 'L5410',
            'Intel(R) Xeon(R) CPU           E5620  @ 2.40GHz' : 'E5620',
            'Intel(R) Xeon(R) CPU           E5345  @ 2.33GHz' : 'E5345',
            'Intel(R) Xeon(R) CPU           E5410  @ 2.33GHz' : 'E5410',
            'Intel(R) Xeon(R) CPU           E5462  @ 2.80GHz' : 'E5462',
            'Intel(R) Xeon(R) CPU           E5630  @ 2.53GHz' : 'E5630',
            'AMD Opteron(tm) Processor 6168'                  : 'AMD6168',
            'Intel(R) Xeon(R) CPU E5-2667 0 @ 2.90GHz'        : 'E5-2667',
        }

        if self.os == 'FreeBSD':
            model = (ctypes.c_char * 1000)()
            ln = ctypes.c_size_t(1000)
            rc = self.libc.sysctlbyname('hw.model', model, ctypes.byref(ln), None, 0)
            if rc == -1:
                raise Exception, 'Sysctl for getting hw.model failed'
            model = model.value
        elif self.os == 'Linux':
            model = re.search('model name([^\n]*)', open('/proc/cpuinfo').read()).group(0).partition(': ')[2]

        if model not in MODELS:
            _doLog('ERROR', 'Unknown model <<%s>>' % model)
            return model
        return MODELS[model]

    def _ncpu(self):
        if self.os == 'FreeBSD':
            ncpu = ctypes.c_uint()
            ln = ctypes.c_size_t(ctypes.sizeof(ctypes.c_uint))
            rc = self.libc.sysctlbyname('hw.ncpu', ctypes.byref(ncpu), ctypes.byref(ln), None, 0)
            if rc == -1:
                raise Exception, 'Sysctl for getting hw.ncpu failed'
            return ncpu.value
        elif self.os == 'Linux':
            return int(os.sysconf('SC_NPROCESSORS_ONLN'))

    def _groups(self):
        try:
            outdata, errdata = _runCommand(['/db/bin/bsconfig', 'listtags', '--yasm-format'])
            return filter(lambda x: re.match('[A-Z0-9_]+$', x), outdata.split())
        except: # do not have bsconfig
            return []

    def run(self):
        stopped = False

        while True:
            if os.path.exists(STOP_FILE):
                if not stopped:
                    _doLog('INFO', 'Main: stopping (file %s exists)' % STOP_FILE)
                stopped = True
            else:
                if stopped:
                    _doLog('INFO', 'Main: starting (file %s does not exists)' % STOP_FILE)
                stopped = False

                for checker in self.checkers:
                    if checker.runAt < time.time():
                        try:
                            checker.run(self)
                        except Exception, e:
                            _doLog('ERROR', '%s: got error: %s' % (checker.name(), e))

                        checker.runAt = max(checker.runAt + checker.TIMEOUT, time.time())

            time.sleep(1)

def _isRunning():
    if not os.path.exists(PID_FILE):
        return False

    pid = open(PID_FILE).read().strip()

    # check if process exists
    stdoutdata, stderrdata = _runCommand(['ps', '-p', pid, '-o', 'pid'], fail_nonzero_code = False)
    if not len(filter(lambda x: x == pid, stdoutdata.split())):
        os.unlink(PID_FILE)
        return False

    # check if proccess is me
    cmdline = open(os.path.abspath(os.path.join('/proc', pid, 'cmdline',))).read()
    if string.find(cmdline, os.path.basename(sys.argv[0])) == -1: #FIXME: very inaccurate
        os.unlink(PID_FILE)
        return False

    return True

if __name__ == '__main__':
    try:
        is_running = _isRunning()
        if is_running:
            _doLog('INFO', 'Service already running, exiting')
            sys.exit(0)
        else:
            _doLog('INFO', 'Service not running, strating ...')
    except Exception, e:
        _doLog('ERROR', 'Error while check running: %s' %e)
        sys.exit(1)

    try:
        runner = TRunner()
        runner.run()
    except Exception, e:
        _doLog('ERROR', 'Got exception during run: %s' % e)
        os.unlink(PID_FILE)
    finally:
        os.unlink(PID_FILE)
