#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# last edited by Viacheslav Biriukov 21.05.2013
#

import os
import re
import subprocess
import sys
import statvfs
import socket
import threading
import platform
import signal
import time
import getopt
import logging
import logging.handlers

"""
script for checking amount of tcpdump raw output files
"""

SIZE = 600000  # in packets ~400Mb
DELAY = 1  # seconds
CRITICAL = 30  # minimum free space, program won`t start tcpdump, percents
TDIR = "/var/tmp"
DIR = TDIR + "/dump/"
MAXDUMP = 2048 #limit one log size in MB
MAXRAWFILES = 4 # limit max count of the raw files
#LOGCOUNT = 400  # max number of gziped files
LOGCOUNT = 30
NAME = DIR + ("/dump_raw_%s.log" % socket.gethostname())
CFORMAT = "gz"  # gziped files
PIDFILE = "/var/run/td_control.pid"

_os_type = platform.system()
_mv_bin = '/bin/mv'
_td_bin = '/usr/sbin/tcpdump'

LOG_DIR = DIR + 'td_control_logs/'
LOG_FILENAME =  LOG_DIR + 'app.log'
logger = logging.getLogger("td_control2")

if _os_type == 'Linux':
    _gzip_bin = '/bin/gzip -1'
    _ifconfig_cmd = "/sbin/ifconfig -s |/bin/grep eth |/usr/bin/awk '{print $1}'"
elif _os_type in ('FreeBSD', 'Darwin'):
    _gzip_bin = '/usr/bin/gzip -1'
    _ifconfig_cmd = "/sbin/ifconfig -l -u"
else:
    sys.stderr.write("Error: Unknown os\n") 
    os.exit()

is_active = True

def init_logs():

    # init app_logs dir
    if not os.path.exists(LOG_DIR):
        try:
            os.makedirs(LOG_DIR)
        except OSError, e:
            logger.error(e)
            sys.exit(1)

    # create logger
    logger.setLevel(logging.DEBUG)
    # create console handler and set level to debug
    ch = logging.handlers.RotatingFileHandler(LOG_FILENAME, 'a', 50000, 10)
    ch.setLevel(logging.DEBUG)
    # create formatter
    formatter = logging.Formatter("%(asctime)s - %(name)s - %(levelname)s - %(message)s")
    # add formatter to ch
    ch.setFormatter(formatter)
    # add ch to logger
    logger.addHandler(ch)

def become_daemon(our_home_dir=DIR, out_log='/dev/null',
                      err_log='/dev/null', umask=0o022, pidfile=PIDFILE):
        "Robustly turn into a UNIX daemon, running in our_home_dir."
        # First fork
        try:
            if os.fork() > 0:
                sys.exit(0)     # kill off parent
        except OSError as e:
            sys.stderr.write("fork #1 failed: (%d) %s\n" % (e.errno, e.strerror))
            sys.exit(1)
        os.setsid()
        os.chdir(our_home_dir)
        os.umask(umask)

        # Second fork
        try:
            if os.fork() > 0:
                os._exit(0)
        except OSError as e:
            sys.stderr.write("fork #2 failed: (%d) %s\n" % (e.errno, e.strerror))
            os._exit(1)
        si = open('/dev/null', 'r')
        so = open(out_log, 'a+', 0)
        se = open(err_log, 'a+', 0)
        os.dup2(si.fileno(), sys.stdin.fileno())
        os.dup2(so.fileno(), sys.stdout.fileno())
        os.dup2(se.fileno(), sys.stderr.fileno())
        # Set custom file descriptors so that they get proper buffering.
        sys.stdout, sys.stderr = so, se

        pid = str(os.getpid())
        try:
            file(pidfile, 'w').write(pid)
        except OSError, e:
            os.exit(1)


class DumpTr(threading.Thread):
    """
    Dumper thread class
    """
    def __init__(self, iface):
        super(DumpTr, self).__init__()
        self.iface = iface
        self._stop = threading.Event()
        self.dir = DIR

    def run(self):
            # limit for raw files
            rawfiles_count = 0
            for filename in os.listdir(self.dir):
                pattern = re.compile('\.log\.tmp\.(\d+)$')
                matched = pattern.search(filename) 
                if matched:
                    rawfiles_count += 1
                    if rawfiles_count > MAXRAWFILES:
                        logger.warning('reached max raw files count %s' % MAXRAWFILES)
                        return

            # dump traffic
            now = int(time.time())

            CMD = ("%s -i %s -n -p -c  %d  -w - >> %s  " %
                   (_td_bin, self.iface, SIZE, NAME))
            proc = subprocess.Popen(CMD, stdout=subprocess.PIPE,  stderr=subprocess.PIPE, shell=True)
            #os.popen("touch /tmp/dump/dump_raw_bsd.log.0")
            # mv filename.log  -> filename.log.0 if filename.log.0 not exist
            proc.wait()
            logger.debug(proc.stdout.read())
            logger.debug(proc.stderr.read())
            # move for gzipping
            gName = NAME + '.tmp.' + str(now)
            if not os.path.isfile(gName):
                CMD = ("%s %s %s " % (_mv_bin, NAME, gName))
                proc = subprocess.Popen(CMD, shell=True, stdout=subprocess.PIPE,  stderr=subprocess.PIPE)
                proc.wait()
                # else do nothing

    def stop(self):
        self._stop.set()

class WatchDog(threading.Thread):
    """
    WatchDog thread class
    """
    def __init__(self, dir, maxfiles):
        super(WatchDog, self).__init__()
        self.maxfiles = maxfiles
        self.listfiles = []  # files in file.log.0.gz format
        self._stop = threading.Event()
        self.dir = dir

    def clean(self):
        for filename in os.listdir(self.dir):
            pattern = re.compile('\.(\d+)\.%s$' % CFORMAT)
            matched = pattern.search(filename)

            if matched:
                #make operation list
                self.listfiles.append(filename)

                if int(matched.group(1)) >= self.maxfiles:
                    #print("%s/%s :deleted" % (DIR,filename))
                    os.remove("%s/%s" % (self.dir, filename))

    def rotate(self):
        count_down = self.maxfiles - 1
        while count_down >= 0:
            for item in self.listfiles:
                pattern = re.compile('\.((\d+)\.%s)$' % CFORMAT)
                matched = pattern.search(item)
                if int(matched.group(2)) == count_down:
        #        print("%s/%s :moved" % (DIR,item))
                    string = (item[:-len(matched.group(1))] +
                              str((count_down + 1)) +
                              "." +
                              CFORMAT)
                    src = self.dir+ "/" + item
                    dst = self.dir + "/" + string
                    if os.path.isfile(src):
                        CMD = "%s %s %s" % (_mv_bin, src, dst)
                        proc = subprocess.Popen(CMD, shell=True, stdout=subprocess.PIPE,  stderr=subprocess.PIPE)
                        proc.wait()
            count_down -= 1

    def move(self):
        zName = NAME + ".0"
        first = True
        for filename in os.listdir(self.dir):
            pattern = re.compile('\.log\.tmp\.(\d+)$')
            matched = pattern.search(filename) 
            if matched:
                # first in 
                if first:
                    older_timestamp = int(matched.group(1))
                    first = False
                    continue
                if int(matched.group(1)) < older_timestamp:
                    older_timestamp = int(matched.group(1))
            
        if not os.path.isfile(zName) and older_timestamp:
                CMD = ("%s %s %s " % (_mv_bin, NAME + '.tmp.' + str(older_timestamp), zName))
                proc = subprocess.Popen(CMD, shell=True, stdout=subprocess.PIPE,  stderr=subprocess.PIPE)
                proc.wait()
                # else do nothing

    def run(self):
        self.clean()
        self.move()
        zName = NAME + ".0"
        if os.path.isfile(zName):
            self.rotate()
            CMD = "%s %s " % (_gzip_bin, zName)
            proc = subprocess.Popen(CMD, shell=True, stdout=subprocess.PIPE,  stderr=subprocess.PIPE)
            proc.wait()

    def stop(self):
        self._stop.set()


def freespace(partition):
    f = os.statvfs(partition)
    free = float((f.f_bavail * f.f_frsize) / 1024 / 1024)  # megabytes
    total = float((f.f_blocks * f.f_frsize) / 1024 / 1024)
    percent = free / (total / 100)
    space = int(percent)
    return space

def check_pid(pidfile):
    """ Check For the existence of a unix pid. """
    if os.path.isfile(pidfile):
        pid = long(open(pidfile, 'r').read())
        try:
          os.kill(pid, 0)
        except OSError:
            print "%s already exists, but isn't running, statring..." % (pidfile, )
        else:
            print "%s already exists, and running %d, exiting..." % (pidfile, pid)
            os._exit(0)
    logger.info("starting...") 
    return True


# decorator for signals
def sig_handler(sig):
    def handler(f):
        signal.signal(sig, f)
        return f
    return handler

@sig_handler(signal.SIGINT)
@sig_handler(signal.SIGTERM)
def exit_handler(sig_num, frame):
    global is_active
    is_active = False
    logger.info('exiting...')
    
@sig_handler(signal.SIGHUP)
def rotate_log(sig_num, frame):
    pass

def usage(e):
    print (str(e))
    sys.stderr.write("Usage: %s -i interface\n" % sys.argv[0])
    sys.exit(2)

def arg_parse():
    # check args
    try:
        myopts, args = getopt.getopt(sys.argv[1:],"i:")
    except getopt.GetoptError as e:
        usage(e)

    # init interface
    iface=''
    iface_list = []
    for o, a in myopts:
        if o == '-i':
            grep_interface = '| grep %s' % a
            if subprocess.Popen(_ifconfig_cmd + grep_interface, stdout=subprocess.PIPE, shell=True).stdout.read():
                iface=a
            else:
                usage('Wrong -i parametr: there is no such intarface')

    if not iface:
        iface_list = subprocess.Popen(_ifconfig_cmd, stdout=subprocess.PIPE, shell=True).stdout.read().split()

    for interface in iface_list:
        p = re.compile('ipfw*|lo*|vlan*')
        if not p.match(interface):
            iface = interface
            break  # find first active and exit

    if not iface:
        logger.error("Error: Iface not found\n")
        os.exit() 

    return iface

def main(iface):

    # partitions
    PARTITION = os.path.realpath(os.path.normpath(TDIR))

    if PARTITION:
        p_arr = PARTITION.split(os.sep)
        PARTITION = os.sep + p_arr[1]
    else:
        logger.error("Error: No valid partition")
        sys.exit(1)

    
    # init log dir
    if not os.path.exists(DIR):
        try:
            os.makedirs(DIR)
        except OSError, e:
            logger.error(e)
            sys.exit(1)


    logcount = LOGCOUNT

    # start threads
    try:
        d = DumpTr(iface)
        d.start()


        w = WatchDog(DIR, logcount)
        w.start()
    except Exception, e:
        logger.error(e)


    while is_active:
        time.sleep(1)

        # restart watchDog
        if not w.isAlive():
            w = WatchDog(DIR, logcount)
            w.start()

        # space monitoring
        if freespace(PARTITION) < CRITICAL:
            if logcount > 5:
                logcount -= 5
                logger.error("Error: not enough disk space, logcount decrised to %d\n" % int(logcount))
            else:
                logger.error("Error: not enough disk space, logcount is minimal and not enough...exiting\n") 
                sys.exit(1)
        else:
            if not d.isAlive():
                logger.debug("nope")
                d = DumpTr(iface)
                d.start()
            else:
                logger.debug("running")
                time.sleep(1)

            # auto increasing log count after space issue
            if logcount < LOGCOUNT:
                logger.error("Info: logcount  has been increased from %s to %s\n" % (logcount, logcount+5 )) 
                logcount += 5




    # waiting for graceful shutdown all threads
    logger.info('waiting for graceful exiting...')
    if d.isAlive():
        d.join()
    if w.isAlive():
        w.join()

    logger.info('exited')
    try:
        os.unlink(PIDFILE)
    except Exception, e:
        logger.error(e)    
    sys.exit(0)


if __name__ == '__main__':
    iface = arg_parse()
    init_logs()
    if check_pid(PIDFILE):
        become_daemon()
        main(iface)



