#!/bin/sh -e
#
# Script for upgrade firmware on Dell PERC 5/6 RAID Adapters.
#
# $Id$

PATH=/bin:/sbin:/usr/bin:/usr/sbin:/usr/local/bin:/usr/local/sbin
export PATH


#-- Subroutines --------------------------------------------------------

err()
{       
	local _exitval

	_exitval=$1
	shift

	echo 1>&2 "ERROR: $*"
	exit $_exitval
}

verbose()
{
	local _d

	if [ -n "$verbose" ]; then
		_d=$(date "+%F %T")
		echo "[${_d}] VERBOSE: $*"
	fi
}

usage()
{
	echo 1>&2 "Usage: ${thiscmd} [-v]"
	echo 1>&2 "Options:"
	echo 1>&2 "  -v           verbose output"

	exit 1
}

get_options()
{
	local _opt

	while getopts "v" _opt; do
		case "$_opt" in
			v) verbose="yes" ;;
			*) usage ;;
		esac
	done

	shift $(($OPTIND - 1))

	if [ $# -ne 0 ]; then
		usage
	fi
}

check_depends()
{
	if ! which -s $ctl; then
		err 1 "Can't find ${ctl} utility !"
	fi

	case "$ctl" in
		megacli) ;;
		*) err 1 "Utility ${ctl} is not supported !" ;;
	esac

	verbose "Use utility \"${ctl}\" for operations."
}

add_to_tmp_files()
{
	tmp_files="${tmp_files}${tmp_files:+ }$*"
}

get_adp_properties()
{
	eval $($ctl -adpallinfo -a$_adp -nolog | awk '
		/Product Name/ {
			sub(/\//,"_",$5)
			_product=sprintf("%s_%s", $4, $5)
		}
		/FW Package Build:/ {
			_fw_cur_ver=sprintf("%s", $4)
		}
		END{
			printf "_product=\"%s\";\n", _product
			printf "_fw_cur_ver=\"%s\";\n", _fw_cur_ver
		}
	')

	if [ -z "${_product}" -o -z "${_fw_cur_ver}" ]; then
		err 1 "There is not enough info about adapter ${_adp}" \
			"(_product: ${_product}," \
			"_fw_cur_ver: ${_fw_cur_ver}) !"
	fi

	verbose " Product: ${_product}."
	verbose " Current firmware: ${_fw_cur_ver}."
}

check_firmware_version()
{
	verbose " Info URL: ${_url_info}."

	eval $(fetch $fetch_args -o - $_url_info | awk '
		/MD5/ {
			sub(/\(/,"",$2)
			sub(/\)/,"",$2)
			_fw_file=$2
			_fw_md5=$4
		}
		/PACKAGE/ {
			sub(/.*=/,"",$1)
			_fw_new_ver=$1
		}
		END{
			printf "_fw_file=\"%s\";\n", _fw_file
			printf "_fw_md5=\"%s\";\n", _fw_md5
			printf "_fw_new_ver=\"%s\";\n", _fw_new_ver
		}
	')

	if [ -z "${_fw_file}" -o -z "${_fw_md5}" -o \
		-z "${_fw_new_ver}" ]
	then
		err 1 "There is not enough info about firmware" \
			"on ${_url_info}" \
			"(_fw_file: ${_fw_file}," \
			"_fw_md5: ${_fw_md5}," \
			"_fw_new_ver: ${_fw_new_ver}) !"
	fi

	if [ "${_fw_cur_ver}" = "${_fw_new_ver}" ]; then
		verbose " This adapter already has the same version" \
			"of firmware as on update server."
		return 1
	fi

	verbose " New firmware: ${_fw_new_ver}."
}

fetch_firmware()
{
	add_to_tmp_files $_local_fw

	if md5 -q $_local_fw 2>/dev/null | grep -q "$_fw_md5"; then
		verbose " New firmware is already fetched to" \
			"${_local_fw}."
		return 0
	fi

	if ! fetch $fetch_args -o $_local_fw $_url_fw; then
		err 1 "Can't fetch firmware from ${_url_fw} to" \
			"${_local_fw} !"
	fi

	verbose " Firmware is fetched from ${_url_fw} to ${_local_fw}."

	if ! md5 -q $_local_fw | grep -q "$_fw_md5"; then
		err 1 "MD5 for ${_local_fw} doesn't match !"
	fi

	verbose " MD5 for ${_local_fw} is correct."
}

flash_firmware()
{
	local _download_done _flash_done _exit_code

	eval $($ctl -AdpFwFlash -f $_local_fw -a$_adp | awk '
		BEGIN{
			_download_done = _flash_done = 0
		}
		/Download Completed./ {
			_download_done=1
		}
		/Adapter [0-9]+: Flash Completed./ {
			_flash_done=1
		}
		/Exit Code:/ {
			_exit_code=$3
		}
		END{
			printf "_download_done=\"%d\";\n", _download_done
			printf "_flash_done=\"%d\";\n", _flash_done
			printf "_exit_code=\"%s\";\n", _exit_code
		}
	')

	if ! [ $_download_done -eq 1 -a $_flash_done -eq 1 -a \
		"$_exit_code" = "0x00" ]
	then
		err 1 "Can't flash ${_local_fw} on adapter ${_adp} (" \
			"_download_done = ${_download_done}," \
			"_flash_done = ${_flash_done}," \
			"_exit_code = ${_exit_code}) !"
	fi

	verbose " Firmware is flashed."
}

upgrade_firmware()
{
	local _adp _product _url_info
	local _fw_cur_ver _fw_new_ver _fw_file _fw_md5

	_adp="$1"

	if [ -z "${_adp}" ]; then
		err 1 "Adapter is not defined !"
	fi

	verbose "Select adapter ${adp}."

	get_adp_properties

	_url_info="${download_root}/${_product}/${info_file}"

	if ! check_firmware_version; then
		# Firmware is upgraded already
		return 0
	fi

	_url_fw="${download_root}/${_product}/${_fw_file}"
	_local_fw="${tmp_dir}/${_fw_file}"

	fetch_firmware
	flash_firmware

	verbose "Adapter ${_adp} is handled."
}

cleanup()
{
	if [ -n "${tmp_files}" ]; then
		rm -f $tmp_files 2>/dev/null || true
	fi
}


#-- Variables ----------------------------------------------------------

thiscmd=$(basename $0)

ctl="megacli"
mgm_iface="/dev/mfi"

fetch_args="-T 15 -q"
download_root="https://distillatory.yandex.ru/packages/firmware/dell"
info_file="INFO"

tmp_dir="${TMPDIR:-"/tmp"}"


#-- Main ---------------------------------------------------------------

trap cleanup EXIT

get_options $*
check_depends

for adp in $(ls -1 ${mgm_iface}? | sed -e "s,${mgm_iface},,")
do
	upgrade_firmware $adp
done

