#!/usr/bin/env python
# coding: utf-8

import argparse
import glob
import json
import logging
import os
import re
import shlex
import socket
import subprocess
import time
import urllib2
import sys
import fcntl


LOCK_FILE = '/var/tmp/ip_tunnel.lock'
RT_TOKEN = 'Eisa2eesoBaevoochairaibu'
SUDO_IP_CMD = '/usr/bin/sudo -n /sbin/ip'
TUN_NAME = 'ipip6-ext0'
A_TAG = 'a_ipip6_ext_tunnel'
IPv4_NETS = {
    'ext': {
        'myt':'178.154.167.0/24',
        'iva':'178.154.167.0/24',
        'fol':'178.154.167.0/24',
        'ugr':'178.154.167.0/24',
        'sas':'93.158.129.0/24',
        'sas-09':'95.108.153.0/24',
        'man1':'87.250.245.0/24',
        'man2':'87.250.245.0/24',
        'man-3':'87.250.245.0/24',
        'man-4':'5.45.225.128/25',
        'man-5':'87.250.245.0/24',
        'vla':'93.158.163.0/24'
    },
    'int': {
        'myt':'87.250.237.120/29',
        'man1':'93.158.153.8/29',
        'man2':'93.158.153.8/29',
        'man-3':'93.158.153.8/29',
        'man-4':'93.158.153.8/29',
        'man-5':'93.158.153.8/29',
        'sas':'141.8.182.32/29',
        'vla':'93.158.162.112/29'
    }
}
logging.basicConfig(format='[%(asctime)s]:%(levelname)s %(message)s', level = logging.DEBUG)


class Iss(object):
    def __init__(self):
        self.iss_host = 'localhost'
        self.iss_port = '25536'

    def run(self, handle):
        request = urllib2.Request('http://%s:%s/%s' %(self.iss_host, self.iss_port, handle))
        try:
            response = urllib2.urlopen(request, timeout=20)
            return response.read()
        except urllib2.HTTPError as err:
            logging.critical('ISS HTTPError %d: %s', err.code, err)
        except urllib2.URLError as err:
            logging.critical('ISS URLError %s: %s', err.reason, err)
        except Exception as err:
            logging.critical('ISS Unhandled exception: %s', err)

    def get_listtags(self):
        listtags_dict = {}
        logging.info('Getting listtags from ISS')
        iss_data = self.run("instances")
        self.iss_json = json.loads(iss_data)
        for inst in self.iss_json:
            listtags_dict[inst.get('metaContainerPath')] = inst.get('instanceData').get('properties/tags').split()
        return listtags_dict


class RT(object):
    """empty doc string"""

    def get_map(self):
        self.request = urllib2.Request('https://racktables.yandex.net/export/map64.json')
        return self.run()

    def get_ip(self, fqdn, net):
        self.request = urllib2.Request('https://racktables.yandex.net/export/map64request.php')
        self.request.add_header('Accept', 'application/json')
        self.request.add_header('Auth-Token', RT_TOKEN)
        self.request.get_method = lambda: 'PUT'
        self.request.add_data(json.dumps({
            'fqdn': fqdn,
            'net': net
        }))
        return self.run()

    def del_ip(self, fqdn, net):
        self.request = urllib2.Request('https://racktables.yandex.net/export/map64request.php')
        self.request.add_header('Accept', 'application/json')
        self.request.add_header('Auth-Token', RT_TOKEN)
        self.request.get_method = lambda: 'DELETE'
        self.request.add_data(json.dumps({
            'fqdn': fqdn,
            'net': net,
        }))
        logging.info('Delete IP from RT')
        return self.run()

    def run(self):
        found = False
        attempts = 4
        logging.debug(self.request.get_method()+" "+self.request.get_full_url()+" "+str(self.request.get_data()))
        for attempt in xrange(attempts):
            try:
                response = urllib2.urlopen(self.request, timeout=10)
                result = json.load(response)
                return result
            except urllib2.HTTPError as err:
                logging.warn('RT HTTPError %d: %s', err.code, err)
                time.sleep(2 ** attempt)
            except urllib2.URLError as err:
                logging.warn('RT URLError %s: %s', err.reason, err)
                time.sleep(2 ** attempt)
            except Exception as err:
                logging.critical('RT Unhandled exception: %s', err)
                time.sleep(2 ** attempt)
        if not found:
            logging.critical('Failed attempts in requesting ipv4_ip for tunnel from racktables')
            sys.exit(1)


class TunnelCommon(object):
    """ Common """
    def __init__(self, tunnel_type, geo):
        self.tunnel_mtu = 1400
        self.tunnel_type = tunnel_type
        self.geo = geo
        self.net = IPv4_NETS[self.tunnel_type][self.geo]
        self.rt = RT()

    @staticmethod
    def cmd_string_to_list(cmd=None):
        """convert cmd string to list"""
        if isinstance(cmd, basestring):
            return shlex.split(cmd)
        elif isinstance(cmd, list):
            return cmd

    def run_subprocess(self, cmd):
        """Обертка над subprocess.check_output"""

        logging.debug('Running command: %s', cmd)
        try:
            subprocess.check_output(
                self.cmd_string_to_list(cmd),
                stderr=subprocess.STDOUT
            )
        except subprocess.CalledProcessError as err:
            if 'password is required to run sudo' in err.output:
                logging.critical('Not enough rights to run sudo %s', cmd)
                raise RuntimeError
            else:
                logging.error('Subprocess exited with non-zero cede: %s', err)
                raise RuntimeError
        except Exception as err:
            logging.error('Error occurs: %s', err)
            raise RuntimeError

    @staticmethod
    def ip_route_get():
        """Получаем и парсим результат работы функции ip_route_get"""
        # Паттерн для строк вида
        # 2a02:6b8::3 from :: via fe80::1 dev eth1  src 2a02:6b8:0:1498::b29a:8df9  metric 0
        # 77.88.8.8 via 178.154.141.254 dev eth1  src 178.154.141.249
        line_re = re.compile(r'[.:/0-9a-z]+\s+from\s+::\s+?via\s+([.:/0-9a-z]+)\s+dev\s+(\w+).+src\s+([.:/0-9a-z]+)')
        result = dict()
        for proto in ['ipv4', 'ipv6']:
            dst_ip = '77.88.8.8' if proto == 'ipv4' else '2a02:6b8::3'
            try:
                output = subprocess.check_output(
                    shlex.split('/sbin/ip route get {}'.format(dst_ip)),
                    stderr=subprocess.STDOUT
                )
            except subprocess.CalledProcessError as err:
                if 'Network is unreachable' in err.output:
                    logging.warn('It seems host without %s connectivity: %s', proto, err.output)
                else:
                    logging.error('Subprocess exited witn non-zero code: %s', err)
                result['empty'] = None
            except Exception as err:
                logging.error('Error occurs: %s', err)
                result['empty'] = None
            else:
                for line in output.splitlines():
                    if line_re.search(line):
                        line_dict = line_re.search(line).groups()
                        if proto == 'ipv6':
                            result[proto] = {
                                'gw': line_dict[0],
                                'dev': line_dict[1],
                                'ip': line_dict[2]
                            }
                        elif proto == 'ipv4':
                            result[proto] = {
                                'gw': line_dict[0],
                                'dev': line_dict[1],
                                'ip': line_dict[2]
                            }
        return result

    def ip_link_show(self):
        """
        Метод для получения состояния текущих интерфейсов.
        Возращает словарь вида {'interface_name': 'interface_status'},
        либо {'empty': None}, если процесс работы завершился неудачей
        """
        try:
            output = subprocess.check_output(
                self.cmd_string_to_list('/sbin/ip link show')
            )
        except subprocess.CalledProcessError as err:
            logging.error('Subprocess exited witn non-zero code: %s', err)
            return {'empty': None}
        except Exception as err:
            logging.error('Error occurs: %s', err)
            return {'empty': None}
        else:
            result = dict()
            # Паттерн для строк вида
            # 5: dummy0: <BROADCAST,NOARP,UP,LOWER_UP> mtu 8910
            #           qdisc htb state UNKNOWN mode DEFAULT group default
            line_re = re.compile(r'\d+:\s+[@0-9a-zA-Z-]+:\s+')
            for line in output.splitlines():
                if line_re.search(line):
                    _, name, options = line.split(': ')
                    result[name.replace('@NONE', '')] = options.split()[6]
            return result

    def setup_ipip6_tunnel(self, addrs):
        """ Missing method docstring """
        self.run_subprocess(
            '{} -6 tunnel add {} mode ipip6 remote {} local {}'.format(
                SUDO_IP_CMD,
                TUN_NAME,
                '2a02:6b8:b010:a0ff::1',
                addrs['addr6']
            )
        )
        self.run_subprocess(
            '{} addr add {} dev {}'.format(SUDO_IP_CMD, addrs['tunip'], TUN_NAME)
        )
        self.run_subprocess('{} link set dev {} up'.format(SUDO_IP_CMD, TUN_NAME))
        self.run_subprocess('{} link set dev {} mtu {}'.format(
            SUDO_IP_CMD,
            TUN_NAME,
            self.tunnel_mtu,
        ))
        self.run_subprocess('{} route replace default dev {}'.format(SUDO_IP_CMD, TUN_NAME))


    def ipip6_tunnel_stop(self):
        """ Missing method docstring """
        logging.info('Stopping ipip6 tunnel')
        ip_data = self.ip_route_get()
        map64_data = self.rt.get_map()
        for ip in map64_data:
            if map64_data[ip]['fqdn'] == ip_data['ipv6']['ip']:
                logging.info('IP found in RT map64.json')
                self.rt.del_ip(ip_data['ipv6']['ip'], self.net)
                break
        if self.ip_link_show().get(TUN_NAME):
            logging.info('Tunnel link %s exist.', TUN_NAME)
            self.run_subprocess('{} link set dev {} down'.format(SUDO_IP_CMD, TUN_NAME))
            self.run_subprocess('{} -6 tunnel del {}'.format(SUDO_IP_CMD, TUN_NAME))
        else:
            logging.info('Tunnel link %s not found.', TUN_NAME)


    def ipip6_tunnel_start(self):
        """ Missing method docstring """
        logging.info('Starting ipip6 tunnel')
        ip_data = self.ip_route_get()
        if self.ip_link_show().get(TUN_NAME):
            logging.info('Tunnel link '+TUN_NAME+' already exist. Quit')
        elif ip_data.get('ipv4') and ip_data['ipv4'].get('ip'):
            logging.info('IPv4 route already exist. Quit')
        elif 'empty' in ip_data and len(ip_data) == 1:
            raise RuntimeError('It seems there are no valid network on host. Quit')
        else:
            logging.info('Tunnel link %s not found.', TUN_NAME)
            logging.info('IPv4 route not found.')
            addrs = {}
            ### maybe raise
            #map64_data = self.rt.get_map()
            #for ip in map64_data:
            #    if map64_data[ip]['fqdn'] == ip_data['ipv6']['ip']:
            #        addrs['addr6'] = map64_data[ip]['addr6']
            #        addrs['tunip'] = ip
            #        break
            if not addrs:
                ipv4_addr_struct = self.rt.get_ip(ip_data['ipv6']['ip'], self.net)
                addrs['addr6'] = ip_data['ipv6']['ip']
                addrs['tunip'] = ipv4_addr_struct['addr']
            self.setup_ipip6_tunnel(addrs)


def main():
    """Функция парсинга входящих аргументов"""
    parser = argparse.ArgumentParser(formatter_class=argparse.RawTextHelpFormatter)
    parser.add_argument('action', type=str, choices=['start', 'stop', 'restart', 'force-stop'], help='''
        Available actions are:
        * start - set up tunnel for ipv4 connectivity
        * stop - add new tunnel for ipv4 connectivity
        * restart - stop and start tunnel
        * force-stop - force stop tunnel'''
    )
    parser.add_argument('--geo', type = str, dest = 'geo', choices = sorted(list(set([y for x in IPv4_NETS.values() for y in x.keys()]))))
    parser.add_argument('--type', type = str, dest = 'tunnel_type', default = 'ext', choices = ['int', 'ext'])
    args = parser.parse_args()

    lockf = open(LOCK_FILE, 'w')

    try:
        fcntl.flock(lockf, fcntl.LOCK_EX | fcntl.LOCK_NB)
    except Exception as err:
        logging.critical('Resource %s temporarily unavailable', LOCK_FILE)
        sys.exit(1)

    iss = Iss()
    tunnel_type = args.tunnel_type
    if args.geo:
        geo = args.geo
    else:
        bsconfig_itags = os.getenv('BSCONFIG_ITAGS')
        logging.debug('BSCONFIG_ITAGS: %s', os.getenv('BSCONFIG_ITAGS'))
        if bsconfig_itags is None:
            raise RuntimeError("There is no BSCONFIG_ITAGS in env. Quit")
        for g in IPv4_NETS[tunnel_type].keys():
            if re.findall("a_line_"+g, os.getenv('BSCONFIG_ITAGS')):
                geo = g
                break
        if 'geo' not in locals():
            raise RuntimeError("Network prefix for "+tunnel_type+" not found. Quit")
    logging.info('Geo location: %s', geo.upper())
    tunnel = TunnelCommon(tunnel_type, geo)
    if args.action == 'start':
        try:
            tunnel.ipip6_tunnel_start()
        except RuntimeError as err:
            logging.critical('Setup ipip6 tunnel failed: %s', err)
            sys.exit(1)
        else:
            logging.info('Starting ipip6 tunnel have finished')
    elif args.action == 'restart':
        logging.info('Restarting ipip6 tunnel')
        try:
            tunnel.ipip6_tunnel_stop()
            tunnel.ipip6_tunnel_start()
        except RuntimeError as err:
            logging.critical('Restarting ipip6 tunnel failed: %s', err)
            sys.exit(1)
        else:
            logging.info('Restarting ipip6 tunnel have finished')
    elif args.action == 'force-stop':
        try:
            tunnel.ipip6_tunnel_stop()
        except RuntimeError as err:
            logging.critical('Stopping ipip6 tunnel failed: %s', err)
            sys.exit(1)
        else:
            logging.info('Stopping ipip6 tunnel have finished')
    elif args.action == 'stop':
        try:
            listtags_dict = iss.get_listtags()
            inst_with_ipip6_tunnel = []
            for inst in listtags_dict:
                if A_TAG in listtags_dict[inst]:
                    inst_with_ipip6_tunnel.append(inst)
            if len(inst_with_ipip6_tunnel) > 1:
                logging.info('IPv4 using on this host:\n%s', inst_with_ipip6_tunnel)
            else:
                logging.info('Instances with tag %s not found', A_TAG)
                tunnel.ipip6_tunnel_stop()
        except RuntimeError as err:
            logging.critical('Stopping ipip6 tunnel failed: %s', err)
            sys.exit(1)
        else:
            logging.info('Stopping ipip6 tunnel have finished')

    fcntl.flock(lockf, fcntl.LOCK_UN)
if __name__ == '__main__':
    main()
