try:
    import logging
    import os
    import errno

    import click
    import walle_api

    from infra.rtc.rebootctl.lib import script as script_cmd
    from infra.rtc.rebootctl.lib import terminal
except KeyboardInterrupt:
    raise SystemExit(0)


@click.group()
def rebootctl():
    pass


@rebootctl.command(help='Run reboot script')
@click.option('-f', help='Read configuration from file',
              type=click.Path(exists=True, dir_okay=False),
              required=True)
@click.option('-b', '--batch', 'batch', help='Dont ask for confirmation before start reboots',
              default=False, type=bool)
@click.option('-k', '--keep-going', 'keep_going', help='Dont ask for confirmation while executing script',
              default=False, type=bool)
@click.option('-l', '--log-file', 'log_file', help='Log file', default='rebootctl.log', type=str)
@click.option('-v', '--verbosity', 'verbosity', help='Log verbosity',
              type=click.Choice(['INFO', 'WARNING', 'ERROR', 'DEBUG']), default='INFO')
@click.option('-s', '--strict', 'strict', help='Strict hosts resolving(stop on error)', default=True, type=bool)
@click.option('--terminal-out', 'terminal_out', help='User interface out stream. Default: CLI', default=None, type=str)
@click.pass_context
def script(ctx, f, batch, keep_going, log_file, verbosity, strict, terminal_out):
    setup_logging(log_file, verbosity)
    term = setup_terminal(terminal_out)
    try:
        err = script_cmd.run(ctx.obj.get_walle(), f, batch, keep_going, strict, term)
    except KeyboardInterrupt:
        term.info('')
        raise SystemExit(0)
    if err is not None:
        term.info(err)
        raise SystemExit(1)
    raise SystemExit(0)


def setup_terminal(out_filename):
    return terminal.ConsoleTerminal() if out_filename is None else terminal.StreamTerminal(out_filename)


def setup_logging(log_file, verbosity):
    logging.basicConfig(
        format="%(asctime)s [%(levelname)-4.4s] %(name)s: %(message)s",
        level=getattr(logging, verbosity),
        filename=log_file
    )


class Ctx(object):
    def get_walle_token(self):
        token = os.environ.get('WALLE_TOKEN')
        if not token:
            try:
                with open(os.path.expanduser(os.path.join("~", ".wall-e", "access_token"))) as stream:
                    token = stream.read().strip()
            except IOError as exc:
                if exc.errno != errno.ENOENT:
                    raise
        return token

    def get_walle(self):
        return walle_api.WalleClient(access_token=self.get_walle_token())


def main():
    rebootctl(obj=Ctx())


if __name__ == "__main__":
    main()
