import logging
from concurrent import futures

from infra.rtc.rebootctl.proto import reboot_pb2

log = logging.getLogger(__name__)


class Status(object):
    IN_QUEUE = 'IN_QUEUE'
    IN_PROGRESS = 'IN_PROGRESS'
    DONE = 'DONE'
    FAILED = 'FAILED'
    CANCELLED = 'CANCELLED'

    def __init__(self, condition=IN_QUEUE, err=''):
        self.condition = condition
        self.err = err

    def __str__(self):
        return '[{}] {}'.format(self.condition, self.err)

    def cancel(self, err=''):
        self.condition = Status.CANCELLED
        self.err = err

    def in_progress(self):
        self.condition = Status.IN_PROGRESS
        self.err = ''

    def fail(self, err):
        self.condition = Status.FAILED
        self.err = err

    def done(self):
        self.condition = Status.DONE

    def is_terminal(self):
        return self.condition in [Status.DONE, Status.FAILED, Status.CANCELLED]

    def error(self):
        return self.err


class Task(object):
    def __init__(self):
        self._status = Status()

    def _run(self, ctx):
        """
        Initiates action, as in POST /reboot.
        """
        raise NotImplementedError

    def run(self, ctx):
        if ctx.done():
            self._status.cancel(ctx.error())
            return
        self._status.in_progress()
        try:
            self._run(ctx)
        except Exception as e:
            self._status.fail(str(e))
        else:
            self._status.done()

    def status(self):
        return self._status


class Job(object):
    def __init__(self, max_workers=4):
        self.max_workers = max_workers

    def run(self, ctx, tasks):
        ex = futures.ThreadPoolExecutor(self.max_workers)
        fs = []
        for task in tasks:
            fs.append(ex.submit(task.run, ctx))
        return fs


class Sleep(Task):
    """
    Task to hack up while testing TUI in production.
    """
    id = 'sleep'
    NAME = 'SLEEP'

    def __init__(self, w, hostname, reason, options):
        super(Sleep, self).__init__()
        self.w = w
        self.h = hostname
        self.reason = reason
        self.options = options
        self._status = Status()

    def _run(self, _):
        import time
        time.sleep(self.options.timeout)

    @classmethod
    def filter(cls, host):
        if 'health' not in host:
            return False
        if host['status'] != 'ready':
            return False
        return True


class KernelUpdate(Task):
    """
    :type w: walle_api.WalleClient
    """
    id = 'kernel_update'
    NAME = 'KERNEL_UPDATE'

    def __init__(self, w, hostname, reason, options):
        super(KernelUpdate, self).__init__()
        self.w = w
        self.h = hostname
        self.reason = reason
        self.options = options
        self._status = Status()

    def _run(self, ctx):
        log.info("sending %s to reboot", self.h)
        self.w.reboot_host(self.h,
                           ssh='forbid',
                           reason=self.reason,
                           with_auto_healing=self.options.auto_healing == reboot_pb2.KernelUpdate.ENABLE,
                           ignore_cms=self.options.cms_policy == reboot_pb2.KernelUpdate.IGNORE)

    @classmethod
    def filter(cls, host):
        if 'health' not in host:
            return False
        if host['status'] != 'ready':
            return False
        return host['health']['check_statuses'].get('need_reboot_kernel') == 'failed'


class Reboot(Task):
    """
    :type w: walle_api.WalleClient
    """
    id = 'reboot'
    NAME = 'REBOOT'

    def __init__(self, w, hostname, reason, options):
        super(Reboot, self).__init__()
        self.w = w
        self.h = hostname
        self.reason = reason
        self.options = options
        self._status = Status()

    def _run(self, ctx):
        log.info("sending %s to reboot", self.h)
        self.w.reboot_host(self.h,
                           ssh='forbid',
                           reason=self.reason,
                           with_auto_healing=self.options.auto_healing == reboot_pb2.KernelUpdate.ENABLE,
                           ignore_cms=self.options.cms_policy == reboot_pb2.KernelUpdate.IGNORE)

    @classmethod
    def filter(cls, host):
        if 'health' not in host:
            return False
        if host['status'] != 'ready':
            return False
        return True


class Redeploy(Task):
    """
    :type w: walle_api.WalleClient
    """
    id = 'redeploy'
    NAME = 'REDEPLOY'

    def __init__(self, w, hostname, reason, options):
        super(Redeploy, self).__init__()
        self.w = w
        self.h = hostname
        self.reason = reason
        self.options = options
        self._status = Status()

    def _run(self, ctx):
        log.info("sending %s to redeploy", self.h)
        self.w.redeploy_host(self.h,
                             reason=self.reason,
                             with_auto_healing=self.options.auto_healing == reboot_pb2.Redeploy.ENABLE,
                             ignore_cms=self.options.cms_policy == reboot_pb2.Redeploy.IGNORE)

    @classmethod
    def filter(cls, host):
        if 'health' not in host:
            return False
        if host['status'] != 'ready':
            return False
        return True


class FirmwareUpdate(Task):
    """
    :type w: walle_api.WalleClient
    """
    id = 'firmware_update'
    NAME = 'FIRMWARE_UPDATE'

    def __init__(self, w, hostname, reason, options):
        super(FirmwareUpdate, self).__init__()
        self.w = w
        self.h = hostname
        self.reason = reason
        self.options = options
        self._status = Status()

    def _run(self, ctx):
        log.info("sending %s to profile", self.h)
        self.w.profile_host(self.h,
                            profile=self.options.profile_name,
                            profile_tags=list(self.options.profile_tags),
                            reason=self.reason,
                            with_auto_healing=self.options.auto_healing == reboot_pb2.FirmwareUpdate.ENABLE,
                            ignore_cms=self.options.cms_policy == reboot_pb2.FirmwareUpdate.IGNORE)

    @classmethod
    def filter(cls, host):
        if 'health' not in host:
            return False
        if host['status'] != 'ready':
            return False
        return host['health']['check_statuses'].get('walle_firmware') == 'failed'
        # return True


class FirmwareKernelUpdate(Task):
    """
    :type w: walle_api.WalleClient
    """
    id = 'firmware_kernel_update'
    NAME = 'FIRMWARE_KERNEL_UPDATE'

    def __init__(self, w, hostname, reason, options):
        super(FirmwareKernelUpdate, self).__init__()
        self.w = w
        self.h = hostname
        self.reason = reason
        self.options = options
        self._status = Status()

    def _run(self, ctx):
        log.info("sending %s to profile", self.h)
        self.w.profile_host(self.h,
                            profile=self.options.profile_name,
                            profile_tags=list(self.options.profile_tags),
                            reason=self.reason,
                            with_auto_healing=self.options.auto_healing == reboot_pb2.FirmwareUpdate.ENABLE,
                            ignore_cms=self.options.cms_policy == reboot_pb2.FirmwareUpdate.IGNORE)

    @classmethod
    def filter(cls, host):
        if 'health' not in host:
            return False
        if host['status'] != 'ready':
            return False
        checks = host['health']['check_statuses']
        return checks.get('walle_firmware') == 'failed' and checks.get('need_reboot_kernel') == 'failed'


class SoftwareUpdate(Task):
    """
    :type w: walle_api.WalleClient
    """
    id = 'software_update'
    NAME = 'SOFTWARE_UPDATE'

    def __init__(self, w, hostname, reason, options):
        super(SoftwareUpdate, self).__init__()
        self.w = w
        self.h = hostname
        self.reason = reason
        self.options = options
        self._status = Status()

    def _run(self, ctx):
        log.info("sending %s to redeploy", self.h)
        self.w.redeploy_host(self.h,
                             reason=self.reason,
                             with_auto_healing=self.options.auto_healing == reboot_pb2.SoftwareUpdate.ENABLE,
                             ignore_cms=self.options.cms_policy == reboot_pb2.SoftwareUpdate.IGNORE)

    @classmethod
    def filter(cls, host):
        if 'health' not in host:
            return False
        if host['status'] != 'ready':
            return False
        return True


class FixClocksource(Task):
    """
    :type w: walle_api.WalleClient
    """
    id = 'fix_clock_source'
    NAME = 'FIX_CLOCK_SOURCE'

    def __init__(self, w, hostname, reason, options):
        super(FixClocksource, self).__init__()
        self.w = w
        self.h = hostname
        self.reason = reason
        self.options = options
        self._status = Status()

    def _run(self, ctx):
        log.info("sending %s to reboot (fixing clocksource)", self.h)
        self.w.reboot_host(self.h,
                           ssh='forbid',
                           reason=self.reason,
                           with_auto_healing=self.options.auto_healing == reboot_pb2.KernelUpdate.ENABLE,
                           ignore_cms=self.options.cms_policy == reboot_pb2.KernelUpdate.IGNORE)

    @classmethod
    def filter(cls, host):
        if 'health' not in host:
            return False
        if host['status'] != 'ready':
            return False
        return host['health']['check_statuses'].get('walle_clocksource') == 'failed'


class Profile(Task):
    """
    :type w: walle_api.WalleClient
    """
    id = 'profile'
    NAME = 'PROFILE'

    def __init__(self, w, hostname, reason, options):
        super(Profile, self).__init__()
        self.w = w
        self.h = hostname
        self.reason = reason
        self.options = options
        self._status = Status()

    def _run(self, ctx):
        log.info("sending %s to profile", self.h)
        self.w.profile_host(self.h,
                            profile=self.options.profile_name,
                            profile_tags=list(self.options.profile_tags),
                            reason=self.reason,
                            with_auto_healing=self.options.auto_healing == reboot_pb2.Profile.ENABLE,
                            ignore_cms=self.options.cms_policy == reboot_pb2.Profile.IGNORE)

    @classmethod
    def filter(cls, host):
        if 'health' not in host:
            return False
        if host['status'] != 'ready':
            return False
        return True


class PowerOff(Task):
    """
    :type w: walle_api.WalleClient
    """
    id = 'power_off'
    NAME = 'POWER_OFF'

    def __init__(self, w, hostname, reason, options):
        super(PowerOff, self).__init__()
        self.w = w
        self.h = hostname
        self.reason = reason
        self.options = options
        self._status = Status()

    def _run(self, ctx):
        log.info("sending %s to power off", self.h)
        self.w.set_host_maintenance(self.h,
                                    reason=self.reason,
                                    power_off=True,
                                    operation_state=self.options.operation_state,
                                    ticket_key=self.options.ticket_key,
                                    ignore_cms=self.options.cms_policy == reboot_pb2.PowerOff.IGNORE)

    @classmethod
    def filter(cls, host):
        if 'health' not in host:
            return False
        if host['status'] != 'ready':
            return False
        return True


class PowerOn(Task):
    """
    :type w: walle_api.WalleClient
    """
    id = 'power_on'
    NAME = 'POWER_ON'

    def __init__(self, w, hostname, reason, options):
        super(PowerOn, self).__init__()
        self.w = w
        self.h = hostname
        self.reason = reason
        self.options = options
        self._status = Status()

    def _run(self, ctx):
        log.info("sending %s to power on", self.h)
        self.w.set_host_assigned(self.h,
                                 reason=self.reason,
                                 power_on=True)

    @classmethod
    def filter(self, host, ticket_key):
        if host['ticket'] != ticket_key:
            return False
        return True


def get_task_class(name):
    for subclass in Task.__subclasses__():
        if subclass.id == name:
            return subclass
    return None
