from __future__ import print_function
import logging
import color


class Buffer(object):
    def __init__(self, term):
        self._term = term
        self._buf = []

    def write(self, s):
        self._buf.append(s)

    def flush(self):
        self._term.info(''.join(self._buf))
        del self._buf[:]


class Terminal(object):
    def __init__(self):
        pass

    def info(self, message):
        """
        :type: message str
        """
        raise NotImplementedError

    def read_yes_no(self, prompt):
        """
        :type: prompt str
        """
        raise NotImplementedError

    def get_buf(self):
        return Buffer(self)

    def clr(self):
        raise NotImplementedError

    def colorize(self, s, status):
        raise NotImplementedError

    def read_line(self, prompt):
        raise NotImplementedError


class ConsoleTerminal(Terminal):
    def __init__(self):
        Terminal.__init__(self)

    def info(self, *args):
        print(*args)

    def read_yes_no(self, prompt):
        while 1:
            ans = raw_input(prompt + " [Y/n]: ").strip()
            if ans.lower() == 'n':
                return False
            if ans.lower() == 'y':
                return True
            print("Please answer Y[es] or n[o]...")

    def clr(self):
        print(chr(27) + '[2j')
        print('\033c')
        print('\x1bc')

    def colorize(self, s, c):
        return color.colored(s, c)

    def read_line(self, prompt):
        while 1:
            resp = raw_input(prompt).strip()
            if not resp:
                self.info(prompt)
                continue
            return resp


class StreamTerminal(Terminal):
    def __init__(self, out_stream):
        Terminal.__init__(self)
        logger = logging.getLogger("terminal")
        logger.setLevel(logging.INFO)
        fh = logging.FileHandler(out_stream)
        formatter = logging.Formatter('[%(asctime)s] - %(message)s')
        fh.setFormatter(formatter)
        logger.addHandler(fh)
        self.logger = logger

    def info(self, *args):
        self.logger.info(' '.join(str(i) for i in args))

    def read_yes_no(self, prompt):
        self.logger.info("Assuming 'Yes' response to '{}'".format(prompt))
        return True

    def clr(self):
        pass

    def colorize(self, s, c):
        return s

    def read_line(self, prompt):
        raise Exception('User input is not supported, tried to ask: {}'.format(prompt))
