
import datetime
import json
import os

import nirvana.job_context as nv
from infra.rtc.rtc_billing.utils import RTC_SYSTEM_OBJECT_MAPPING, PrepareBillingData

SYSTEM_OUTPUT_MAPPING = {
    "yp": "OUTPUT_YP",
    "gencfg": "OUTPUT_GENCFG",
    "qloud": "OUTPUT_QLOUD"
}

DATE_HOUR_FORMAT = "%Y-%m-%d_%H"


def get_table_path(system, date_hour_string):
    """
    :param date_hour_string: YYYY-mm-dd HH
    """

    if not PrepareBillingData.labels_config[system]["base_path"]:
        return

    return "/".join([PrepareBillingData.labels_config[system]["base_path"], date_hour_string.split("_")[0]])


def create_billing_metrics(date_hour_string):
    """
    :param date_hour_string: YYYY-mm-dd HH
    :return:
    """
    data_to_send = {}
    transformed_datetime = transform_datetime(date_hour_string)

    for data_collector in RTC_SYSTEM_OBJECT_MAPPING.values():
        collector = data_collector("hahn")
        data_to_send[collector.system_name] = []

        for record in collector.prepare_data(get_table_path(collector.system_name, date_hour_string)):
            record["source_wt"], record["usage"]["start"], record["usage"]["finish"] = transformed_datetime, \
                                                                                       transformed_datetime, \
                                                                                       transformed_datetime
            data_to_send[collector.system_name].append(record)

    return data_to_send


def transform_datetime(datehour_string):
    return (datetime.datetime.strptime(datehour_string, DATE_HOUR_FORMAT) - datetime.timedelta(seconds=1)).timestamp()


def get_date_hour_array(start_time, end_time):

    if not end_time:
        return [start_time]

    start_time, end_time = datetime.datetime.strptime(start_time, DATE_HOUR_FORMAT), \
                           datetime.datetime.strptime(end_time, DATE_HOUR_FORMAT)

    dates_to_process = []

    while start_time <= end_time:
        dates_to_process.append(start_time.strftime(DATE_HOUR_FORMAT))
        start_time += datetime.timedelta(hours=1)

    return dates_to_process


if __name__ == '__main__':

    job_context = nv.context()
    outputs = job_context.get_outputs()
    parameters = job_context.get_parameters()

    start, end = parameters.get("DATE_HOUR_STRING_START"), parameters.get("DATE_HOUR_STRING_FINISH")

    dates_to_process = get_date_hour_array(start, end)

    result = {}

    for date_hour_string in dates_to_process:

        for system, records in create_billing_metrics(date_hour_string).items():
            if system not in result:
                result[system] = records
            else:
                result[system].extend(records)

    for system, records in result.items():

        # FIXME: implement a way to collect dispenser data
        if system == "dispenser":
            continue

        with open(outputs.get(os.environ.get(SYSTEM_OUTPUT_MAPPING[system])), "w") as write_file:
            json.dump(result[system], write_file)
