
import yt.wrapper
import datetime
import requests
import urllib3
import time
import math
import uuid
from copy import deepcopy
import pandas as pd
import os
from collections import Counter
import json

from requests.adapters import HTTPAdapter


METRIC_SCHEMA_MAPPING = {}
CPU_UNIT, MEMORY_UNIT, STORAGE_UNIT, IO_UNIT, GPU_UNIT = "core*hour", "gigabyte*hour", "gigabyte*hour", \
                                                         "megabyte*hour", "gpu*hour"
GB_DIVISOR, MB_DIVISOR = 1024*1024*1024, 1024*1024

VALID_DISPENSER_CLOUDS = {"yp", "gencfg", "qloud"}
METRIC_SCHEMA_MAPPING["gencfg"] = {
    "cpu": {"schema": "gencfg.cpu.quota", "unit": CPU_UNIT},
    "memory": {"schema": "gencfg.memory.quota", "unit": MEMORY_UNIT, "divisor": GB_DIVISOR},
    "hdd_storage": {"schema": "gencfg.hdd_storage.quota", "unit": STORAGE_UNIT, "divisor": GB_DIVISOR},
    "ssd_storage": {"schema": "gencfg.ssd_storage.quota", "unit": STORAGE_UNIT, "divisor": GB_DIVISOR}
}
METRIC_SCHEMA_MAPPING["qloud"] = {
    "cpu": {"schema": "qloud.cpu.quota", "unit": CPU_UNIT},
    "memory": {"schema": "qloud.memory.quota", "unit": MEMORY_UNIT, "divisor": GB_DIVISOR},
}
METRIC_SCHEMA_MAPPING["yp"] = {
    "vcpu": {"schema": "yp.vcpu.quota", "unit": "millivcpu*hour"},
    "memory": {"schema": "yp.memory.quota", "unit": MEMORY_UNIT, "divisor": GB_DIVISOR},
    "hdd_storage": {"schema": "yp.hdd_storage.quota", "unit": STORAGE_UNIT, "divisor": GB_DIVISOR},
    "ssd_storage": {"schema": "yp.ssd_storage.quota", "unit": STORAGE_UNIT, "divisor": GB_DIVISOR}
}
METRIC_SCHEMA_MAPPING["yp_gpu"] = {
    "gpu_geforce_1080ti": {"schema": "yp.gpu_geforce_1080ti.quota", "unit": GPU_UNIT},
    "gpu_tesla_k40": {"schema": "yp.gpu_tesla_k40.quota", "unit": GPU_UNIT},
    "gpu_tesla_m40": {"schema": "yp.gpu_tesla_m40.quota", "unit": GPU_UNIT},
    "gpu_tesla_v100": {"schema": "yp.gpu_tesla_v100.quota", "unit": GPU_UNIT},
    "gpu_tesla_v100_nvlink": {"schema": "yp.gpu_tesla_v100_nvlink.quota", "unit": GPU_UNIT}
}


GPU_SEGMENTS = {"gpu-default", "gpu-dev"}
BASE_SEARCH_SEGMENTS = {"base-search", "base-search-cohabitation", "yt_arnold_colocation"}


EXCLUDED_ABC = ["1172", "470", "1979"]


def get_hour_end():
    return (datetime.datetime.now() - datetime.timedelta(hours=1)).replace(minute=59, second=59,
                                                                           microsecond=0).timestamp()


class PrepareBillingData:

    blacklist_abc = {"abc:service:0"}

    system_name = None

    labels_config = {
        "qloud": {
            "has_labels": True, "base_path": "//home/data_com/cubes/qloud/qloud_accounts"
        },
        "gencfg": {
            "has_labels": False, "base_path": "//home/data_com/cubes/gencfg/groups_data"
        },
        "yp": {
            "has_labels": True, "base_path": "//home/data_com/cubes/yp/yp_accounts"
        },
        "yp_gpu": {
            "has_labels": True, "base_path": "//home/data_com/cubes/yp/yp_accounts"
        },
        "dispenser": {
            "has_labels": False, "base_path": None
        }
    }

    timestamp_table_path = "//home/runtimecloud/billing_logbroker_{0}_counter/last_launch"

    def __init__(self, proxy):
        self.proxy = proxy
        self.excluded_abcs = self.get_excluded_abc_ids()

    def read_data(self, table=None):
        """
        :param proxy:
        :return: array of rows
        """

    def get_last_timestamp(self):
        yt.wrapper.config.set_proxy(self.proxy)
        return max([i["last_launch_time"] for i in
                    yt.wrapper.read_table(self.timestamp_table_path.format(self.system_name))])

    def get_table_name(self, table=None, table_timestamp=None):

        if table:
            return table

        base_path = self.labels_config[self.system_name]["base_path"]

        if table_timestamp:
            table_name = "/".join([base_path, datetime.datetime.fromtimestamp(table_timestamp).strftime("%Y-%m-%d")])
        else:
            table_name = "/".join([base_path, datetime.datetime.today().strftime("%Y-%m-%d")])

        if (yt.wrapper.exists(table_name) and not self.check_table_with_data(table_name)) \
                or not yt.wrapper.exists(table_name):
            table_name = "/".join([base_path,
                                   (datetime.datetime.today() - datetime.timedelta(days=1)).strftime("%Y-%m-%d")])

            if (yt.wrapper.exists(table_name) and not self.check_table_with_data(table_name)) \
                or not yt.wrapper.exists(table_name):
                table_name = "/".join([base_path,
                                       (datetime.datetime.today() - datetime.timedelta(days=2)).strftime(
                                           "%Y-%m-%d")])
                if (yt.wrapper.exists(table_name) and not self.check_table_with_data(table_name)) \
                        or not yt.wrapper.exists(table_name):
                    table_name = "/".join([base_path,
                                       (datetime.datetime.today() - datetime.timedelta(days=3)).strftime(
                                           "%Y-%m-%d")])
                    if not self.check_table_with_data(table_name):
                        raise Exception("Yesterday table {0} is empty".format(table_name))

        return table_name

    def transform_quota_to_metrics(self, row, billing_datetime=None):
        metrics = []

        if not billing_datetime:
            current_hour_end = int(get_hour_end())
        else:
            current_hour_end = billing_datetime
        system_name = self.system_name
        try:
            abc_id = int(row["account_id"].split(":")[-1])
        except ValueError:
            return

        if str(abc_id) in self.excluded_abcs:
            return

        try:
            metric_base = {
                "source_id": "localhost",
                "version": "1",
                "source_wt": current_hour_end,
                "usage": {
                    "start": current_hour_end,
                    "finish": current_hour_end,
                },
                "abc_id": abc_id
            }
        except:
            return

        if system_name == "yp":
            if row["segment"] in GPU_SEGMENTS:
                system_name = "yp_gpu"

        if self.labels_config[system_name]["has_labels"]:
            metric_base["labels"] = {"geo": row["geo"], "segment": row["segment"]}

        metric_schema_mapping = METRIC_SCHEMA_MAPPING[system_name]

        try:
            for sku, schema in metric_schema_mapping.items():
                if row[sku] > 0:

                    if schema.get("divisor"):
                        quantity = row[sku] / schema["divisor"]
                    else:
                        quantity = row[sku]

                    metric = deepcopy(metric_base)
                    metric["id"] = uuid.uuid4().hex
                    metric["schema"] = schema["schema"]
                    metric["usage"]["unit"] = schema["unit"]
                    metric["usage"]["quantity"] = int(math.ceil(quantity))
                    metric["tags"] = {}
                    metrics.append(metric)
        except:
            raise

        return metrics

    def prepare_data(self, table_name=None, billing_datetime=None):
        data = self.read_data(table_name)
        metric_message = []

        for record in data:
            metrics = self.transform_quota_to_metrics(record, billing_datetime=billing_datetime)

            if metrics:
                metric_message.extend(metrics)

        return metric_message

    def send_nv_context(self, context):
        outputs = context.get_outputs()
        previous_hour_end = int(get_hour_end() + 1)
        metrics_filename, timestamp_filename = os.environ.get("OUTPUT_FILE"), os.environ.get("METRICS_FILE")

        last_successfull_timestamp = self.get_last_timestamp()

        data = []

        while last_successfull_timestamp < previous_hour_end:
            last_successfull_timestamp += 3600
            table_name = self.get_table_name(table_timestamp=last_successfull_timestamp)

            data.extend(self.prepare_data(table_name=table_name, billing_datetime=last_successfull_timestamp - 1))

        with open(outputs.get(metrics_filename), "w") as write_file:
            json.dump(data, write_file)
        with open(outputs.get(timestamp_filename), "w") as successful_launch:
            json.dump([
                {
                    "last_launch_time": previous_hour_end
                }
            ], successful_launch)

    @staticmethod
    def get_excluded_abc_ids():

        id_to_exclude = set()

        for abc_id in EXCLUDED_ABC:
            with_childer = "https://abc-back.yandex-team.ru/api/v4/services/?parent__with_descendants={0}".format(
                abc_id
            )
            _headers = {"Authorization": f"OAuth {os.environ.get('ABC_TOKEN')}"}
            result = requests.get(with_childer, headers=_headers).json()["results"]

            for record in result:
                id_to_exclude.add(str(record["id"]))
            id_to_exclude.add(abc_id)

        # FIXME: костыль для dummy сервиса в Qloud
        id_to_exclude.add("0")

        return id_to_exclude

    @staticmethod
    def check_table_with_data(tablename):
        return True if len(list(yt.wrapper.read_table(yt.wrapper.TablePath(tablename, start_index=1,
                                                                           end_index=2)))) > 0 else False


class PrepareBillingQloud(PrepareBillingData):

    system_name = "qloud"

    def read_data(self, table_name=None):
        yt.wrapper.config.set_proxy(self.proxy)

        table_name = self.get_table_name(table_name)
        # c точки зрения биллинга qloud и qloud-ext равнозначны, поэтому суммируем по колонке quota_type
        frame = pd.DataFrame([i for i in yt.wrapper.read_table(table_name)
                              if i["account_id"] not in PrepareBillingData.blacklist_abc]).groupby(["account_id",
                                                                                                    "geo", "segment"])[
            "cpu", "memory"].apply(lambda x: x.astype(int).sum())
        frame.reset_index(inplace=True)

        return frame.to_dict('records')


class PrepareBillingGencfg(PrepareBillingData):

    system_name = "gencfg"

    def read_data(self, table_name=None):
        yt.wrapper.config.set_proxy(self.proxy)
        table_name = self.get_table_name(table_name)

        data = []

        for record in yt.wrapper.read_table(table_name):
            if record["is_rtc_group"] is True and record["account_id"] != "":
                data.append(record)

        frame = pd.DataFrame(data).groupby(["account_id"])["cpu", "memory", "hdd_storage", "ssd_storage"].apply(lambda x : x.astype(int).sum())
        frame.reset_index(inplace=True)

        return frame.to_dict("records")


class PrepareBillingYP(PrepareBillingData):

    system_name = "yp"

    def read_data(self, table_name=None):
        yt.wrapper.config.set_proxy(self.proxy)
        table_name = self.get_table_name(table_name)

        data = []

        for record in yt.wrapper.read_table(table_name):

            if record["segment"] in BASE_SEARCH_SEGMENTS or record["geo"] == "sas-test":
                continue

            if record["quota_type"] == "abcd_quota":
                record["hdd_io"] = record.pop("hdd_bandwidth")
                record["ssd_io"] = record.pop("ssd_bandwidth")
                data.append(record)

        return data


class PrepareBillingOrders(PrepareBillingData):

    system_name = "dispenser"

    @staticmethod
    def get_dispenser_orders():
        adapter = HTTPAdapter(max_retries=5)
        http = requests.Session()
        http.mount("https://", adapter)
        result = []

        url = "https://dispenser.yandex-team.ru/common/api/v1/quota-requests/?status=NEW&status=" \
              "READY_FOR_REVIEW&status=APPROVED&status=CONFIRMED&service=yp&service=qloud&service=gencfg&" \
              "pagination=true&page={0}"
        current_page = 1

        while True:
            try:
                response = http.get(
                        url.format(str(current_page)),
                        verify=False,
                        headers={'Authorization': 'OAuth {}'.format(os.environ.get("DISPENSER_TOKEN"))},
                        timeout=1800
                    )
                pages_number = response.json()["totalPages"]
                result.extend(response.json()["result"])
                break
            except (urllib3.exceptions.ProtocolError, requests.exceptions.ChunkedEncodingError):
                time.sleep(30)

        for page_number in range(2, pages_number+1):
            current_page += 1
            try:
                response = http.get(
                    url.format(str(current_page)),
                    verify=False,
                    headers={'Authorization': 'OAuth {}'.format(os.environ.get("DISPENSER_TOKEN"))},
                    timeout=1800
                )
                result.extend(response.json()["result"])
            except (urllib3.exceptions.ProtocolError, requests.exceptions.ChunkedEncodingError):
                time.sleep(30)

        return result

    @staticmethod
    def _parse_resource(amount, resource_name, provider):
        value = amount["value"]

        unit = amount["unit"]
        if unit == "BYTE":
            if resource_name in {"memory", "hdd_storage", "ssd_storage"}:
                value /= 1024. ** 3
            else:
                assert False
        elif unit == "PERMILLE_CORES" and provider != "yp":
            value /= 1000
        elif unit == "PERMILLE" and resource_name == "gpu_tesla_v100":
            value /= 1000

        return value

    def _prepare_dispenser_resourses(self):

        orders_data = self.get_dispenser_orders()
        order_abc_mapping = {}

        for order in orders_data:
            quota_remaining = self._parse_changes(order["changes"])

            if len(quota_remaining) > 0:
                abc_id = ":".join(["abc", "service", str(order["project"]["abcServiceId"])])

                if abc_id not in order_abc_mapping:
                    order_abc_mapping[abc_id] = {}

                for provider, quota in quota_remaining.items():
                    if provider not in order_abc_mapping[abc_id]:
                        order_abc_mapping[abc_id][provider] = quota
                    else:
                        order_abc_mapping[abc_id][provider] += quota

        return order_abc_mapping

    def _parse_changes(self, changes):
        resources_remaining = {}

        names_map = {
            "ram": "memory",
            "cpu_segmented": "cpu",
            "ram_segmented": "memory",
            "hdd_segmented": "hdd_storage",
            "ssd_segmented": "ssd_storage",
            "hdd": "hdd_storage",
            "ssd": "ssd_storage",
            "gpu_segmented": "gpu_tesla_v100",
            "gpu": "gpu_tesla_v100"
        }

        for data in changes:

            if data["service"]["key"] not in VALID_DISPENSER_CLOUDS:
                continue

            resource_name = data["resource"]["key"]

            if resource_name in {"io_hdd", "io_ssd"}:
                continue

            resource_name = names_map.get(resource_name, resource_name)

            if data["service"]["key"] == "yp" and resource_name == "cpu":
                resource_name = "vcpu"

            value_ready = self._parse_resource(data["amountReady"], resource_name, data["service"]["key"])
            value_allocated = self._parse_resource(data["amountAllocated"], resource_name, data["service"]["key"])

            if value_ready - value_allocated <= 0:
                continue

            value_remaining = value_ready - value_allocated

            if data["service"]["key"] not in resources_remaining:
                resources_remaining[data["service"]["key"]] = Counter()

            resources_remaining[data["service"]["key"]].update({resource_name: value_remaining})

        return resources_remaining

    def prepare_data(self, table_name=None, billing_datetime=None):
        metrics = []
        current_hour_end = int(get_hour_end())

        data = self._prepare_dispenser_resourses()

        for abc_service, quota in data.items():

            abc_id = int(abc_service.replace("abc:service:", ""))

            if str(abc_id) in self.excluded_abcs:
                continue

            metric_base = {
                "source_id": "localhost",
                "version": "1",
                "source_wt": current_hour_end,
                "usage": {
                    "start": current_hour_end,
                    "finish": current_hour_end,
                },
                "abc_id": abc_id
            }

            for provider, sku in quota.items():

                for sku_name, sku_value in sku.items():

                    if "gpu" in sku_name:
                        schema = METRIC_SCHEMA_MAPPING["yp_gpu"][sku_name]
                    else:
                        try:
                            schema = METRIC_SCHEMA_MAPPING[provider][sku_name]
                        except KeyError:
                            continue

                    metric = deepcopy(metric_base)
                    metric["id"] = uuid.uuid4().hex
                    metric["schema"] = schema["schema"].replace("quota", "order")
                    metric["usage"]["unit"] = schema["unit"]
                    metric["usage"]["quantity"] = int(math.ceil(sku_value))
                    metric["tags"] = {}
                    metrics.append(metric)

        return metrics

    def send_nv_context(self, context):
        data = self.prepare_data()
        outputs = context.get_outputs()

        output_data_mapping = {
            "gencfg": {"output": os.environ.get("OUTPUT_GENCFG"), "data": []},
            "yp": {"output": os.environ.get("OUTPUT_YP"), "data": []},
            "qloud": {"output": os.environ.get("OUTPUT_QLOUD"), "data": []}
        }

        for record in data:
            output_data_mapping[record["schema"].split(".")[0]]["data"].append(record)

        for cloud, data in output_data_mapping.items():
            with open(outputs.get(data["output"]), "w") as write_file:
                json.dump(data["data"], write_file)


RTC_SYSTEM_OBJECT_MAPPING = {
    "gencfg": PrepareBillingGencfg,
    "yp": PrepareBillingYP,
    "qloud": PrepareBillingQloud,
    "dispenser": PrepareBillingOrders
}
