# coding: utf-8

import click

import yt.yson
from yp.client import YpClient
from infra.ya_salt.proto import ya_salt_pb2
from walle_api import WalleClient

import infra.rtc.sbin.hostman_status.utils as utils


YP_IVA_ADDRESS = "iva.yp.yandex.net:8090"
YP_MAN_ADDRESS = "man.yp.yandex.net:8090"
YP_MAN_PRE_ADDRESS = "man-pre.yp.yandex.net:8090"
YP_MYT_ADDRESS = "myt.yp.yandex.net:8090"
YP_SAS_ADDRESS = "sas.yp.yandex.net:8090"
YP_SAS_TEST_ADDRESS = "sas-test.yp.yandex.net:8090"
YP_VLA_ADDRESS = "vla.yp.yandex.net:8090"

YP_ADDRESSES = (
    YP_SAS_TEST_ADDRESS,
    YP_MAN_PRE_ADDRESS,
    YP_IVA_ADDRESS,
    YP_MYT_ADDRESS,
    YP_MAN_ADDRESS,
    YP_VLA_ADDRESS,
    YP_SAS_ADDRESS
)

YP_ADDRESS_MAP = {}


def _get_cluster_from_address(address):
    cluster_name = address.split(".", 1)[0]
    return cluster_name.replace("-", "_")


def _fill_address_map():
    global YP_ADDRESSES
    for address in YP_ADDRESSES:
        YP_ADDRESS_MAP[_get_cluster_from_address(address)] = address

_fill_address_map()


def create_yp_client(address):
    return YpClient(address, config=dict(token=utils.get_oauth_token()))


def get_hostman_status(yp_client, hostname):
    response = yp_client.get_object(
        "node",
        hostname,
        selectors=["/meta/id", "/status/host_manager"]
    )
    node_id, host_manager = response
    hostman_status = ya_salt_pb2.HostmanStatus()
    hostman_status.MergeFromString(yt.yson.get_bytes(host_manager["value"]))
    return hostman_status


def get_host_cluster(walle_client, hostname):
    result = walle_client.get_hosts([hostname], fields=["location.short_datacenter_name", "tags"])
    node_spec = (result["result"] or [None])[0]
    if not node_spec:
        raise click.ClickException("No node {} exists".format(hostname))
    for tag in node_spec.get("tags", []):
        if tag.startswith("rtc.ypmaster-"):
            return tag[13:]
    datacenter = node_spec.get("location", {}).get("short_datacenter_name")
    if not datacenter:
        raise click.ClickException("No DC for {}".format(hostname))
    return datacenter


@click.command()
@click.argument('hostname')
def cli(hostname):
    """
    HostManager status.
    """
    utils.read_token()

    walle_client = WalleClient()
    cluster_name = get_host_cluster(walle_client, hostname)

    with create_yp_client(YP_ADDRESS_MAP[cluster_name]) as yp_client:
        print(get_hostman_status(yp_client, hostname))
