#!/usr/bin/env python
# -*- coding: utf-8 -*-

from __future__ import unicode_literals
import os
import sys
import argparse
import requests
import json
import logging
import json
import time
from pprint import pprint
from collections import defaultdict
from jinja2 import Environment, FileSystemLoader
from startrek_client import Startrek
from walle_api import WalleClient
# import itertools
# import pdb

PROG_NAME = "rtc_maintenance"
PROG_VER = '0.1'

ABC_API = "https://abc-back.yandex-team.ru/api/v3"
BOT_API = "https://bot.yandex-team.ru/api"
STAFF_API = "https://staff-api.yandex-team.ru/v3"
WALLE_API = "https://api.wall-e.yandex-team.ru/v1"
VAULT_NAME_OAUTH = "robot-rtc-autoadmin"
# StartTrek component: rtc_maintenance
ST_COMPONENT_ID = '51105'


class Order():
    """
    state:
        - new
        - inwork
    type_work:
        - host_add
        - host_rm
    """
    scenario_types = {'host_add': 'wait', 'host_poweroff': 'noop', 'host_rm': 'noop'}

    def __init__(self, scenario_id=None):
        self._state = None
        self._type_work = None
        self.scenario_ticket_key = None
        self._scenario_hosts = []
        self.scenario_autostart = 'false'
        self.scenario_issuer = []
        self.scenario_reason = ''
        self.scenario_type = 'noop'
        self.scenario_schedule_type = 'all'
        self._scenario_status = None
        self._scenario_labels = {'type_work': self.type_work, 'source': PROG_NAME, 'source_ver':PROG_VER, 'state': self.state}
        self._scenario_script_args = {'responsible':[]}
        self._scenario_id = scenario_id

    def __repr__(self):
        data = dict(self.__dict__)
        for i in ['hosts_info', 'hosts_by_virtual_group', 'invs_by_dc', 'ok_owners', 'cmd_for_duty']:
            data.pop(i)
        return json.dumps(data, indent=4, sort_keys=True)

    @property
    def state(self):
        return self._state

    @state.setter
    def state(self, value):
        log.debug("Changing Order state: {}".format(value))
        self._state = value
        new_labels = dict(self.scenario_labels)
        new_labels.update({'state':value})
        self.scenario_labels = new_labels

    @property
    def type_work(self):
        return self._type_work

    @type_work.setter
    def type_work(self, value):
        self._type_work = value
        new_labels = dict(self.scenario_labels)
        new_labels.update({'type_work':value})
        self.scenario_labels = new_labels

    @property
    def scenario_id(self):
        return self._scenario_id

    @scenario_id.setter
    def scenario_id(self, scenario_id):
        log.info("Getting scenario: {}".format(scenario_id))
        scenario = self.get_scenario(scenario_id)
        log.debug(scenario)
        self._scenario_id = scenario['scenario_id']
        self._scenario_name = scenario['name']
        # TODO scenario.get('autostart', 'false')
        self.scenario_autostart = scenario.get('autostart', 'false')
        # TODO scenario.get('reason', '')
        self.scenario_reason = scenario.get('reason', '')
        self.scenario_type = scenario['scenario_type']
        self.scenario_schedule_type = 'all'
        self._scenario_status = scenario['status']
        self.scenario_issuer = scenario['issuer']
        self._scenario_labels = scenario.get('labels', {})
        self._scenario_script_args = scenario.get('script_args', {})
        self.scenario_ticket_key = scenario['ticket_key']
        self._state = self.scenario_labels.get('state')
        self._type_work = self.scenario_labels.get('type_work')
        self.scenario_hosts = [i['inv'] for i in scenario['hosts']]

    @property
    def scenario_name(self):
        # TODO fix name
        self._scenario_name = str(time.time())
        return self._scenario_name

    @property
    def scenario_hosts(self):
        return self._scenario_hosts

    @scenario_hosts.setter
    def scenario_hosts(self, value):
        self._scenario_hosts = [int(i) for i in value]
        self.hosts_info, self.hosts_by_virtual_group, self.invs_by_dc, self.ok_owners = get_hosts_info(self.scenario_hosts)
        self.cmd_for_duty = self.get_cmd_for_duty(type_work=self.type_work, ticket_key=self.scenario_ticket_key,
                                                    hosts_info=self.hosts_info, hosts_by_virtual_group=self.hosts_by_virtual_group,
                                                    invs_by_dc=self.invs_by_dc, scenario_script_args=self.scenario_script_args, scenario_id=self.scenario_id
        )

    @property
    def scenario_labels(self):
        return self._scenario_labels

    @scenario_labels.setter
    def scenario_labels(self, value):
        new_labels = dict(self.scenario_labels)
        new_labels.update(value)
        data = {"labels": new_labels}
        self._scenario_labels = new_labels
        if self.scenario_id:
            self.scenario_modify(json.dumps(data))

    @property
    def scenario_script_args(self):
        return self._scenario_script_args

    @scenario_script_args.setter
    def scenario_script_args(self, value):
        new_args = dict(self.scenario_script_args)
        new_args.update(value)
        data = {"scenario_script_args": new_args}
        self._scenario_script_args = new_args
        if self.scenario_id:
            self.scenario_modify(json.dumps(data))

    @property
    def scenario_status(self):
        return self._scenario_status

    def get_scenario(self, scenario_id):
        return walle_client.get_scenario(scenario_id)

    def create_scenario(self):
        log.info("Creating scenario")
        try:
            self.state = 'new'
            # TODO reason don't work
            #scenario = walle_client.add_scenario(name=self.scenario_name, scenario_type=self.scenario_types[self.type_work], ticket_key=self.scenario_ticket_key, hosts=self.scenario_hosts, labels=self.scenario_labels, script_args=self.scenario_script_args, schedule_type=self.scenario_schedule_type, reason=self.scenario_reason)
            scenario = walle_client.add_scenario(name=self.scenario_name, scenario_type=self.scenario_types[self.type_work],
                                                ticket_key=self.scenario_ticket_key, hosts=self.scenario_hosts, labels=self.scenario_labels,
                                                script_args=self.scenario_script_args, schedule_type=self.scenario_schedule_type
            )
            log.info("Wall-e scenario ID {} created".format(scenario['scenario_id']))
            return scenario['scenario_id']
        except Exception as e:
            log.critical(e)
            return False

    def create(self, dry_run):
        log.info("Creating order")
        if dry_run:
            return True
        scenario_id = self.create_scenario()

    def start_scenario(self):
        walle_client.start_scenario(self.scenario_id)

    def run(self, dry_run):
        issue = None
        if self.scenario_ticket_key:
            try:
                log.debug("Finding ST key: {}".format(self.scenario_ticket_key))
                issue = st_client.issues[self.scenario_ticket_key]
            except Exception as e:
                log.critical(e)
                log.info('ST Ticket not found: {}'.format(self.scenario_ticket_key))
        if self.state == 'new' and self.scenario_status == 'created':
            log.debug("Wall-e scenario is new")
            if issue and not dry_run:
                log.info("Updating ST ticket {}".format(self.scenario_ticket_key))
                transition = issue.transitions['open']
                transition.execute(comment=self.cmd_for_duty, pendingReplyFrom=self.ok_owners)
                self.state = 'inwork'
            else:
                log.info("\n{}".format(self.cmd_for_duty))
        elif self.state == 'inwork' and self.scenario_status == 'finished':
            log.debug("Wall-e scenario is finished")
            # TODO
            # add comment to ST
            #issue = st_client.issues[self.scenario_ticket_key]
            #issue.comments.create(text="**Wall-e scenario ID**: {} !!finished!!".format(self.scenario_id))
            #self.state = 'closed'
        elif issue and issue.status.key == 'new' and 'Форма_{5c3c5d9a830fa5004786401e}' in issue.tags:
            if not dry_run:
                log.info("Updating ST ticket {}".format(self.scenario_ticket_key))
                transition = issue.transitions['open']
                transition.execute(comment=self.cmd_for_duty, pendingReplyFrom=self.ok_owners)
        else:
            log.info("Skip order. Nothing to do")
            print(self.cmd_for_duty)

    def scenario_modify(self, data):
        if not self.scenario_id:
            return True
        hand = '/scenarios/{}'.format(self.scenario_id)
        log.info("Patching scenario: {}".format(self.scenario_id))
        resps = session.patch('{}{}'.format(WALLE_API, hand), data=data)
        if resps.ok:
            log.info("Patched {}: {}".format(self.scenario_id, resps.json()))
            return True
        else:
            log.critical('{}\n{}'.format(resps.url, resps.text))
            return False

    def render_jinja(self, tpl_name, **kwargs):
        tpl_name = '{}.jinja'.format(tpl_name)
        template_path = os.path.dirname(os.path.abspath(__file__))
        env = Environment(loader=FileSystemLoader(template_path))
        log.info("Loading Jinja template: {}".format(tpl_name))
        template = env.get_template(tpl_name)
        return template.render(**kwargs)

    def get_cmd_for_duty(self, fmt='wiki', **kwargs):
        if fmt == 'wiki':
            result = self.render_jinja(self.type_work, **kwargs)
        return result

class Host():
    OWNERS_BY_W_TAG = {
        "yabs": ["bahbka"],
        "yp_masters": ["slonn", "stunder"],
        "yp_dev": ["golomolzin", "slonn", "frolstas"],
        "market": ["maxk"],
        "qloud": ["danichk", "earthling"],
        "yt": ["stunder"],
        "rtc": ["sivanichkin", "staggot"]
    }
    RTC_OWNERS_FOR_OPERATIONS = [i for i in OWNERS_BY_W_TAG.keys() if i != "market"]

    def __init__(self, inv):
        self.name = None
        self.abc_prj = None
        self.abc_prj_id = None
        self._dc = None
        self.switch = None
        self.w_prj = None
        self.w_prj_tags = []
        self.w_prj_owners = []
        self.virtual_group = "other"
        self.owners = []
        self.prj = None
        self.inv = inv

    def __repr__(self):
        return json.dumps(self.__dict__, indent=4, sort_keys=True)

    @property
    def inv(self):
        return self._inv

    @inv.setter
    def inv(self, value):
        self._inv = value
        self.name, self.abc_prj, self.dc = self._get_bot_host_info(self.inv)
        if self.name and self.abc_prj and self.dc:
            log.info("Host info from BOT: fqdn={}, abc_prj={}, dc={}".format(self.name, self.abc_prj, self.dc))
            data = walle_client.get_host(self.inv, fields=['project', 'location.switch'])
            self.w_prj, self.switch = data['project'], data['location']['switch']
            log.info("Host info from Walle: w_prj={}, switch={}".format(self.w_prj, self.switch))
            if self.w_prj:
                data = walle_client.get_project(self.w_prj, fields=["tags", "owners"])
                self.w_prj_tags, self.w_prj_owners = data.get('tags', []), data.get('owners', [])
                log.info("Walle project info from Walle: w_prj_tags={}, w_prj_owners={}".format(self.w_prj_tags, self.w_prj_owners))
                for tag, owners in self.OWNERS_BY_W_TAG.items():
                    if tag in self.w_prj_tags:
                        self.prj = tag
                        self.owners = owners
                        if tag in self.RTC_OWNERS_FOR_OPERATIONS:
                            self.virtual_group = "rtc"
                        break
            if not self.owners:
                self.owners = self._get_owners_by_abc(self.abc_prj)
            if not self.owners:
                for i in self.w_prj_owners:
                    if i.startswith('@'):
                        self.owners += self._resolve_staff_grp_to_logins(i)
                    else:
                        self.owners.append(i)
                self.prj = self.w_prj
            self.owners = set(self.owners)

    @property
    def dc(self):
        return self._dc

    @dc.setter
    def dc(self, val):
        CITY_TO_DC = {
            "VLADIMIR": "vla",
            "MANTSALA": "man",
        }
        if val:
            self._dc = CITY_TO_DC.get(val, val.lower())

    def _get_bot_host_info(self, inv):
        fqdn, abc_prj, dc = None, None, None
        hand = "/osinfo.php"
        log.info("Geting info from BOT for inv: {}".format(inv))
        resps = session.get('{}{}'.format(BOT_API, hand), params={"inv": inv, "output": "XXCSI_FQDN|GROUP_OWNED|loc_object"})
        if resps.ok:
            resp = resps.json()
            if resp.get("os", {}) and not resp.get('msg'):
                fqdn, abc_prj, dc = resp["os"][0]["XXCSI_FQDN"], resp["os"][0]["GROUP_OWNED"], resp["os"][0]["loc_object"]
        return fqdn, abc_prj, dc

    def _get_owners_by_abc(self, slug):
        self.abc_prj_id = self._resolve_abc_prj_to_id(slug)
        owners = defaultdict(list)
        hand = "/services/members/"
        # 1369 - Maintenance manager
        # 16 - System Administrator
        params = {"service": self.abc_prj_id, "fileds": "person", "role": [16, 1369]}
        log.info("Getting owners from ABC")
        resps = session.get('{}{}'.format(ABC_API, hand), params=params)
        if resps.ok:
            resp = resps.json()["results"]
            for i in resp:
                owners[i["role"]["id"]].append(i["person"]["login"])
        else:
            log.info("Problems with getting owners from ABC: {}".format(resps.text))
        self.prj = self.abc_prj
        if owners[1369]:
            return owners[1369]
        else:
            return owners[16]

    def _resolve_abc_prj_to_id(self, slug):
        hand = "/services/"
        params = {"name": self.abc_prj, "fileds": "id"}
        resps = session.get('{}{}'.format(ABC_API, hand), params=params)
        if resps.ok:
            resp = resps.json()["results"]
            if len(resp) == 1:
                return resp[0]['id']

    def _resolve_staff_grp_to_logins(self, group_name):
        hand = "/groupmembership"
        params = {"_fields": "person.login", "group.url": group_name.strip('@')}
        resps = session.get('{}{}'.format(STAFF_API, hand), params=params)
        if resps.ok:
            data = resps.json()
            return [i["person"]["login"] for i in data["result"]]

def parse_st_description(description):
    questions = {
        "Тип работ": "type_work",
        "Список ID серверов": "hosts",
        "Требуется очистка": "need_clean",
        "Очередь": "line",
        "Стойка": "rack",
        "Изменятся ли IP адреса хостов": "ip_change",
        "Оценка по времени даунтайма": "date",
        "Wall-e проект": "walle_prj",
        "Комментарий": "comment",
        "Кто сдает хосты": "responsible"
    }
    type_works = {
        "ввод хостов в wall-e проект": "host_add",
        "перемещение хостов между wall-e проектами": "host_change_prj",
        "прошивка компонет сервера": "host_firmware",
        "выключение/переезд стойки": "host_poweroff",
        "вЫвод хостов из wall-e": "host_rm"
    }
    result = {}
    question_en = None
    for line in description.split('\n'):
        # line is empty
        if not line.strip():
            question_en = None
            continue
        if line[-1:] == ':':
            question = line[:-1].strip().replace('<{', '')
            question_en = questions.get(question, None)
            if question_en == 'hosts':
                result[question_en] = []
            else:
                result[question_en] = ""
        else:
            answer = line.strip().replace('}>', '')
            if answer and question_en:
                if isinstance(result[question_en], list):
                    result[question_en].append(int(answer))
                else:
                    if question_en == 'type_work':
                        result[question_en] = type_works[answer]
                    else:
                        result[question_en] = answer
    return result

def get_hosts_info(hosts_list):
    hosts_info = defaultdict(lambda: defaultdict(lambda: {"owners": [], "hosts": []}))
    hosts_by_virtual_group = defaultdict(list)
    invs_by_dc = defaultdict(list)
    responsible_list = []
    for h in hosts_list:
        host = Host(h)
        hosts_by_virtual_group[host.virtual_group].append(host.inv)
        invs_by_dc[host.dc].append(host.inv)
        if not hosts_info[host.virtual_group][host.prj]["owners"]:
            hosts_info[host.virtual_group][host.prj]["owners"] = [",".join([i+"@" if i != "NONE" else i for i in host.owners])]
            responsible_list += [i for i in host.owners if i != "NONE"]
        hosts_info[host.virtual_group][host.prj]["hosts"].append(host)
    return hosts_info, hosts_by_virtual_group, invs_by_dc, responsible_list

def get_scenarios_by_filter(**kwargs):
    log.info("Getting Wall-e scenarios by filter")
    orders_list = []
    for scenario in walle_client.get_scenarios_by_labels(fields=['scenario_id'], **kwargs).get('result', []):
        order = Order()
        order.scenario_id = scenario['scenario_id']
        yield order

if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="Maintenance in RTC", prog=PROG_NAME, add_help=True)
    parser.add_argument('-v', action='version', version=PROG_VER)
    parser.add_argument('-l', default="error", dest='log_level', choices=['error', 'info', 'debug'])
    parser.add_argument('--oauth', required=False, dest='oauth', help='oauth token')
    parser.add_argument('--dry_run', required=False, action='store_true', dest='dry_run')
    subparsers = parser.add_subparsers(dest='cmd')
    subparsers.required = True

    list_parser = subparsers.add_parser('list', help='показать существующие регламенты')
    list_parser.add_argument('--id', required=False, dest='id', help='Фильтр по Wall-e scenarioID')

    modify_parser = subparsers.add_parser('modify', help='изменить существующий регламент')
    modify_parser.add_argument('--id', required=True, dest='id', help='Wall-e scenarioID')
    modify_parser.add_argument('--data', required=True, dest='data', help="'{\"labels\":{\"process\":\"inprogress\"}}'")

    start_parser = subparsers.add_parser('start', help='запустить Wall-e scenarioID')
    start_parser.add_argument('--id', required=True, dest='id', help='Wall-e scenario ID')

    create_parser = subparsers.add_parser('create', help='создать сценарий')
    create_parser.add_argument('--id', required=True, dest='id', help='RUNTIMECLOUD-13508')
    create_parser.add_argument('--type_work', required=True, dest='type_work', choices=['host_add', 'host_poweroff', 'host_rm'], help='тип работ')
    create_parser.add_argument('--hosts', required=True, dest='hosts', help='900313317, 000000000, 950284, 100052033, 220515')
    create_parser.add_argument('--walle_prj', required=False, dest='walle_prj', help='rtc-mtn')
    create_parser.add_argument('--responsible', required=False, dest='responsible', help='sivanichkin, ')
    create_parser.add_argument('--comment', required=False, dest='comment', default='None', help='test')

    run_parser = subparsers.add_parser('run', help='запустить процесс регламентов')
    run_parser.add_argument('--id', required=False, dest='id', default=None, help='ST key или Walle scenario ID зависит от --src')
    run_parser.add_argument('--src', required=False, dest='src', default='walle', choices=['walle', 'st', 'custom'], help='источник входных данных')
    run_parser.add_argument('--type_work', required=False, dest='type_work', default=None, choices=['host_add', 'host_rm'], help='тип работ')
    run_parser.add_argument('--hosts', required=False, dest='hosts', default=None, help='900313317, 000000000, 950284, 100052033, 220515')
    run_parser.add_argument('--walle_prj', required=False, dest='walle_prj', default=None, help='rtc-mtn')
    run_parser.add_argument('--responsible', required=False, dest='responsible', default=None, help='sivanichkin, ')
    run_parser.add_argument('--comment', required=False, dest='comment', default='None', help='test')

    args = parser.parse_args()
    logging.basicConfig(format='[%(asctime)s]:%(levelname)s %(message)s', level="ERROR")
    log = logging.getLogger(PROG_NAME)
    log.setLevel(getattr(logging, args.log_level.upper()))
    log.debug(args)
    if not args.oauth:
        try:
            with open(os.getenv("HOME")+"/.oauth") as f:
                oauth_token = f.readline().strip('\n')
                args.oauth = oauth_token
        except IOError as e:
            log.error(e)
            log.error("{} {}".format("Get the token from https://oauth.yandex-team.ru/authorize?response_type=token&client_id=<your client_id> and put in", os.getenv("HOME")+"/.oauth"))
            sys.exit(1)
        except Exception as e:
            log.error(e)
            sys.exit(1)
    #global st_client, walle_client, session
    session = requests.Session()
    session.headers["Authorization"] = "OAuth " + args.oauth
    session.headers["Content-Type"] = "application/json;charset=UTF-8"
    st_client = Startrek(useragent=PROG_NAME, token=args.oauth)
    walle_client = WalleClient(access_token=args.oauth)
    if args.cmd == 'list':
        for scenario in get_scenarios_by_filter(scenario_id=args.id, labels={'source':PROG_NAME}):
            print(scenario)
    if args.cmd == 'start':
        for scenario in get_scenarios_by_filter(scenario_id=args.id):
            scenario.start_scenario()
    elif args.cmd == 'modify':
        order = Order()
        order.scenario_id = args.id
        order.scenario_modify(args.data)
    elif args.cmd == 'create':
        hosts_list = [int(i) for i in args.hosts.split(',')]
        order = Order()
        order.type_work = args.type_work
        order.scenario_ticket_key = args.id
        order.scenario_hosts = [int(i) for i in args.hosts.split(',')]
        order.scenario_reason = args.comment
        if args.responsible:
            order.scenario_script_args['responsible'] = args.responsible.split(',')
        if args.walle_prj:
            order.scenario_script_args['target_project_id'] = args.walle_prj
        order.create(dry_run=args.dry_run)
        print(order)
    elif args.cmd == 'run' and args.src == 'walle':
        for order in get_scenarios_by_filter(scenario_id=args.id, labels={'source':PROG_NAME}):
            order.run(dry_run=args.dry_run)
    elif args.cmd == 'run' and args.src == 'st':
        if args.id:
            st_filter = "Key: {}".format(args.id)
        else:
            st_filter = "Components: {} AND Status: New".format(ST_COMPONENT_ID)
        log.debug('Find tickets by ST filter: {}'.format(st_filter))
        for ticket in st_client.issues.find(st_filter):
            log.debug("Key: {}, Summary: {}".format(ticket.key, ticket.summary))
            log.debug("Description:\n{}".format(ticket.description))
            ticket_answers = parse_st_description(ticket.description)
            log.debug(ticket_answers)
            order = Order()
            order.type_work = ticket_answers['type_work']
            order.scenario_hosts = ticket_answers['hosts']
            order.scenario_ticket_key = ticket.key
            if ticket_answers.get('walle_prj'):
                order.scenario_script_args['target_project_id'] = ticket_answers['walle_prj']
            order.run(dry_run=args.dry_run)
    elif args.cmd == 'run' and args.src == 'custom':
        if not args.hosts or not args.type_work:
            parser.error('the following arguments are required: --hosts, --type_work')
        if args.type_work == 'host_add' and not args.walle_prj:
            parser.error('the following arguments are required: --walle_prj for --type_work=host_add')
            sys.exit(1)
        order = Order()
        order.type_work = args.type_work
        order.scenario_hosts = args.hosts.split(',')
        order.scenario_ticket_key = args.id
        if args.walle_prj:
            order.scenario_script_args['target_project_id'] = args.walle_prj
        order.run(dry_run=args.dry_run)
