# coding: utf-8

import collections
import walle_api
import prettytable
import argparse

ProjectHosts = collections.namedtuple("ProjectHosts", ("good", "bad", "unknown"))


def main():
    parser = argparse.ArgumentParser(description="Show Kernel Distribution", add_help=True)
    group = parser.add_mutually_exclusive_group(required=True)
    group.add_argument("-p", dest="project", help="Filter by projects")
    group.add_argument("-t", dest="tags", help="Filter by tags")
    parser.add_argument("--print_fqdn", dest="print_fqdn", choices=["good", "bad"], help="Print fqdns")
    args = parser.parse_args()
    client = walle_api.WalleClient()
    projects = collections.defaultdict(lambda: ProjectHosts([], [], []))
    for host in client.iter_hosts(state="assigned", fields=["name", "health", "project"], tags=args.tags, project=args.project):
        project_hosts = projects[host["project"]]
        name = host["name"]
        need_reboot_kernel = host.get("health", {}).get("check_statuses", {}).get("need_reboot_kernel")
        if need_reboot_kernel == "passed":
            project_hosts.good.append(name)
        elif need_reboot_kernel == "failed":
            project_hosts.bad.append(name)
        else:
            project_hosts.unknown.append(name)

    if args.print_fqdn:
        for project, project_info in projects.iteritems():
            fqdn_list = getattr(project_info, args.print_fqdn)
            print("{}: {}".format(project, len(fqdn_list)))
            print("-"*len(project))
            print(" ".join(fqdn_list))
            print("\n")
    else:
        table = prettytable.PrettyTable()
        table.field_names = ["Project", "Good", "Bad", "Unknown", "Total"]

        total_good, total_bad, total_unknown = 0, 0, 0
        key_func = lambda (project, project_info): (len(project_info.bad), len(project_info.good), project)
        for project, project_info in sorted(projects.iteritems(), key=key_func, reverse=True):
            counter_good, counter_bad, counter_unknown = len(project_info.good), len(project_info.bad), len(project_info.unknown)
            total_good += counter_good
            total_bad += counter_bad
            total_unknown += counter_unknown
            table.add_row([project, counter_good, counter_bad, counter_unknown, counter_good + counter_bad + counter_unknown])
        table.add_row(["Total", total_good, total_bad, total_unknown, total_good + total_bad + total_unknown])

        print table.get_string()


if __name__ == "__main__":
    main()
