package main

import (
	"crypto/x509"
	"encoding/pem"
	"errors"
	"fmt"
	"io/ioutil"
	"log"
	"os/user"
	"path"
	"syscall"

	"golang.org/x/crypto/ssh"
	"golang.org/x/crypto/ssh/terminal"
)

func GetDefaultUserName() string {
	usr, err := user.Current()
	if err != nil {
		log.Fatal(err)
	}
	return usr.Username
}

func GetDefaultPrivateKey() string {
	usr, err := user.Current()
	if err != nil {
		log.Fatal(err)
	}
	return path.Join(usr.HomeDir, ".ssh", "id_rsa")
}

func readPassword() ([]byte, error) {
	fmt.Print("Password: ")
	bytePassword, err := terminal.ReadPassword(int(syscall.Stdin))
	if err != nil {
		return nil, err
	}
	fmt.Println()
	return bytePassword, nil
}

func decodePrivateKey(key []byte) ([]byte, error) {
	block, rest := pem.Decode(key)
	if len(rest) > 0 {
		return nil, errors.New("extra data included in key")
	}

	if x509.IsEncryptedPEMBlock(block) {
		password, err := readPassword()
		if err != nil {
			return nil, fmt.Errorf("password not provided: %v", err)
		}

		der, err := x509.DecryptPEMBlock(block, password)
		if err != nil {
			return nil, fmt.Errorf("decrypt failed: %v", err)
		}
		return pem.EncodeToMemory(&pem.Block{Type: block.Type, Bytes: der}), nil
	}
	return key, nil
}

func ReadPrivateKey(path string) (signer ssh.Signer, err error) {
	key, err := ioutil.ReadFile(path)
	if err != nil {
		return
	}

	key, err = decodePrivateKey(key)
	if err != nil {
		return
	}

	signer, err = ssh.ParsePrivateKey(key)
	if err != nil {
		return
	}

	return
}
