package main

import (
	"encoding/json"
	"flag"
	"io/ioutil"
	"log"
	"strings"
	"sync"

	"golang.org/x/net/context"

	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/clients/netmon"
)

var (
	userName       = flag.String("user", GetDefaultUserName(), "user name")
	workerCount    = flag.Int("workers", 16, "worker count")
	privateKeyPath = flag.String("private-key", GetDefaultPrivateKey(), "path to private key")
	reportPath     = flag.String("report-path", "report.json", "report path")
)

type CheckResult struct {
	HostInfo WalleHostInfo `json:"host_info"`
	BootID   string        `json:"boot_id"`
	Error    string        `json:"error"`
	Success  bool          `json:"success"`
}

type HostExecutor struct {
	Client           *SSHClient
	Hosts            chan WalleHostInfo
	Results          chan CheckResult
	CollectedResults []CheckResult
	WaitGroup        sync.WaitGroup
	Done             chan bool
}

func (executor *HostExecutor) Start(workerCount int) {
	for i := 0; i < workerCount; i++ {
		executor.WaitGroup.Add(1)
		go func() {
			for {
				hostInfo, more := <-executor.Hosts
				if more {
					executor.doWork(hostInfo)
				} else {
					executor.WaitGroup.Done()
					return
				}
			}
		}()
	}

	go func() {
		executor.CollectedResults = make([]CheckResult, 0)
		for {
			result, more := <-executor.Results
			if more {
				executor.CollectedResults = append(executor.CollectedResults, result)
			} else {
				executor.Done <- true
				return
			}
		}
	}()
}

func (executor *HostExecutor) AddHostInfo(hostInfo WalleHostInfo) {
	executor.Hosts <- hostInfo
}

func (executor *HostExecutor) Wait() {
	close(executor.Hosts)
	executor.WaitGroup.Wait()

	close(executor.Results)
	<-executor.Done
}

func (executor *HostExecutor) doWork(hostInfo WalleHostInfo) {
	checkResult := CheckResult{HostInfo: hostInfo}
	checkResult.Success = false
	defer func() {
		executor.Results <- checkResult
	}()

	if hostInfo.Health.CheckStatuses.SSH != "passed" {
		log.Printf("ssh on host %s not working properly: %s", hostInfo.Name, hostInfo.Health.CheckStatuses.SSH)
		return
	}

	sshConn, err := executor.Client.Dial(hostInfo.Name)
	if err != nil {
		checkResult.Error = err.Error()
		log.Printf("unable to establish connection to %s: %v", hostInfo.Name, err)
		return
	}
	defer sshConn.Close()

	bootID, err := sshConn.GetBootID()
	if err != nil {
		checkResult.Error = err.Error()
		log.Printf("unable to execute command on %s: %v", hostInfo.Name, err)
		return
	}

	checkResult.Success = true
	checkResult.BootID = bootID
	log.Printf("boot id for %s: %s", hostInfo.Name, bootID)
}

func main() {
	flag.Parse()

	expression := strings.Join(flag.Args(), " ")
	if len(expression) == 0 {
		expression = "walle_project=test"
	}
	log.Printf(`Checking ssh on hosts from expression "%v" with username "%v"`, expression, *userName)

	signer, err := ReadPrivateKey(*privateKeyPath)
	if err != nil {
		log.Fatalf("unable to load private key: %v", err)
	}

	ctx := context.Background()
	ctx, cancel := context.WithCancel(ctx)
	defer cancel()

	netmonClient, err := netmon.NewNetmonClient()
	if err != nil {
		log.Fatalf("can't create netmon client: %v", err)
	}

	hosts, err := netmonClient.GetHostsInExpression(ctx, expression)
	if err != nil {
		log.Fatalf("unable to get hosts from netmon: %v", err)
	}
	log.Printf("%d hosts found via netmon", len(hosts))

	walleClient := NewWalleClient()
	hostsInfo, err := walleClient.GetHostsInfo(ctx, hosts)
	if err != nil {
		log.Fatalf("unable to get hosts from wall-e: %v", err)
	}
	log.Printf("%d hosts found via wall-e", len(hostsInfo))

	executor := HostExecutor{
		Client:  NewSSHClient(*userName, signer),
		Hosts:   make(chan WalleHostInfo),
		Results: make(chan CheckResult),
		Done:    make(chan bool),
	}
	executor.Start(*workerCount)
	for _, hostInfo := range hostsInfo {
		executor.AddHostInfo(hostInfo)
	}
	executor.Wait()
	log.Printf("%d results found", len(executor.CollectedResults))

	reportBody, err := json.MarshalIndent(executor.CollectedResults, "", "  ")
	if err != nil {
		log.Fatalf("unable to create report: %v", err)
	}

	err = ioutil.WriteFile(*reportPath, reportBody, 0644)
	if err != nil {
		log.Fatalf("unable to write report: %v", err)
	}
}
