package main

import (
	"errors"
	"fmt"
	"net"
	"time"

	"golang.org/x/crypto/ssh"
)

type SSHClient struct {
	Config *ssh.ClientConfig
}

type SSHConnection struct {
	Connection ssh.Conn
	Client     *ssh.Client
}

func NewSSHClient(user string, signer ssh.Signer) (client *SSHClient) {
	client = new(SSHClient)
	client.Config = &ssh.ClientConfig{
		User: user,
		Auth: []ssh.AuthMethod{
			ssh.PublicKeys(signer),
		},
		HostKeyCallback: ssh.InsecureIgnoreHostKey(),
	}
	return
}

func (c *SSHClient) Dial(host string) (conn SSHConnection, err error) {
	address := fmt.Sprintf(`%s:22`, host)
	tcpConn, err := net.DialTimeout("tcp6", address, 5*time.Second)
	if err != nil {
		return
	}

	if err = tcpConn.SetDeadline(time.Now().Add(15 * time.Second)); err != nil {
		return
	}

	sshConn, chans, reqs, err := ssh.NewClientConn(tcpConn, host, c.Config)
	if err != nil {
		_ = tcpConn.Close()
		return
	}

	conn.Connection = sshConn
	conn.Client = ssh.NewClient(conn.Connection, chans, reqs)
	return
}

func (c *SSHConnection) GetBootID() (output string, err error) {
	session, err := c.Client.NewSession()
	if err != nil {
		return
	}
	defer session.Close()

	rawOutput, err := session.Output("cat /proc/sys/kernel/random/boot_id")
	if err != nil {
		return
	}
	if len(rawOutput) == 0 {
		err = errors.New("empty output")
		return
	}

	output = string(rawOutput[:len(rawOutput)-1])
	return
}

func (c *SSHConnection) Close() {
	_ = c.Connection.Close()
}
