package main

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"time"

	"golang.org/x/net/context"
)

type WalleCheckStatuses struct {
	SSH string `json:"ssh"`
}

type WalleHealth struct {
	CheckStatuses WalleCheckStatuses `json:"check_statuses"`
}

type WalleHostInfo struct {
	Name   string      `json:"name"`
	Health WalleHealth `json:"health"`
}

type WalleHostInfoRequest struct {
	Names []string `json:"names"`
}

type WalleHostInfoReply struct {
	Hosts []WalleHostInfo `json:"result"`
	Total int             `json:"total"`
}

type WalleClient struct {
	Client *http.Client
}

func NewWalleClient() (client *WalleClient) {
	client = new(WalleClient)
	client.Client = &http.Client{Timeout: time.Duration(5 * time.Second)}
	return
}

func (c *WalleClient) makeRequest(ctx context.Context, path string, requestBody []byte) (responseBody []byte, err error) {
	url := fmt.Sprintf(`https://api.wall-e.yandex-team.ru%s`, path)
	req, err := http.NewRequest("POST", url, bytes.NewReader(requestBody))
	if err != nil {
		return
	}

	req.Header.Set("Content-Type", "application/json")
	req = req.WithContext(ctx)
	resp, err := c.Client.Do(req)
	if err != nil {
		return
	}

	defer resp.Body.Close()
	responseBody, err = ioutil.ReadAll(resp.Body)
	return
}

func (c *WalleClient) rawGetHostsInfo(ctx context.Context, names []string) (hostsInfo []WalleHostInfo, err error) {
	request := WalleHostInfoRequest{Names: names}
	requestBody, err := json.Marshal(request)
	if err != nil {
		return
	}

	path := fmt.Sprintf(`/v1/get-hosts?fields=name,health.check_statuses&strict=true&limit=%d`, len(names))
	responseBody, err := c.makeRequest(ctx, path, requestBody)
	if err != nil {
		return
	}

	var result WalleHostInfoReply
	err = json.Unmarshal(responseBody, &result)
	if err != nil {
		return
	}

	hostsInfo = result.Hosts
	return
}

func (c *WalleClient) GetHostsInfo(ctx context.Context, names []string) (hostsInfo []WalleHostInfo, err error) {
	var chunk []WalleHostInfo
	chunkSize := 1000
	for i := 0; i < len(names); i += chunkSize {
		end := i + chunkSize

		if end > len(names) {
			end = len(names)
		}

		chunk, err = c.rawGetHostsInfo(ctx, names[i:end])
		if err != nil {
			return
		}

		hostsInfo = append(hostsInfo, chunk...)
	}
	return
}
