package main

import (
	"flag"
	"io/ioutil"
	"os"
	"os/user"
	"path/filepath"
	"strings"

	aLog "a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	wiki "a.yandex-team.ru/locdoc/libs/gowiki"

	zp "go.uber.org/zap"
	"go.uber.org/zap/zapcore"
	"gopkg.in/yaml.v2"
)

var (
	logger = zap.Must(zp.Config{
		Level:            zp.NewAtomicLevelAt(zapcore.DebugLevel),
		Encoding:         "console",
		OutputPaths:      []string{"stdout"},
		ErrorOutputPaths: []string{"stderr"},
		EncoderConfig: zapcore.EncoderConfig{
			MessageKey:     "msg",
			LevelKey:       "level",
			TimeKey:        "ts",
			CallerKey:      "caller",
			EncodeLevel:    zapcore.CapitalLevelEncoder,
			EncodeTime:     zapcore.ISO8601TimeEncoder,
			EncodeDuration: zapcore.StringDurationEncoder,
			EncodeCaller:   zapcore.ShortCallerEncoder,
		},
	})
	wikiCluster = flag.String("wiki-cluster", "/runtime-cloud", "path to your cluster")
	wikiDepth   = flag.Int("wiki-depth", 10, "depth to dive in")
	outDir      = flag.String("out-dir", "out", "where to put result")
)

type TocItem struct {
	Name  string    `yaml:"name"`
	Link  string    `yaml:"href,omitempty"`
	Items []TocItem `yaml:"items,omitempty"`
}

type TocTree struct {
	Title string    `yaml:"title"`
	Items []TocItem `yaml:"items"`
}

func getWikiToken() (string, error) {
	token := os.Getenv("WIKI_TOKEN")
	if token != "" {
		return token, nil
	}

	usr, err := user.Current()
	if err != nil {
		return "", err
	}

	path := filepath.Join(usr.HomeDir, ".st", "token")
	raw, err := ioutil.ReadFile(path)
	if err != nil {
		return "", err
	}

	return strings.TrimSpace(string(raw)), nil
}

func iterateSubPages(pages []*wiki.SubPage, parentTocItem *TocItem, f func(*wiki.Page, *TocItem)) {
	for _, subPage := range pages {
		tocItem := &TocItem{
			Items: make([]TocItem, 0),
		}
		f(subPage.Page, tocItem)
		iterateSubPages(subPage.SubPages, tocItem, f)
		parentTocItem.Items = append(parentTocItem.Items, *tocItem)
	}
}

func checkError(err error) {
	if err != nil {
		logger.Error("request to wiki failed", aLog.Error(err))
		os.Exit(1)
	}
}

func checkWikiError(wikiErr *wiki.WikiError) {
	if wikiErr != nil {
		logger.Error("request to wiki finished with error", aLog.String("DebugMessage", wikiErr.DebugMessage))
		os.Exit(1)
	}
}

func ensureDir(dirName string) error {
	if _, serr := os.Stat(dirName); serr != nil {
		merr := os.MkdirAll(dirName, os.ModePerm)
		if merr != nil {
			return merr
		}
	}
	return nil
}

func main() {
	flag.Parse()

	wikiToken, err := getWikiToken()
	if err != nil {
		logger.Error("can't get token", aLog.Error(err))
		os.Exit(1)
	}

	wikiClient := wiki.NewClient(wiki.Options{
		BaseURL: wiki.ProductionURL,
		Token:   wikiToken,
	})

	trasformer := NewWikiTransformer()

	processPage := func(page *wiki.Page, tocItem *TocItem) {
		logger.Info("processing page", aLog.String("URL", page.URL), aLog.String("Type", page.Type), aLog.String("Title", page.Title))

		tocItem.Name = page.Title

		if page.Type != "P" && page.Type != "article" {
			logger.Warn("page type not supported")
			return
		}

		res, err := wikiClient.GetPageSource(page.URL)
		checkError(err)
		checkWikiError(res.Error)

		outMarkdown, err := trasformer.Transform([]byte(res.Data.Body))
		if err != nil {
			logger.Error("can't transform from wiki syntax", aLog.Error(err))
			os.Exit(1)
		}

		relativePath := strings.TrimPrefix(page.URL, "/") + ".md"
		tocItem.Link = relativePath

		destPath := filepath.Join(*outDir, relativePath)
		destDir := filepath.Dir(destPath)
		_ = ensureDir(destDir)

		err = ioutil.WriteFile(destPath, outMarkdown, 0644)
		if err != nil {
			logger.Error("page write failed", aLog.Error(err), aLog.String("Path", destPath))
			os.Exit(1)
		}

		pageFiles, err := wikiClient.GetPageFiles(page.URL)
		if err != nil {
			logger.Error("can't list files", aLog.Error(err))
			return
		}

		if pageFiles.Error != nil {
			logger.Error("list files finished with error", aLog.String("DebugMessage", pageFiles.Error.DebugMessage))
		} else if pageFiles.Data != nil {
			for _, pageFile := range pageFiles.Data.Files {
				logger.Info("processing file", aLog.String("URL", pageFile.URL), aLog.String("Name", pageFile.Name))

				fileName := pageFile.URL[strings.LastIndex(pageFile.URL, "/"):]
				content, err := wikiClient.GetPageFile(page.URL, fileName)
				if err != nil {
					logger.Error("can't download file", aLog.Error(err))
					continue
				}

				fileDestPath := filepath.Join(*outDir, "assets", strings.TrimPrefix(page.URL, "/"), fileName)
				fileDestDir := filepath.Dir(fileDestPath)
				_ = ensureDir(fileDestDir)

				err = ioutil.WriteFile(fileDestPath, content, 0644)
				if err != nil {
					logger.Error("file write failed", aLog.Error(err), aLog.String("Path", fileDestPath))
					os.Exit(1)
				}
			}
		}
	}

	logger.Info("processing cluster", aLog.String("Cluster", *wikiCluster), aLog.Int("Depth", *wikiDepth), aLog.String("OutDir", *outDir))

	tocItem := &TocItem{
		Items: make([]TocItem, 0),
	}

	rootPageRes, err := wikiClient.GetPage(*wikiCluster)
	checkError(err)
	checkWikiError(rootPageRes.Error)
	processPage(&wiki.Page{
		URL:   rootPageRes.Data.URL,
		Type:  rootPageRes.Data.PageType,
		Title: rootPageRes.Data.Title,
	}, tocItem)

	treeRes, err := wikiClient.GetPageTree(*wikiCluster, *wikiDepth)
	checkError(err)
	checkWikiError(treeRes.Error)
	if treeRes.Data.LimitExceeded {
		logger.Error("grow up wiki depth")
		os.Exit(1)
	}

	iterateSubPages(treeRes.Data.SubPages, tocItem, processPage)

	logger.Info("generating toc")

	tocTree := TocTree{
		Title: "WikiDump",
		Items: []TocItem{*tocItem},
	}
	tocYaml, err := yaml.Marshal(&tocTree)
	if err != nil {
		logger.Error("can't marshal toc.yaml", aLog.Error(err))
		os.Exit(1)
	}

	err = ioutil.WriteFile(filepath.Join(*outDir, "toc.yaml"), tocYaml, 0644)
	if err != nil {
		logger.Error("can't write toc", aLog.Error(err))
		os.Exit(1)
	}
}
