package parser

import (
	"bytes"

	"github.com/yuin/goldmark"
	"github.com/yuin/goldmark/ast"
	"github.com/yuin/goldmark/parser"
	"github.com/yuin/goldmark/text"
	"github.com/yuin/goldmark/util"
)

type wikiFencedCodeBlockParser struct {
}

var defaultFencedCodeBlockParser = &wikiFencedCodeBlockParser{}

// NewWikiFencedCodeBlockParser returns a new BlockParser that
// parses fenced code blocks.
func NewWikiFencedCodeBlockParser() parser.BlockParser {
	return defaultFencedCodeBlockParser
}

type wikiFenceData struct {
	char   byte
	indent int
	length int
	node   ast.Node
}

var wikiFencedCodeBlockInfoKey = parser.NewContextKey()

func (b *wikiFencedCodeBlockParser) Trigger() []byte {
	return []byte{'%'}
}

func (b *wikiFencedCodeBlockParser) Open(parent ast.Node, reader text.Reader, pc parser.Context) (ast.Node, parser.State) {
	line, segment := reader.PeekLine()
	pos := pc.BlockOffset()
	if pos < 0 || line[pos] != '%' {
		return nil, parser.NoChildren
	}
	findent := pos
	fenceChar := line[pos]
	i := pos
	for ; i < len(line) && line[i] == fenceChar; i++ {
	}
	oFenceLength := i - pos
	if oFenceLength != 2 {
		return nil, parser.NoChildren
	}
	node := ast.NewFencedCodeBlock(nil)
	if i < len(line)-1 {
		rest := line[i:]
		left := util.TrimLeftSpaceLength(rest)
		right := util.TrimRightSpaceLength(rest)
		if len(rest)-right-left >= 3 && rest[left] == '(' && rest[len(rest)-right-1] == ')' {
			infoStart, infoStop := segment.Start-segment.Padding+i+left+1, segment.Stop-right-1
			if fenceChar == '%' && bytes.IndexByte(rest, '%') > -1 {
				return nil, parser.NoChildren
			} else if infoStart != infoStop {
				node.Info = ast.NewTextSegment(text.NewSegment(infoStart, infoStop))
			}
		} else if !util.IsBlank(rest) {
			infoStart, infoStop := segment.Start-segment.Padding+i, segment.Stop
			if fenceChar == '%' && bytes.IndexByte(rest, '%') > -1 {
				return nil, parser.NoChildren
			} else if infoStart != infoStop {
				node.Lines().Append(text.NewSegmentPadding(infoStart, infoStop, 0))
			}
		}
	}
	pc.Set(wikiFencedCodeBlockInfoKey, &wikiFenceData{fenceChar, findent, oFenceLength, node})
	return node, parser.NoChildren
}

func (b *wikiFencedCodeBlockParser) Continue(node ast.Node, reader text.Reader, pc parser.Context) parser.State {
	line, segment := reader.PeekLine()
	fdata := pc.Get(wikiFencedCodeBlockInfoKey).(*wikiFenceData)
	w, pos := util.IndentWidth(line, reader.LineOffset())
	if w < 4 {
		i := pos
		for ; i < len(line) && line[i] == fdata.char; i++ {
		}
		length := i - pos
		if length >= fdata.length && util.IsBlank(line[i:]) {
			newline := 1
			if line[len(line)-1] != '\n' {
				newline = 0
			}
			reader.Advance(segment.Stop - segment.Start - newline - segment.Padding)
			return parser.Close
		}
	}

	if len(line) > 0 {
		start := -1
		end := -1
		for i := 0; i < len(line); i++ {
			if line[i] == fdata.char {
				if start < 0 {
					start = i
				}
			} else if start >= 0 && end < 0 {
				end = i
			}
		}
		if end > start && end-start == fdata.length {
			seg := text.NewSegmentPadding(segment.Start, segment.Start+start, 0)
			node.Lines().Append(seg)
			reader.Advance(end)
			return parser.Close
		}
	}

	pos, padding := util.DedentPositionPadding(line, reader.LineOffset(), segment.Padding, fdata.indent)

	seg := text.NewSegmentPadding(segment.Start+pos, segment.Stop, padding)
	node.Lines().Append(seg)
	reader.AdvanceAndSetPadding(segment.Stop-segment.Start-pos-1, padding)
	return parser.Continue | parser.NoChildren
}

func (b *wikiFencedCodeBlockParser) Close(node ast.Node, reader text.Reader, pc parser.Context) {
	fdata := pc.Get(wikiFencedCodeBlockInfoKey).(*wikiFenceData)
	if fdata.node == node {
		pc.Set(wikiFencedCodeBlockInfoKey, nil)
	}
}

func (b *wikiFencedCodeBlockParser) CanInterruptParagraph() bool {
	return true
}

func (b *wikiFencedCodeBlockParser) CanAcceptIndentedLine() bool {
	return false
}

type wikifencedcodeblock struct {
}

var WikiFencedCodeBlock = &wikifencedcodeblock{}

func (e *wikifencedcodeblock) Extend(m goldmark.Markdown) {
	m.Parser().AddOptions(parser.WithBlockParsers(
		util.Prioritized(NewWikiFencedCodeBlockParser(), 750),
	))
}
