package parser

import (
	"github.com/yuin/goldmark"
	"github.com/yuin/goldmark/ast"
	"github.com/yuin/goldmark/parser"
	"github.com/yuin/goldmark/text"
	"github.com/yuin/goldmark/util"
)

var attrNameID = []byte("id")

type wikiHeadingParser struct {
}

// NewWikiHeadingParser return a new BlockParser that can parse wiki headings.
func NewWikiHeadingParser() parser.BlockParser {
	p := &wikiHeadingParser{}
	return p
}

func (b *wikiHeadingParser) Trigger() []byte {
	return []byte{'='}
}

func (b *wikiHeadingParser) Open(parent ast.Node, reader text.Reader, pc parser.Context) (ast.Node, parser.State) {
	line, segment := reader.PeekLine()
	pos := pc.BlockOffset()
	if pos < 0 {
		return nil, parser.NoChildren
	}
	i := pos
	for ; i < len(line) && line[i] == '='; i++ {
	}
	level := i - pos
	if i == pos || level > 6 {
		return nil, parser.NoChildren
	}
	l := util.TrimLeftSpaceLength(line[i:])
	start := i + l
	if start >= len(line) {
		start = len(line) - 1
	}
	stop := len(line) - util.TrimRightSpaceLength(line)

	node := ast.NewHeading(level)

	if line[start] == '(' {
		k := start + 1
		for ; k < stop && line[k] != ')'; k++ {
		}
		if k < stop && line[k] == ')' {
			node.SetAttribute(attrNameID, line[start+1:k])
			start = k + 1
		}
	}

	stop = len(line) - util.TrimRightSpaceLength(line)
	if stop <= start { // empty headings like '==[space]'
		stop = start
	} else {
		i = stop - 1
		for ; line[i] == '=' && i >= start; i-- {
		}
		if i != stop-1 && !util.IsSpace(line[i]) {
			i = stop - 1
		}
		i++
		stop = i
	}

	if len(util.TrimRight(line[start:stop], []byte{'='})) != 0 { // empty heading like '=== ==='
		node.Lines().Append(text.NewSegment(segment.Start+start-segment.Padding, segment.Start+stop-segment.Padding))
	}

	return node, parser.NoChildren
}

func (b *wikiHeadingParser) Continue(node ast.Node, reader text.Reader, pc parser.Context) parser.State {
	return parser.Close
}

func (b *wikiHeadingParser) Close(node ast.Node, reader text.Reader, pc parser.Context) {
}

func (b *wikiHeadingParser) CanInterruptParagraph() bool {
	return true
}

func (b *wikiHeadingParser) CanAcceptIndentedLine() bool {
	return false
}

type wikiheading struct {
}

var WikiHeading = &wikiheading{}

func (e *wikiheading) Extend(m goldmark.Markdown) {
	m.Parser().AddOptions(parser.WithBlockParsers(
		util.Prioritized(NewWikiHeadingParser(), 650),
	))
}
