package parser

import (
	"strings"

	"github.com/yuin/goldmark"
	"github.com/yuin/goldmark/ast"
	"github.com/yuin/goldmark/parser"
	"github.com/yuin/goldmark/text"
	"github.com/yuin/goldmark/util"
)

type wikiImageParser struct {
}

var defaultWikiImageParser = &wikiImageParser{}

func NewWikiImageParser() parser.InlineParser {
	return defaultWikiImageParser
}

func (s *wikiImageParser) Trigger() []byte {
	return []byte{' '}
}

func (s *wikiImageParser) Parse(parent ast.Node, block text.Reader, pc parser.Context) ast.Node {
	line, segment := block.PeekLine()
	lineString := string(line)
	widthIndex := strings.IndexByte(lineString, 'x')
	heightIndex := strings.IndexByte(lineString, ':')
	spaceIndex := strings.IndexByte(lineString, ' ')
	slashIndex := strings.LastIndex(lineString, "/")
	if spaceIndex < 0 {
		spaceIndex = strings.IndexByte(lineString, '\n')
	}
	if widthIndex < 0 || heightIndex < 0 || widthIndex > heightIndex {
		return nil
	}
	if spaceIndex < 0 {
		spaceIndex = len(lineString)
	}
	if heightIndex > spaceIndex {
		return nil
	}
	if slashIndex < 0 {
		slashIndex = heightIndex
	}
	width := line[0:widthIndex]
	height := line[widthIndex+1 : heightIndex]
	url := line[heightIndex+1 : spaceIndex]
	block.Advance(spaceIndex)

	link := ast.NewLink()
	link.AppendChild(link, ast.NewTextSegment(text.NewSegment(segment.Start+slashIndex+1, segment.Start+spaceIndex)))
	link.Destination = url
	image := ast.NewImage(link)
	image.SetAttribute([]byte("width"), width)
	image.SetAttribute([]byte("height"), height)
	return image
}

type wikiimage struct {
}

var WikiImage = &wikiimage{}

func (e *wikiimage) Extend(m goldmark.Markdown) {
	m.Parser().AddOptions(parser.WithInlineParsers(
		util.Prioritized(NewWikiImageParser(), 50),
	))
}
