#!/bin/bash

PATH=/usr/local/bin:/usr/local/sbin:/usr/bin:/usr/sbin:/bin:/sbin
REPO_URL="http://mirror.yandex.ru/ubuntu"

# defaults
IMAGE_NAME=${IMAGE_NAME:-"ubuntu-20.04"}
KEEP_CHROOT=${KEEP_CHROOT:-NO}
DEBUG=${DEBUG:-NO}
CHROOT_PATH=${CHROOT_PATH:-"$(pwd)/$IMAGE_NAME"}
DISTRIB=${DISTRIB:-focal}
ARCH=${ARCH:-amd64}
INTERACTIVE=${INTERACTIVE:-NO}


# packages
EXCLUDE_PACKAGES="resolvconf,irqbalance,locate,ppp,pppoeconf,qmail,nscd"
INCLUDE_PACKAGES="acpid,athena-jot,bash-completion,bc,curl,daemon,dbus,dmidecode,\
edac-utils,ethtool,file,grub-pc,hdparm,hwloc-nox,ifplugd,info,ipmitool,\
language-pack-en,less,libcap2-bin,libpam-cap,libpam-sss,logrotate,lsof,lsscsi,ltrace,mdadm,\
ngrep,numactl,openipmi,openssh-server,pciutils,psmisc,pv,python3,python3-httplib2,python3-lxml,\
rsyslog,sudo,time,tree,tzdata,wget,lvm2,gpg,dirmngr,gpg-agent"

log_debug()
{
    if [ "${DEBUG}" = "YES" ] ; then
        echo "$@"
    fi
}

log_info()
{
    echo "INFO: $@"
}

log_error()
{
    echo "ERROR: $@"
}

bootstrap_image()
{
    local path=${1}
    local source_list=${path}/etc/apt/sources.list
    local output
    local mybootstrap=$(which debootstrap)

    ${mybootstrap} --components=main,restricted,universe,multiverse \
        --exclude="$EXCLUDE_PACKAGES" --include="$INCLUDE_PACKAGES" \
        "$DISTRIB" ${path} "$REPO_URL" || return 3

    sed -i -r "/\s${DISTRIB}-(updates|security)\s/d" ${source_list} 2> /dev/null
    echo "deb $REPO_URL ${DISTRIB}-security main restricted universe multiverse" >> ${source_list}
    echo "deb $REPO_URL ${DISTRIB}-updates main restricted universe multiverse" >> ${source_list}
    cat <<EOF > ${path}/etc/resolv.conf
search yandex.ru
nameserver 2a02:6b8:0:3400::5005
nameserver 2a02:6b8::1:1
nameserver 2a02:6b8:0:3400::1
EOF
    return 0
}

chroot_prepare()
{
    local chroot=${1}
    rm -vf ${chroot}/etc/resolv.conf
    cat <<EOF > ${chroot}/etc/resolv.conf
search yandex.ru
nameserver 2a02:6b8:0:3400::5005
nameserver 2a02:6b8::1:1
nameserver 2a02:6b8:0:3400::1
EOF

    chroot ${chroot} /bin/sh <<EOS
        mkdir /root/sbin && \
        (cp /bin/true /root/sbin/initctl ;
        cp /bin/true /root/sbin/invoke-rc.d ;
        cp /bin/true /root/sbin/restart ;
        cp /bin/true /root/sbin/start ;
        cp /bin/true /root/sbin/stop ;
        cp /bin/true /root/sbin/start-stop-daemon ;
        cp /bin/true /root/sbin/service ;
        true ) && \
        echo 'localhost' > /etc/hostname && \
        mount -t proc none /proc && \
        mount -t devtmpfs devtmpfs /dev && \
        mount -t devpts devpts /dev/pts && \
        mount -t sysfs sysfs /sys && \
        mount -t tmpfs tmpfs /run && \
        ln -sf /proc/self/mounts /etc/mtab && \
        (test -f /var/run/utmp || cat /dev/null > /var/run/utmp) && \
        locale-gen en_US.UTF-8 ru_RU.UTF-8
EOS
}

chroot_update()
{
    local chroot=${1}
    chroot ${chroot} /bin/sh <<EOS
    apt-get update && \
    DEBIAN_FRONTEND=noninteractive PATH=/root/sbin:\$PATH apt-get -y dist-upgrade || true \
    DEBIAN_FRONETED=noninteractive PATH=/root/sbin:\$PATH apt-get -y dist-upgrade && \
    echo 'UPGRADE COMPLETE!!!' && \
    apt-get clean
    apt-key adv --keyserver keyserver.ubuntu.com --recv-keys 7FCD11186050CD1A
    cat <<EOF > /etc/apt/sources.list
# Ubuntu repos
deb http://search-upstream-focal.dist.yandex.ru/search-upstream-focal unstable/amd64/
deb http://search-upstream-focal.dist.yandex.ru/search-upstream-focal unstable/all/
deb http://search-upstream-focal.dist.yandex.ru/search-upstream-focal stable/amd64/
deb http://search-upstream-focal.dist.yandex.ru/search-upstream-focal stable/all/

# These are Yandex repositories for Ubuntu 20.04 (Focal).
deb http://search-focal.dist.yandex.ru/search-focal stable/all/
deb http://search-focal.dist.yandex.ru/search-focal unstable/all/

deb http://search-focal.dist.yandex.ru/search-focal stable/amd64/
deb http://search-focal.dist.yandex.ru/search-focal unstable/amd64/

# These are Yandex repositories for Ubuntu 20.04 (Focal).
deb http://yandex-focal.dist.yandex.ru/yandex-focal stable/all/
deb http://yandex-focal.dist.yandex.ru/yandex-focal stable/amd64/
EOF
    apt-get update && \
    apt-get -y install --allow-downgrades --no-install-recommends adduser=3.118ubuntu2 apt=2.0.5 apt-utils=2.0.5 athena-jot=9.0-7 base-files=11ubuntu5.3 bash=5.0-6ubuntu1.1 bash-completion=1:2.10-1ubuntu1 bc=1.07.1-2build1 bsdutils=1:2.34-0.1ubuntu9.1 busybox-initramfs=1:1.30.1-4ubuntu6.3 ca-certificates=20210119~20.04.1 coreutils=8.30-3ubuntu2 cpio=2.13+dfsg-2 curl=7.68.0-1ubuntu2.5 dash=0.5.10.2-6 dbus=1.12.16-2ubuntu2.1 debconf=1.5.73 debianutils=4.9.1 dirmngr=2.2.19-3ubuntu2.1 dmidecode=3.2-3 dpkg=1.19.7ubuntu3 e2fslibs=1.45.5-2ubuntu1 e2fsprogs=1.45.5-2ubuntu1 edac-utils=0.18-1build1 ethtool=1:5.4-1 file=1:5.38-4 freeipmi-common=1.6.4-3ubuntu1.1 gcc-10-base=10.2.0-5ubuntu1~20.04 gettext-base=0.19.8.1-10build1 gnupg=2.2.19-3ubuntu2.1 gnupg-l10n=2.2.19-3ubuntu2.1 gnupg-utils=2.2.19-3ubuntu2.1 gpg=2.2.19-3ubuntu2.1 gpg-agent=2.2.19-3ubuntu2.1 gpg-wks-client=2.2.19-3ubuntu2.1 gpg-wks-server=2.2.19-3ubuntu2.1 gpgconf=2.2.19-3ubuntu2.1 gpgsm=2.2.19-3ubuntu2.1 gpgv=2.2.19-3ubuntu2.1 grub-common=2.04-1ubuntu26.12 grub-gfxpayload-lists='0.7' grub-pc=2.04-1ubuntu26.12 grub-pc-bin=2.04-1ubuntu26.12 grub2-common=2.04-1ubuntu26.12 hdparm=9.58+ds-4 hwloc-nox=2.1.0+dfsg-4 ifupdown=0.8.35ubuntu1 init='1.57' init-system-helpers='1.57' initramfs-tools=0.136ubuntu6.5 initramfs-tools-bin=0.136ubuntu6.5 initramfs-tools-core=0.136ubuntu6.5 ipmitool=1.8.18-8 iproute2=5.5.0-1ubuntu1 iputils-ping=3:20190709-3 klibc-utils=2.0.7-1ubuntu5 kmod=27-1ubuntu2 language-pack-en=1:20.04+20210121 language-pack-en-base=1:20.04+20210121 libacl1=2.2.53-6 libapparmor1=2.13.3-7ubuntu5.1 libapt-pkg6.0=2.0.5 libargon2-1=0~20171227-0.2 libasn1-8-heimdal=7.7.0+dfsg-1ubuntu1 libasprintf0v5=0.19.8.1-10build1 libassuan0=2.5.3-7ubuntu2 libattr1=1:2.4.48-5 libaudit-common=1:2.8.5-2ubuntu6 libaudit1=1:2.8.5-2ubuntu6 libblkid1=2.34-0.1ubuntu9.1 libbrotli1=1.0.7-6ubuntu0.1 libbsd0=0.10.0-1 libbz2-1.0=1.0.8-2 libc-bin=2.31-0ubuntu9.3 libc6=2.31-0ubuntu9.3 libcap-ng0=0.7.9-2.1build1 libcap2=1:2.32-1 libcap2-bin=1:2.32-1 libcbor0.6=0.6.0-0ubuntu1 libcom-err2=1.45.5-2ubuntu1 libcomerr2=1.45.5-2ubuntu1 libcrypt1=1:4.4.10-10ubuntu4 libcryptsetup12=2:2.2.2-3ubuntu2.3 libcurl3-gnutls=7.68.0-1ubuntu2.5 libcurl4=7.68.0-1ubuntu2.5 libdb5.3=5.3.28+dfsg1-0.6ubuntu2 libdbus-1-3=1.12.16-2ubuntu2.1 libdevmapper1.02.1=2:1.02.167-1ubuntu1 libedac1=0.18-1build1 libedit2=3.1-20191231-1 libefiboot1=37-2ubuntu2.2 libefivar1=37-2ubuntu2.2 libelf1=0.176-1.1build1 libestr0=0.1.10-2.1 libexpat1=2.2.9-1build1 libext2fs2=1.45.5-2ubuntu1 libfastjson4=0.99.8-2 libfdisk1=2.34-0.1ubuntu9.1 libffi7=3.3-4 libfido2-1=1.3.1-1ubuntu2 libfreeipmi17=1.6.4-3ubuntu1.1 libfreetype6=2.10.1-2ubuntu0.1 libfuse2=2.9.9-3 libgcc-s1=10.2.0-5ubuntu1~20.04 libgcc1=1:10.2.0-5ubuntu1~20.04 libgcrypt20=1.8.5-5ubuntu1 libgdbm-compat4=1.18.1-5 libgdbm6=1.18.1-5 libgmp10=2:6.2.0+dfsg-4 libgnutls-openssl27=3.6.13-2ubuntu1.3 libgnutls30=3.6.13-2ubuntu1.3 libgpg-error0=1.37-1 libgssapi-krb5-2=1.17-6ubuntu4.1 libgssapi3-heimdal=7.7.0+dfsg-1ubuntu1 libhcrypto4-heimdal=7.7.0+dfsg-1ubuntu1 libheimbase1-heimdal=7.7.0+dfsg-1ubuntu1 libheimntlm0-heimdal=7.7.0+dfsg-1ubuntu1 libhogweed5=3.5.1+really3.5.1-2ubuntu0.2 libhwloc15=2.1.0+dfsg-4 libhx509-5-heimdal=7.7.0+dfsg-1ubuntu1 libidn11=1.33-2.2ubuntu2 libidn2-0=2.2.0-2 libip4tc2=1.8.4-3ubuntu2 libjson-c4=0.13.1+dfsg-7ubuntu0.3 libk5crypto3=1.17-6ubuntu4.1 libkeyutils1=1.6-6ubuntu1 libklibc=2.0.7-1ubuntu5 libkmod2=27-1ubuntu2 libkrb5-26-heimdal=7.7.0+dfsg-1ubuntu1 libkrb5-3=1.17-6ubuntu4.1 libkrb5support0=1.17-6ubuntu4.1 libksba8=1.3.5-2 libldap-2.4-2=2.4.49+dfsg-2ubuntu1.8 libldap-common=2.4.49+dfsg-2ubuntu1.8 libltdl7=2.4.6-14 liblz4-1=1.9.2-2ubuntu0.20.04.1 liblzma5=5.2.4-1ubuntu1 libmagic-mgc=1:5.38-4 libmagic1=1:5.38-4 libmnl0=1.0.4-2 libmount1=2.34-0.1ubuntu9.1 libmpdec2=2.4.2-3 libmysqlclient21=8.0.25-0ubuntu0.20.04.1 libncurses5=6.2-0ubuntu2 libncurses6=6.2-0ubuntu2 libncursesw5=6.2-0ubuntu2 libncursesw6=6.2-0ubuntu2 libnet1=1.1.6+dfsg-3.1build1 libnettle7=3.5.1+really3.5.1-2ubuntu0.2 libnghttp2-14=1.40.0-1build1 libnpth0=1.6-1 libnuma1=2.0.12-1 libopenipmi0=2.0.27-0ubuntu2 libp11-kit0=0.23.20-1ubuntu0.1 libpam-modules=1.3.1-5ubuntu4.2 libpam-modules-bin=1.3.1-5ubuntu4.2 libpam-runtime=1.3.1-5ubuntu4.2 libpam0g=1.3.1-5ubuntu4.2 libpcap0.8=1.9.1-3 libpci3=1:3.6.4-1ubuntu0.20.04.1 libpcre2-8-0=10.34-7 libpcre3=2:8.39-12build1 libperl5.30=5.30.0-9ubuntu0.2 libpng16-16=1.6.37-2 libpopt0=1.16-14 libprocps8=2:3.3.16-1ubuntu2.2 libpsl5=0.21.0-1ubuntu1 libpwquality-common=1.4.2-1build1 libpython3.8-minimal=3.8.5-1~20.04.3 libpython3.8-stdlib=3.8.5-1~20.04.3 libreadline8=8.0-4 libroken18-heimdal=7.7.0+dfsg-1ubuntu1 librtmp1=2.4+20151223.gitfa8646d.1-2build1 libsasl2-2=2.1.27+dfsg-2 libsasl2-modules-db=2.1.27+dfsg-2 libseccomp2=2.5.1-1ubuntu1~20.04.1 libselinux1=3.0-1build2 libsemanage-common=3.0-1build2 libsemanage1=3.0-1build2 libsensors-config=1:3.6.0-2ubuntu1 libsensors5=1:3.6.0-2ubuntu1 libsepol1=3.0-1 libsmartcols1=2.34-0.1ubuntu9.1 libsnmp-base=5.8+dfsg-2ubuntu2.3 libsnmp35=5.8+dfsg-2ubuntu2.3 libsqlite3-0=3.31.1-4ubuntu0.2 libss2=1.45.5-2ubuntu1 libssh-4=0.9.3-2ubuntu2.1 libssl1.1=1.1.1f-1ubuntu2.4 libstdc++6=10.2.0-5ubuntu1~20.04 libsysfs2=2.1.0+repack-6 libsystemd0=245.4-4ubuntu3.7 libtasn1-6=4.16.0-2 libtinfo5=6.2-0ubuntu2 libtinfo6=6.2-0ubuntu2 libudev1=245.4-4ubuntu3.7 libunistring2=0.9.10-2 libusb-0.1-4=2:0.1.12-32 libustr-1.0-1=1.0.4-6 libuuid1=2.34-0.1ubuntu9.1 libwind0-heimdal=7.7.0+dfsg-1ubuntu1 libwrap0=7.6.q-30 libxtables12=1.8.4-3ubuntu2 libzstd1=1.4.4+dfsg-3ubuntu0.1 linux-base=4.5ubuntu3.1 locales=2.31-0ubuntu9.3 login=1:4.8.1-1ubuntu5.20.04 logsave=1.45.5-2ubuntu1 lsb-base=11.1.0ubuntu2 lsof=4.93.2+dfsg-1ubuntu0.20.04.1 lsscsi=0.30-0.1 ltrace=0.7.3-6.1ubuntu1 lz4=1.9.2-2ubuntu0.20.04.1 mawk=1.3.4.20200120-2 mdadm=4.1-5ubuntu1.2 mime-support=3.64ubuntu1 mount=2.34-0.1ubuntu9.1 mysql-common=5.8+1.0.5ubuntu2 ngrep=1.47+ds1-2 numactl=2.0.12-1 openipmi=2.0.27-0ubuntu2 openssh-client=1:8.2p1-4ubuntu0.2 openssh-server=1:8.2p1-4ubuntu0.2 openssh-sftp-server=1:8.2p1-4ubuntu0.2 openssl=1.1.1f-1ubuntu2.4 passwd=1:4.8.1-1ubuntu5.20.04 pci.ids=0.0~2020.03.20-1 pciutils=1:3.6.4-1ubuntu0.20.04.1 perl-base=5.30.0-9ubuntu0.2 perl-modules-5.30=5.30.0-9ubuntu0.2 pinentry-curses=1.1.0-3build1 procps=2:3.3.16-1ubuntu2.1 pv=1.6.6-1 python3.8=3.8.5-1~20.04.3 python3.8-minimal=3.8.5-1~20.04.3 readline-common=8.0-4 rsyslog=8.2001.0-1ubuntu1.1 sensible-utils=0.0.12+nmu1 sudo=1.8.31-1ubuntu1.2 systemd=245.4-4ubuntu3.7 systemd-sysv=245.4-4ubuntu3.7 systemd-timesyncd=245.4-4ubuntu3.7 sysvinit-utils=2.96-2.1ubuntu1 tar=1.30+dfsg-7ubuntu0.20.04.1 time=1.7-25.1build1 tzdata=2021a-0ubuntu0.20.04 ubuntu-keyring=2020.02.11.4 ucf=3.0038+nmu1 udev=245.4-4ubuntu3.7 util-linux=2.34-0.1ubuntu9.1 vim-common=2:8.1.2269-1ubuntu5 vim-tiny=2:8.1.2269-1ubuntu5 wget=1.20.3-1ubuntu1 xxd=2:8.1.2269-1ubuntu5 zlib1g=1:1.2.11.dfsg-2ubuntu1.2 sssd-common=2.2.3-3ubuntu0.4
    apt-get -y autoremove
EOS
}

chroot_stop()
{
    local chroot=${1}

    # stop all other process in chroot
    for signal in 15 9 ; do
        for p in $(find /proc/*/root) ; do
            plink=$(readlink -f $p)
            if echo ${plink} | grep -q "${chroot}" 2> /dev/null ; then
                log_debug "$(basename $(dirname ${p})) would be kill by ${signal}"
                kill -${signal} $(basename $(dirname $p))
            fi
        done
        sleep 3
    done

    chroot ${chroot} /bin/sh <<EOS
        rm /etc/mtab
        rm -rf /root/sbin
        umount /proc /sys /run /dev/pts /dev
        find /var/log -type f -size +0 -exec truncate -s 0 {} +
        rm /var/lib/dbus/machine-id
EOS
}

add_dist_yandex_gpg()
{
    local chroot=${1}
    chroot ${chroot} /bin/sh <<EOS
       curl -sL 'http://keyserver.ubuntu.com/pks/lookup?op=get&search=0x7FCD11186050CD1A' | \
        apt-key add -
EOS
}

apply_settings()
{
    local root=${1}

    echo -n > "${root}/root/.bash_history"
    echo -n > "${root}/etc/hostname"
    echo 'localhost.localdomain' > "${root}/etc/mailname"
    sed -i "s/search\s.*/search yandex.ru/" "${root}/etc/resolv.conf"

    if [ -f "${root}/etc/postfix/main.cf" ] ; then
        sed -i 's/myhostname\s.*/myhostname = localhost/i;s/mydestination\s.*/mydestination = localhost, localhost.localdomain/i' \
            "${root}/etc/postfix/main.cf"
    fi

    if [ -f "${root}/etc/mdadm/mdadm.conf" ] ; then
        sed -i "/^ARRAY.*/d" "${root}/etc/mdadm/mdadm.conf"
    fi

    if [ -f "${root}/etc/initramfs-tools/conf.d/mdadm" ] ; then
        sed -i 's/BOOT_DEGRADED=false/BOOT_DEGRADED=true/i' \
         "${root}/etc/initramfs-tools/conf.d/mdadm"
    fi

    if [ -f "${root}/etc/default/rcS" ] ; then
        sed -i '/\s*FSCKFIX/d' "${root}/etc/default/rcS"
        echo 'FSCKFIX=yes' >> "${root}/etc/default/rcS"
    fi

    local grub_config="${root}/etc/default/grub"
    if [ -f "${root}/etc/default/grub" ] ; then
        #remove quiet and splash from kernel boot options
        sed -i '/GRUB_CMDLINE_LINUX/s/\(quiet\|splash\)//g' "${grub_config}"

        #add kernel boot options for legacy named network devices (like 'ethX')
        sed -i 's/^\(\s*GRUB_CMDLINE_LINUX\(_DEFAULT\)\?="\)/\1biosdevname=0 net.ifnames=0 /g' "${grub_config}"

        # disable netplan
        # sed -i 's@^\(\s*GRUB_CMDLINE_LINUX\(_DEFAULT\)\?="\)@\1netcfg/do_not_use_netplan=true @g' "${grub_config}"
    fi
    chroot ${root} /bin/sh <<EOS
        systemctl disable sssd-sudo.socket
        systemctl disable sssd-pam.socket
        systemctl disable sssd-pam-priv.socket
        systemctl disable sssd-nss.socket
EOS

    echo -e "# RNDSVC-357: fix building of raid0 with different size components\noptions raid0 default_layout=2" > "${root}/etc/modprobe.d/raid0.conf" || \
        log_error "Failed to create raid0 layout file in /etc/modprobe.d"
}

mk_script_tun_intel_network_card()
{
    local root=${1}
    local scriptpath="${root}/etc/network/if-pre-up.d/intel-card-tun"

    cat > "${scriptpath}" <<EOT
#!/bin/sh

ETHTOOL=\$(which ethtool) || exit 0
DEFAULTS=/etc/default/intel-card-tun
test -f \${DEFAULTS} && . \${DEFAULTS}

TXRX=\${TXRX:-1024}

\$ETHTOOL -i \${IFACE} 2> /dev/null | grep -iqP 'driver:\s*(ix?gb|e1000)' || exit 0
\$ETHTOOL -g \${IFACE} | sed '0,/^Current/d' | \\
  grep '^[TR]X:' | sed 's/://' | tr '[:upper:]' '[:lower:]' | \\
    while read k v ; do
       [ \$v -lt \$TXRX ] &&  \$ETHTOOL -G \${IFACE} \$k \$TXRX > /dev/null 2>&1
    done

exit 0
EOT

    chmod a+x "${scriptpath}"
}

mk_script_eth_tso_off()
{
    local root=${1}
    local scriptpath="${root}/etc/network/if-pre-up.d/eth-tso-off"

    cat > "${scriptpath}" <<EOT
#!/bin/sh

ETHTOOL=\$(which ethtool) || exit 0

if [ "\$(cat /sys/class/net/\${IFACE}/type 2> /dev/null)" != "1" ] ; then
    exit 0
fi

if \$ETHTOOL -k \${IFACE} 2> /dev/null | grep 'tcp-segmentation-offload' \\
	| grep -qw 'on' ; then
    \$ETHTOOL -K \${IFACE} tso off
fi

exit 0
EOT

    chmod a+x "${scriptpath}"
}

pack_image()
{
    local path=${1}
    local name=${2}

    tar --numeric-owner -cjf "${name}" -C "${path}" \
        --exclude='./dev/*' --exclude='./run/*' .
}

chroot_cleanup()
{
    local path=${1}
    chroot ${path} /bin/sh <<EOS
    export DEBIAN_FORNTEND=noninteractive
    export PATH="/root/sbin:$PATH"
    apt-get -y purge netcat-openbsd libxtables11 isc-dhcp-client isc-dhcp-common dh-python net-tools
    apt-get -y autoremove
EOS
    rm -fv "$path/var/cache/apt"/*.bin
    rm -fv "$path/etc/machine-id"
    find "$path/var/lib/apt/lists"  -type f -delete
}

spawn_shell()
{
    local path="$1"
    chroot "$path" /bin/bash -c "export PATH=/root/sbin:$PATH; exec bash"
}

create_image()
{
    local output

    local image_tar="$(dirname ${CHROOT_PATH})/$IMAGE_NAME.tar.bz2"
    
    log_info "Bootstraping image ${DISTRIB} (${ARCH}) to ${CHROOT_PATH}"
    output=$(bootstrap_image "${CHROOT_PATH}" 2>&1) || {
        log_error "$output"
        return 4
    }
    log_debug "${output}"

    log_info "Preparing ${CHROOT_PATH} for update"
    output=$(chroot_prepare "${CHROOT_PATH}" 2>&1) || {
        log_error "failed mount fs in chroot. Output:\n$output"
        return 5
    }
    log_debug "$output"

    log_info "Updating chrooted system"
    output=$(chroot_update "${CHROOT_PATH}" 2>&1) || {
        log_error "failed update system. Output:\n$output"
        return 6
    }
    log_debug "$output"

    log_info "Add dist.yandex.ru repo key"
    add_dist_yandex_gpg "${CHROOT_PATH}"

    log_info "Cleaning up future image"
    output=$(chroot_cleanup "${CHROOT_PATH}" 2>&1) || {
        log_error "failed to cleanup. Output:\n$output"
        return 9
    }
    log_debug "${output}"

    if [ ${INTERACTIVE} = "YES" ]; then
        log_info "Spawning interactive shell"
        spawn_shell "${CHROOT_PATH}"
    fi

    log_info "Stoping ${CHROOT_PATH}"
    output=$(chroot_stop "${CHROOT_PATH}" 2>&1) || {
        log_error "failed unmount fs in chroot. Output:\n$output"
        return 7
    }
    log_debug "$output"

    log_info "Applying default settings"
    apply_settings "${CHROOT_PATH}" 2>&1

    log_info "Making script to tun tx/rx intel card"
    mk_script_tun_intel_network_card "${CHROOT_PATH}"

    log_info "Making script to disable tcp segments offload"
    mk_script_eth_tso_off "${CHROOT_PATH}"

    log_info "Packing image to ${image_tar}"
    output=$(pack_image "${CHROOT_PATH}" "${image_tar}" 2>&1) || {
        log_error "Error pack image to ${image_tar}. Output:\n"
        return 8
    }
    log_debug "${output}"

    if [ ${KEEP_CHROOT} = "YES" ]; then
        log_info "*NOT* Removing ${CHROOT_PATH}"
    else
        log_info "Removing ${CHROOT_PATH}"
        rm -rf "${CHROOT_PATH}"
    fi

    return 0
}

upload()
{
    local arc="$1"
    local image_name="$(basename $arc)"
    local image_path="$(dirname $arc)"
    pushd "$image_path"
    md5sum "$image_name" > "$image_name.md5"
    s3cmd --no-check-certificate put "$image_name" "s3://setup-rtc-images/$image_name"
    s3cmd --no-check-certificate put "$image_name.md5" "s3://setup-rtc-images/$image_name.md5"
    rm -fv "$image_name.md5"
    popd
}

usage()
{
    echo "$(basename $0) [FLAGS]"
    echo ""
    echo "-h|--help - this message"
    echo "-k|--keep - keep chroot after build"
    echo "-d|--debug - show debugging output"
    echo "-n|--name <name> - image name (will produce <name>.tar.bz2)"
    echo "-s|--stop <path> - stop chroot at <path>"
    echo "-i|--interactive - launch intercative shell in chroot after build"
    echo "--chroot <path> - start intercative chroot shell at <path> (also starts chroot)"
    echo "--upload <archive> - upload archive to s3://setup-rtc-images/<archive>"
}

main()
{
    if [ "$(whoami)" != "root" ]; then
        echo "only root can run $(basename $0). You are not root."
        return 1
    fi

    while (( "$#" )); do
        case "$1" in
            -k|--keep)
            KEEP_CHROOT="YES"
            shift
            ;;

            -d|--debug)
            DEBUG="YES"
            shift
            ;;

            -h|--help)
            usage
            exit 0
            ;;

            -i|--interactive)
            INTERACTIVE="YES"
            ;;

            -n|--name)
            if [ -n "$2" ] && [ ${2:0:1} != "-" ]; then
                IMAGE_NAME=$2
                CHROOT_PATH="$(pwd)/$IMAGE_NAME-$(date +%Y%m%d)"
                shift 2
            else
                log_error "Argument for $1 is missing"
                exit 1
            fi
            ;;

            -s|--stop)
            if [ -n "$2" ] && [ ${2:0:1} != "-" ]; then
                chroot_stop "$2"
                exit $?
            else
                log_error "Argument for $1 is missing"
                exit 1
            fi
            ;;

            --chroot)
            if [ -n "$2" ] && [ ${2:0:1} != "-" ]; then
                chroot_prepare "$2"
                spawn_shell "$2"
                chroot_stop "$2"
                exit $?
            else
                log_error "Argument for $1 is missing"
                exit 1
            fi
            ;;

        --upload)
            if [ -n "$2" ] && [ ${2:0:1} != "-" ]; then
                log_info "Uploading $2 to S3"
                upload $2
                exit $?
            else
                log_error "Argument for $1 is missing"
                exit 1
            fi
            ;;

            

            -*|--*=) # unsupported flags
            log_error "unsupported flags found"
            exit 1
            ;;
        esac
    done

    create_image
    exit $?
}

main "$@"
