#!/bin/bash

PATH=/usr/local/bin:/usr/local/sbin:/usr/bin:/usr/sbin:/bin:/sbin
REPO_URL="http://mirror.yandex.ru/ubuntu"

# defaults
IMAGE_NAME=${IMAGE_NAME:-"ubuntu-16.04"}
KEEP_CHROOT=${KEEP_CHROOT:-NO}
DEBUG=${DEBUG:-NO}
CHROOT_PATH=${CHROOT_PATH:-"$(pwd)/$IMAGE_NAME"}
DISTRIB=${DISTRIB:-xenial}
ARCH=${ARCH:-amd64}
INTERACTIVE=${INTERACTIVE:-NO}


# packages
EXCLUDE_PACKAGES="ubuntu-minimal,resolvconf,irqbalance,kexec-tools,locate,ppp,pppoeconf,qmail,nscd"
INCLUDE_PACKAGES="acpid,athena-jot,bash-completion,bc,curl,daemon,dbus,dmidecode,\
edac-utils,ethtool,file,grub-pc,hdparm,hwloc-nox,ifplugd,info,ipmitool,\
language-pack-en,less,libcap2-bin,libpam-cap,libpam-sss,logrotate,lsof,lsscsi,ltrace,mdadm,\
ngrep,numactl,openipmi,openssh-server,pciutils,psmisc,pv,python3,python3-httplib2,python3-lxml,\
rsyslog,sudo,time,tree,tzdata,wget,lvm2"

log_debug()
{
    if [ "${DEBUG}" = "YES" ] ; then
        echo "$@"
    fi
}

log_info()
{
    echo "INFO: $@"
}

log_error()
{
    echo "ERROR: $@"
}

bootstrap_image()
{
    local path=${1}
    local source_list=${path}/etc/apt/sources.list
    local output
    local mybootstrap=$(which debootstrap)

    ${mybootstrap} --components=main,restricted,universe,multiverse \
        --exclude="$EXCLUDE_PACKAGES" --include="$INCLUDE_PACKAGES" \
        "$DISTRIB" ${path} "$REPO_URL" || return 3

    sed -i -r "/\s${DISTRIB}-(updates|security)\s/d" ${source_list} 2> /dev/null
    echo "deb $REPO_URL ${DISTRIB}-security main restricted universe multiverse" >> ${source_list}
    echo "deb $REPO_URL ${DISTRIB}-updates main restricted universe multiverse" >> ${source_list}

    return 0
}

chroot_prepare()
{
    local chroot=${1}

    chroot ${chroot} /bin/sh <<EOS
        mkdir /root/sbin && \
        (cp /bin/true /root/sbin/initctl ;
        cp /bin/true /root/sbin/invoke-rc.d ;
        cp /bin/true /root/sbin/restart ;
        cp /bin/true /root/sbin/start ;
        cp /bin/true /root/sbin/stop ;
        cp /bin/true /root/sbin/start-stop-daemon ;
        cp /bin/true /root/sbin/service ;
        true ) && \
        echo 'localhost' > /etc/hostname && \
        mount -t proc none /proc && \
        mount -t devtmpfs devtmpfs /dev && \
        mount -t devpts devpts /dev/pts && \
        mount -t sysfs sysfs /sys && \
        mount -t tmpfs tmpfs /run && \
        ln -sf /proc/self/mounts /etc/mtab && \
        (test -f /var/run/utmp || cat /dev/null > /var/run/utmp) && \
        locale-gen en_US.UTF-8 ru_RU.UTF-8
EOS
}

chroot_update()
{
    local chroot=${1}

    chroot ${chroot} /bin/sh <<EOS
    apt-get update && \
    DEBIAN_FRONTEND=noninteractive PATH=/root/sbin:\$PATH apt-get -y dist-upgrade || true \
    DEBIAN_FRONETED=noninteractive PATH=/root/sbin:\$PATH apt-get -y dist-upgrade && \
    echo 'UPGRADE COMPLETE!!!' && \
    apt-get clean
    apt-key adv --keyserver keyserver.ubuntu.com --recv-keys 7FCD11186050CD1A
    cat <<EOF > /etc/apt/sources.list
# Ubuntu repos
deb http://search-upstream-xenial.dist.yandex.ru/search-upstream-xenial unstable/amd64/
deb http://search-upstream-xenial.dist.yandex.ru/search-upstream-xenial unstable/all/
deb http://search-upstream-xenial.dist.yandex.ru/search-upstream-xenial stable/amd64/
deb http://search-upstream-xenial.dist.yandex.ru/search-upstream-xenial stable/all/

# These are Yandex repositories for Ubuntu 16.04 (Xenial).
deb http://search-xenial.dist.yandex.ru/search-xenial stable/all/
deb http://search-xenial.dist.yandex.ru/search-xenial unstable/all/

deb http://search-xenial.dist.yandex.ru/search-xenial stable/amd64/
deb http://search-xenial.dist.yandex.ru/search-xenial unstable/amd64/

# These are Yandex repositories for Ubuntu 16.04 (Xenial).
deb http://yandex-xenial.dist.yandex.ru/yandex-xenial stable/all/
deb http://yandex-xenial.dist.yandex.ru/yandex-xenial stable/amd64/
EOF
    apt-get update && \
    apt-get -y install --allow-downgrades --no-install-recommends libgcc1=1:6.0.1-0ubuntu1 vim-common=2:7.4.1689-3ubuntu1.3 lsb-base=9.20160110ubuntu0.2 libcryptsetup4=2:1.6.6-5ubuntu2.1 libmnl0=1.0.3-5 libpci3=1:3.3.1-1.1ubuntu1.3 libfreeipmi16=1.4.11-1.1ubuntu4.1~0.16.04 libroken18-heimdal=1.7~git20150920+dfsg-4ubuntu1.16.04.1 libreadline6=6.3-8ubuntu2 klibc-utils=2.0.4-8ubuntu1.16.04.4 libgssapi3-heimdal=1.7~git20150920+dfsg-4ubuntu1.16.04.1 libopenipmi0=2.0.18-0ubuntu11.2 libsemanage-common=2.3-1build3 sensible-utils=0.0.9ubuntu0.16.04.1 libusb-0.1-4=2:0.1.12-28 systemd=229-4ubuntu21.27 grub-common=2.04-yandex1 libncursesw5=6.0+20160213-1ubuntu1 libsystemd0=229-4ubuntu21.27 coreutils=8.25-2ubuntu3~16.04 libhwloc5=1.11.2-3 gcc-5-base=5.4.0-6ubuntu1~16.04.12 libcap2-bin=1:2.43-1yandex0 sysv-rc=2.88dsf-59.3ubuntu2 dpkg=1.18.4ubuntu1.5 gpgv=1.4.20-1ubuntu3.3 libkeyutils1=1.5.9-8ubuntu1 ethtool=1:4.5-1 libpam-modules=1.1.8-3.2ubuntu2.1 libselinux1=2.4-3build2 liblzma5=5.1.1alpha+20120614-2ubuntu2 initramfs-tools-core=0.122ubuntu8.14 cpio=2.11+dfsg-5ubuntu1 bash-completion=1:2.1-4.2ubuntu1.1 ca-certificates=20170717~16.04.1 libacl1=2.2.52-3 libncurses5=6.0+20160213-1ubuntu1 readline-common=6.3-8ubuntu2 libpwquality-common=1.3.0-0ubuntu1 libgpg-error0=1.21-2ubuntu1 libpam0g=1.1.8-3.2ubuntu2.3 libkrb5-26-heimdal=1.7~git20150920+dfsg-4ubuntu1.16.04.1 numactl=2.0.11-1ubuntu1 edac-utils=0.18-1 dbus=1.10.6-1ubuntu3.3 libpam-modules-bin=1.1.8-3.2ubuntu2.1 libseccomp2=2.4.3-1ubuntu3.16.04.3 busybox-initramfs=1:1.22.0-15ubuntu1.4 e2fslibs=1.44.1-1ubuntu1.3-yandex0 gcc-6-base=6.0.1-0ubuntu1 gettext-base=0.19.7-2ubuntu3 libpopt0=1.16-10 debconf=1.5.58ubuntu2 libssl1.0.0=1.0.2g-1ubuntu4.17 sysvinit-utils=2.88dsf-59.3ubuntu2 libk5crypto3=1.13.2+dfsg-5ubuntu2.1 libidn11=1.32-3ubuntu1.2 adduser=3.113+nmu3ubuntu4 grub2-common=2.04-yandex1 libudev1=229-4ubuntu21.17+yandex1 udev=229-4ubuntu21.17+yandex1 libfuse2=2.9.4-1ubuntu3.1 grub-gfxpayload-lists=0.7 mdadm=3.3-2ubuntu7.6 libpcap0.8=1.7.4-2ubuntu0.1 kmod=22-1ubuntu5.2 openssh-client=1:7.2p2-4ubuntu2.8 language-pack-en=1:16.04+20161009 libmpdec2=2.4.2-1 insserv=1.14.0-5ubuntu3 libwrap0=7.6.q-25 libsqlite3-0=3.11.0-1ubuntu1.5 libkrb5-3=1.13.2+dfsg-5ubuntu2.1 libsmartcols1=2.27.1-6ubuntu3.10 libpam-runtime=1.1.8-3.2ubuntu2.1 libss2=1.44.1-1ubuntu1.3-yandex0 libmagic1=1:5.25-2ubuntu1.1 bc=1.06.95-9build1 libcap2=1:2.43-1yandex0 libpcre3=2:8.38-3.1 libmount1=2.27.1-6ubuntu3.10 dmidecode=3.0-2ubuntu0.1 libedit2=3.1-20150325-1ubuntu2 mime-support=3.59ubuntu1 libcomerr2=1.44.1-1ubuntu1.3-yandex0 bash=4.3-14ubuntu1.2 libldap-2.4-2=2.4.42+dfsg-2ubuntu3.9 mawk=1.3.3-17ubuntu2 libnet1=1.1.6+dfsg-3 hwloc-nox=1.11.2-3 libattr1=1:2.4.47-2 athena-jot=9.0-5ubuntu1 libsysfs2=2.1.0+repack-4 openssl=1.0.2g-1ubuntu4.15 libp11-kit0=0.23.2-5~ubuntu16.04.1 python3.5-minimal=3.5.2-2ubuntu0~16.04.5 libapt-inst2.0=1.2.32ubuntu0.1 libestr0=0.1.10-1 login=1:4.2-3.1ubuntu5.3 lsof=4.89+dfsg-0.1 linux-base=4.5ubuntu1.2~16.04.1 libbsd0=0.8.2-1ubuntu0.1 libklibc=2.0.4-8ubuntu1.16.04.4 ltrace=0.7.3-5.1ubuntu4 libdevmapper1.02.1=2:1.02.110-1ubuntu10 libhx509-5-heimdal=1.7~git20150920+dfsg-4ubuntu1.16.04.1 libdbus-1-3=1.10.6-1ubuntu3.3 librtmp1=2.4+20151223.gitfa8646d-1ubuntu0.1 apt=1.2.29ubuntu0.1 ucf=3.0036 util-linux=2.27.1-6ubuntu3.6 wget=1.17.1-1ubuntu1.4 libhcrypto4-heimdal=1.7~git20150920+dfsg-4ubuntu1.16.04.1 libperl5.22=5.22.1-9ubuntu0.6 libnuma1=2.0.11-1ubuntu1.1 init-system-helpers=1.29ubuntu4 libc-bin=2.23-0ubuntu10 libbz2-1.0=1.0.6-8ubuntu0.2 pv=1.6.0-1 libpng12-0=1.2.54-1ubuntu1.1 grub-pc-bin=2.04-yandex1 libffi6=3.2.1-4 tzdata=2018g-0ubuntu0.16.04 language-pack-en-base=1:16.04+20160627 libaudit-common=1:2.4.5-1ubuntu2.1 liblz4-1=0.0~r131-2ubuntu2 initramfs-tools-bin=0.122ubuntu8.14 libustr-1.0-1=1.0.4-5 libsepol1=2.4-2 openssh-server=1:7.2p2-4ubuntu2.8 libsemanage1=2.3-1build3 libgmp10=2:6.1.0+dfsg-2 libgcrypt20=1.6.5-2ubuntu0.6 python3.5=3.5.2-2ubuntu0~16.04.5 hdparm=9.48+ds-1ubuntu0.1 debianutils=4.7 bsdutils=1:2.27.1-6ubuntu3.6 libheimntlm0-heimdal=1.7~git20150920+dfsg-4ubuntu1.16.04.1 libgssapi-krb5-2=1.13.2+dfsg-5ubuntu2.1 iputils-ping=3:20121221-5ubuntu2 dash=0.5.8-2.1ubuntu2 libcurl3-gnutls=7.47.0-1ubuntu2.12 libelf1=0.165-3ubuntu1.2 mount=2.27.1-6ubuntu3.6 libapparmor1=2.10.95-0ubuntu2.11 libgnutls-openssl27=3.4.10-4ubuntu1.8 initscripts=2.88dsf-59.3ubuntu2 libdb5.3=5.3.28-11ubuntu0.2 gnupg=1.4.20-1ubuntu3.3 libapt-pkg5.0=1.2.32ubuntu0.1 libasprintf0v5=0.19.7-2ubuntu3.1 procps=2:3.3.10-4ubuntu2.4 libtasn1-6=4.7-3ubuntu0.16.04.3 libltdl7=2.4.6-0.1 libjson-c2=0.11-4ubuntu2.6 tar=1.28-2.1ubuntu0.1 sudo=1.8.16-0ubuntu1.5 libwind0-heimdal=1.7~git20150920+dfsg-4ubuntu1.16.04.1 libfreetype6=2.6.1-0.1ubuntu2.5 apt-utils=1.2.29ubuntu0.1 libprocps4=2:3.3.10-4ubuntu2.5 file=1:5.25-2ubuntu1.1 base-files=9.4ubuntu4.7 curl=7.47.0-1ubuntu2.12 libkrb5support0=1.13.2+dfsg-5ubuntu2.1 initramfs-tools=0.122ubuntu8.14 lsscsi=0.27-3 libpython3.5-stdlib=3.5.2-2ubuntu0~16.04.5 libtinfo5=6.0+20160213-1ubuntu1 libasn1-8-heimdal=1.7~git20150920+dfsg-4ubuntu1.16.04.1 libfdisk1=2.27.1-6ubuntu3.10 freeipmi-common=1.4.11-1.1ubuntu4.1~0.16.04 libaudit1=1:2.4.5-1ubuntu2.1 openipmi=2.0.18-0ubuntu11.2 libgnutls30=3.4.10-4ubuntu1.8 libsnmp30=5.7.3+dfsg-1ubuntu4.6 ipmitool=1.8.16-3ubuntu0.2 libgdbm3=1.8.3-13.1 grub-pc=2.04-yandex1 libkmod2=22-1ubuntu5.2 openssh-sftp-server=1:7.2p2-4ubuntu2.8 libexpat1=2.1.0-7ubuntu0.16.04.5 libnettle6=3.2-1ubuntu0.16.04.1 time=1.7-25.1 libpython3.5-minimal=3.5.2-2ubuntu0~16.04.5 passwd=1:4.2-3.1ubuntu5.3 libsnmp-base=5.7.3+dfsg-1ubuntu4.2 pciutils=1:3.3.1-1.1ubuntu1.3 libcap-ng0=0.7.7-1 multiarch-support=2.23-0ubuntu10 libsasl2-2=2.1.26.dfsg1-14ubuntu0.2 libedac1=0.18-1 zlib1g=1:1.2.8.dfsg-2ubuntu4.3 libc6=2.23-0ubuntu11.2 libsensors4=1:3.4.0-2 ngrep=1.45.ds2-13 libhogweed4=3.2-1ubuntu0.16.04.1 libheimbase1-heimdal=1.7~git20150920+dfsg-4ubuntu1.16.04.1 perl-modules-5.22=5.22.1-9ubuntu0.6 ubuntu-keyring=2012.05.19.1 libstdc++6=5.4.0-6ubuntu1~16.04.12 rsyslog=8.16.0-1ubuntu3.1 e2fsprogs=1.44.1-1ubuntu1.3-yandex0 perl-base=5.22.1-9ubuntu0.6 locales=2.23-0ubuntu10 ifupdown=0.8.10ubuntu1.4 vim-tiny=2:7.4.1689-3ubuntu1.3 iproute2=4.3.0-1ubuntu3.16.04.5 libblkid1=2.27.1-6ubuntu3.10 libuuid1=2.27.1-6ubuntu3.10 libsasl2-modules-db=2.1.26.dfsg1-14ubuntu0.2
    apt-get -y autoremove
EOS
}

chroot_stop()
{
    local chroot=${1}

    # stop all other process in chroot
    for signal in 15 9 ; do
        for p in $(find /proc/*/root) ; do
            plink=$(readlink -f $p)
            if echo ${plink} | grep -q "${chroot}" 2> /dev/null ; then
                log_debug "$(basename $(dirname ${p})) would be kill by ${signal}"
                kill -${signal} $(basename $(dirname $p))
            fi
        done
        sleep 3
    done

    chroot ${chroot} /bin/sh <<EOS
        rm /etc/mtab
        rm -rf /root/sbin
        umount /proc /sys /run /dev/pts /dev
        find /var/log -type f -size +0 -exec truncate -s 0 {} +
        rm /var/lib/dbus/machine-id
EOS
}

add_dist_yandex_gpg()
{
    local chroot=${1}
    chroot ${chroot} /bin/sh <<EOS
       curl -sL 'http://keyserver.ubuntu.com/pks/lookup?op=get&search=0x7FCD11186050CD1A' | \
        apt-key add -
EOS
}

apply_settings()
{
    local root=${1}

    echo -n > "${root}/root/.bash_history"
    echo -n > "${root}/etc/hostname"
    echo 'localhost.localdomain' > "${root}/etc/mailname"
    sed -i "s/search\s.*/search yandex.ru/" "${root}/etc/resolv.conf"

    if [ -f "${root}/etc/postfix/main.cf" ] ; then
        sed -i 's/myhostname\s.*/myhostname = localhost/i;s/mydestination\s.*/mydestination = localhost, localhost.localdomain/i' \
            "${root}/etc/postfix/main.cf"
    fi

    if [ -f "${root}/etc/mdadm/mdadm.conf" ] ; then
        sed -i "/^ARRAY.*/d" "${root}/etc/mdadm/mdadm.conf"
    fi

    if [ -f "${root}/etc/initramfs-tools/conf.d/mdadm" ] ; then
        sed -i 's/BOOT_DEGRADED=false/BOOT_DEGRADED=true/i' \
         "${root}/etc/initramfs-tools/conf.d/mdadm"
    fi

    if [ -f "${root}/etc/default/rcS" ] ; then
        sed -i '/\s*FSCKFIX/d' "${root}/etc/default/rcS"
        echo 'FSCKFIX=yes' >> "${root}/etc/default/rcS"
    fi

    local grub_config="${root}/etc/default/grub"
    if [ -f "${root}/etc/default/grub" ] ; then
        #remove quiet and splash from kernel boot options
        sed -i '/GRUB_CMDLINE_LINUX/s/\(quiet\|splash\)//g' "${grub_config}"

        #add kernel boot options for legacy named network devices (like 'ethX')
        sed -i 's/^\(\s*GRUB_CMDLINE_LINUX\(_DEFAULT\)\?="\)/\1biosdevname=0 net.ifnames=0 /g' "${grub_config}"

        # disable netplan
        # sed -i 's@^\(\s*GRUB_CMDLINE_LINUX\(_DEFAULT\)\?="\)@\1netcfg/do_not_use_netplan=true @g' "${grub_config}"
    fi

    echo -e "# RNDSVC-357: fix building of raid0 with different size components\noptions raid0 default_layout=2" > "${root}/etc/modprobe.d/raid0.conf" || \
        log_error "Failed to create raid0 layout file in /etc/modprobe.d"
}

mk_script_tun_intel_network_card()
{
    local root=${1}
    local scriptpath="${root}/etc/network/if-pre-up.d/intel-card-tun"

    cat > "${scriptpath}" <<EOT
#!/bin/sh

ETHTOOL=\$(which ethtool) || exit 0
DEFAULTS=/etc/default/intel-card-tun
test -f \${DEFAULTS} && . \${DEFAULTS}

TXRX=\${TXRX:-1024}

\$ETHTOOL -i \${IFACE} 2> /dev/null | grep -iqP 'driver:\s*(ix?gb|e1000)' || exit 0
\$ETHTOOL -g \${IFACE} | sed '0,/^Current/d' | \\
  grep '^[TR]X:' | sed 's/://' | tr '[:upper:]' '[:lower:]' | \\
    while read k v ; do
       [ \$v -lt \$TXRX ] &&  \$ETHTOOL -G \${IFACE} \$k \$TXRX > /dev/null 2>&1
    done

exit 0
EOT

    chmod a+x "${scriptpath}"
}

mk_script_eth_tso_off()
{
    local root=${1}
    local scriptpath="${root}/etc/network/if-pre-up.d/eth-tso-off"

    cat > "${scriptpath}" <<EOT
#!/bin/sh

ETHTOOL=\$(which ethtool) || exit 0

if [ "\$(cat /sys/class/net/\${IFACE}/type 2> /dev/null)" != "1" ] ; then
    exit 0
fi

if \$ETHTOOL -k \${IFACE} 2> /dev/null | grep 'tcp-segmentation-offload' \\
	| grep -qw 'on' ; then
    \$ETHTOOL -K \${IFACE} tso off
fi

exit 0
EOT

    chmod a+x "${scriptpath}"
}

pack_image()
{
    local path=${1}
    local name=${2}

    tar --numeric-owner -cjf "${name}" -C "${path}" \
        --exclude='./dev/*' --exclude='./run/*' .
}

chroot_cleanup()
{
    local path=${1}
    chroot ${path} /bin/sh <<EOS
    export DEBIAN_FORNTEND=noninteractive
    export PATH="/root/sbin:$PATH"
    apt-get -y purge ureadahead netcat-openbsd libxtables11 isc-dhcp-client isc-dhcp-common dh-python net-tools
    apt-get -y autoremove
EOS
    rm -fv "$path/var/cache/apt"/*.bin
    rm -fv "$path/etc/machine-id"
    find "$path/var/lib/apt/lists"  -type f -delete
}

spawn_shell()
{
    local path="$1"
    chroot "$path" /bin/bash -c "export PATH=/root/sbin:$PATH; exec bash"
}

create_image()
{
    local output

    local image_tar="$(dirname ${CHROOT_PATH})/$IMAGE_NAME.tar.bz2"
    
    log_info "Bootstraping image ${DISTRIB} (${ARCH}) to ${CHROOT_PATH}"
    output=$(bootstrap_image "${CHROOT_PATH}" 2>&1) || {
        log_error "$output"
        return 4
    }
    log_debug "${output}"

    log_info "Preparing ${CHROOT_PATH} for update"
    output=$(chroot_prepare "${CHROOT_PATH}" 2>&1) || {
        log_error "failed mount fs in chroot. Output:\n$output"
        return 5
    }
    log_debug "$output"

    log_info "Updating chrooted system"
    output=$(chroot_update "${CHROOT_PATH}" 2>&1) || {
        log_error "failed update system. Output:\n$output"
        return 6
    }
    log_debug "$output"

    log_info "Add dist.yandex.ru repo key"
    add_dist_yandex_gpg "${CHROOT_PATH}"

    log_info "Cleaning up future image"
    output=$(chroot_cleanup "${CHROOT_PATH}" 2>&1) || {
        log_error "failed to cleanup. Output:\n$output"
        return 9
    }
    log_debug "${output}"

    if [ ${INTERACTIVE} = "YES" ]; then
        log_info "Spawning interactive shell"
        spawn_shell "${CHROOT_PATH}"
    fi

    log_info "Stoping ${CHROOT_PATH}"
    output=$(chroot_stop "${CHROOT_PATH}" 2>&1) || {
        log_error "failed unmount fs in chroot. Output:\n$output"
        return 7
    }
    log_debug "$output"

    log_info "Applying default settings"
    apply_settings "${CHROOT_PATH}" 2>&1

    log_info "Making script to tun tx/rx intel card"
    mk_script_tun_intel_network_card "${CHROOT_PATH}"

    log_info "Making script to disable tcp segments offload"
    mk_script_eth_tso_off "${CHROOT_PATH}"

    log_info "Packing image to ${image_tar}"
    output=$(pack_image "${CHROOT_PATH}" "${image_tar}" 2>&1) || {
        log_error "Error pack image to ${image_tar}. Output:\n"
        return 8
    }
    log_debug "${output}"

    if [ ${KEEP_CHROOT} = "YES" ]; then
        log_info "*NOT* Removing ${CHROOT_PATH}"
    else
        log_info "Removing ${CHROOT_PATH}"
        rm -rf "${CHROOT_PATH}"
    fi

    return 0
}

upload()
{
    local arc="$1"
    local image_name="$(basename $arc)"
    local image_path="$(dirname $arc)"
    pushd "$image_path"
    md5sum "$image_name" > "$image_name.md5"
    s3cmd --no-check-certificate put "$image_name" "s3://setup-rtc-images/$image_name"
    s3cmd --no-check-certificate put "$image_name.md5" "s3://setup-rtc-images/$image_name.md5"
    rm -fv "$image_name.md5"
    popd
}

usage()
{
    echo "$(basename $0) [FLAGS]"
    echo ""
    echo "-h|--help - this message"
    echo "-k|--keep - keep chroot after build"
    echo "-d|--debug - show debugging output"
    echo "-n|--name <name> - image name (will produce <name>.tar.bz2)"
    echo "-s|--stop <path> - stop chroot at <path>"
    echo "-i|--interactive - launch intercative shell in chroot after build"
    echo "--chroot <path> - start intercative chroot shell at <path> (also starts chroot)"
    echo "--upload <archive> - upload archive to s3://setup-rtc-images/<archive>"
}

main()
{
    if [ "$(whoami)" != "root" ]; then
        echo "only root can run $(basename $0). You are not root."
        return 1
    fi

    while (( "$#" )); do
        case "$1" in
            -k|--keep)
            KEEP_CHROOT="YES"
            shift
            ;;

            -d|--debug)
            DEBUG="YES"
            shift
            ;;

            -h|--help)
            usage
            exit 0
            ;;

            -i|--interactive)
            INTERACTIVE="YES"
            ;;

            -n|--name)
            if [ -n "$2" ] && [ ${2:0:1} != "-" ]; then
                IMAGE_NAME=$2
                CHROOT_PATH="$(pwd)/$IMAGE_NAME-$(date +%Y%m%d)"
                shift 2
            else
                log_error "Argument for $1 is missing"
                exit 1
            fi
            ;;

            -s|--stop)
            if [ -n "$2" ] && [ ${2:0:1} != "-" ]; then
                chroot_stop "$2"
                exit $?
            else
                log_error "Argument for $1 is missing"
                exit 1
            fi
            ;;

            --chroot)
            if [ -n "$2" ] && [ ${2:0:1} != "-" ]; then
                chroot_prepare "$2"
                spawn_shell "$2"
                chroot_stop "$2"
                exit $?
            else
                log_error "Argument for $1 is missing"
                exit 1
            fi
            ;;

        --upload)
            if [ -n "$2" ] && [ ${2:0:1} != "-" ]; then
                log_info "Uploading $2 to S3"
                upload $2
                exit $?
            else
                log_error "Argument for $1 is missing"
                exit 1
            fi
            ;;

            

            -*|--*=) # unsupported flags
            log_error "unsupported flags found"
            exit 1
            ;;
        esac
    done

    create_image
    exit $?
}

main "$@"
