import requests
import json
import argparse
import logging
import sys

log = logging.getLogger(__name__)
RESOLVER_URL = 'https://rtc-instance-resolver.in.yandex-team.ru/v1/switch_instances'


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('-s', '--switches', nargs='*', default=[], help="Switch list", required=True)
    parser.add_argument('-c', '--cookie', required=True, help="Cookie value to access instance resolver")
    return parser.parse_args()


def resolve_switch(name, cookie):
    try:
        r = requests.get(f'{RESOLVER_URL}/{name}', headers={'Cookie': cookie})
        j = r.json()
    except Exception as e:
        return None, f'failed to resolve switch {name}: {e}'
    groups = set()
    services_to_get = set()
    if 'Instances' not in j:
        return None, f'failed to resolve switch {name}: no instances found'
    for i in j['Instances']:
        if 'NannyServiceId' in i and 'GencfgGroup' in i:
            groups.add(i['GencfgGroup'])
            services_to_get.add(i['NannyServiceId'])
    resps = set()
    for s in j['ResponsibleGroups']:
        for k in services_to_get:
            if 'NannyServiceId' in s and s['NannyServiceId'] == k:
                resps.update(s['Logins'])
    return {'groups': sorted(list(groups)), 'responsibles': sorted(list(resps))}, None


def main():
    args = parse_args()
    sws = {}
    for sw in args.switches:
        r, err = resolve_switch(sw, args.cookie)
        if err is not None:
            log.error(f'Switch resolve failed: {err}')
            continue
        sws[sw] = r
    json.dump(sws, sys.stdout)
    sys.stdout.write('\n')


if __name__ == '__main__':
    main()
