import logging
import sys

from dataclasses import dataclass

from walle_api.client import WalleClient
from walle_api.constants import version as sdk_version, PRODUCTION_URL


DEFAULT_HOSTS_LIMIT = 100


@dataclass
class SwitchFlags:
    yp_flag: bool = False
    yt_flag: bool = False
    gencfg_flag: bool = False
    cohabitation_flag: bool = False
    rtc: bool = False
    non_rtc: bool = False


class SwitchDefiner:
    # simple switch definer class, require wall-e cli
    def __init__(self):
        # parse args
        self.walle_client = WalleClient(name="Sre.Hodoc", url=PRODUCTION_URL)

        # logging
        logging.basicConfig(
            level=logging.INFO,
            format="%(asctime)s [%(levelname)s] %(message)s",
            handlers=[
                logging.FileHandler("sw_resolver.log"),
                logging.StreamHandler()
            ]
        )

    def list_switch_projects(self, switch: str, limit=DEFAULT_HOSTS_LIMIT):
        options = dict(tags='rtc', switch=switch, fields=['project'])
        return [h['project'] for h in self.walle_client.iter_hosts(limit=limit, **options)]

    def _is_gencfg_cohabitation_switch(self, switch: str, limit=DEFAULT_HOSTS_LIMIT):
        options = dict(tags=['rtc', 'rtc.reboot_segment-gencfg', 'rtc.cohabitation-enabled'], switch=switch,
                       fields=['project'])
        _coh = [coh['project'] for coh in self.walle_client.iter_hosts(limit=limit, **options)]
        return bool(_coh)

    def _is_true_gencfg_switch(self, switch: str, limit=DEFAULT_HOSTS_LIMIT):
        coh_options = dict(tags=['rtc', 'rtc.reboot_segment-gencfg', 'rtc.cohabitation-enabled'], switch=switch,
                           fields=['project'])
        gen_options = dict(tags=['rtc', 'rtc.reboot_segment-gencfg'], switch=switch, fields=['project'])
        _coh = [coh['project'] for coh in self.walle_client.iter_hosts(limit=limit, **coh_options)]
        _gencfg = [h['project'] for h in self.walle_client.iter_hosts(limit=limit, **gen_options)]
        return bool([prj for prj in _gencfg if prj not in _coh])

    def _get_switch_flags(self, switch: str):
        _flagset = SwitchFlags()
        rtc_projects = set(self.list_switch_projects(switch))
        if not rtc_projects:
            return _flagset
        _flagset.gencfg_flag = self._is_true_gencfg_switch(switch)
        _flagset.cohabitation_flag = self._is_gencfg_cohabitation_switch(switch)
        if any(['yt-' in p for p in rtc_projects]):
            _flagset.yt_flag = True
        if any(['yp-' in p for p in rtc_projects if 'yt-' not in p]):
            _flagset.yp_flag = True
        return _flagset


if __name__ == "__main__":
    s = SwitchDefiner()

    _switches = []
    if len(sys.argv) > 1:
        _switches = sys.argv[1:]
    else:
        for line in sys.stdin:
            _switches.append(line.strip())
    logging.info(f"Observing {_switches}")

    result_by_cms_type = {'only_yt': [],
                          'only_yp': [],
                          'yp_yt': [],
                          'only_gencfg_coh': [],
                          'only_true_gencfg': [],
                          'yp_gencfg_coh': [],
                          'yt_gencfg_coh': [],
                          'yt_yp_gencfg_coh': [],
                          'yt_true_gencfg': [],
                          'yp_true_gencfg': [],
                          'yt_yp_true_gencfg': [],
                          'others': []
                          }

    for _ss in _switches:
        flagset = s._get_switch_flags(_ss)

        if flagset.yt_flag and not flagset.yp_flag and not flagset.gencfg_flag and not flagset.cohabitation_flag:
            result_by_cms_type['only_yt'].append(_ss)
        elif not flagset.yt_flag and flagset.yp_flag and not flagset.gencfg_flag and not flagset.cohabitation_flag:
            result_by_cms_type['only_yp'].append(_ss)
        elif flagset.yt_flag and flagset.yp_flag and not flagset.gencfg_flag and not flagset.cohabitation_flag:
            result_by_cms_type['yp_yt'].append(_ss)
        elif not flagset.yt_flag and not flagset.yp_flag and flagset.gencfg_flag and flagset.cohabitation_flag:
            result_by_cms_type['only_gencfg_coh'].append(_ss)
        elif not flagset.yt_flag and not flagset.yp_flag and flagset.gencfg_flag and not flagset.cohabitation_flag:
            result_by_cms_type['only_true_gencfg'].append(_ss)
        elif flagset.yt_flag and not flagset.yp_flag and not flagset.gencfg_flag and flagset.cohabitation_flag:
            result_by_cms_type['yt_gencfg_coh'].append(_ss)
        elif not flagset.yt_flag and flagset.yp_flag and not flagset.gencfg_flag and flagset.cohabitation_flag:
            result_by_cms_type['yp_gencfg_coh'].append(_ss)
        elif flagset.yt_flag and flagset.yp_flag and not flagset.gencfg_flag and flagset.cohabitation_flag:
            result_by_cms_type['yt_yp_gencfg_coh'].append(_ss)
        elif flagset.yt_flag and not flagset.yp_flag and flagset.gencfg_flag and flagset.cohabitation_flag:
            result_by_cms_type['yt_true_gencfg'].append(_ss)
        elif not flagset.yt_flag and flagset.yp_flag and flagset.gencfg_flag and flagset.cohabitation_flag:
            result_by_cms_type['yp_true_gencfg'].append(_ss)
        elif flagset.yt_flag and flagset.yp_flag and flagset.gencfg_flag and flagset.cohabitation_flag:
            result_by_cms_type['yt_yp_true_gencfg'].append(_ss)
        elif not flagset.yp_flag and not flagset.yt_flag and not flagset.cohabitation_flag and not flagset.gencfg_flag:
            result_by_cms_type['others'].append(_ss)

    for k, v in result_by_cms_type.items():
        logging.info(f'category: {k} switches: {v}')
