# -*- coding: utf-8 -*-

from dateutil import parser as dateparser
from dataclasses import dataclass

from infra.rtc.srebot.utils.abc_client import AbcClient
from infra.rtc.srebot.utils.calendar_intapi import get_num_workdays
from infra.rtc.srebot.utils.staff_helper import get_duty_telegram
from infra.rtc.srebot.utils.st_helper import ST_URL, STQuery, short_url
from infra.rtc.srebot.utils.telegram_api import telegram_simple as tg
from infra.rtc.srebot.utils.walle_helper import WalleAbstract
from infra.rtc.srebot.utils.solomon_client import SolomonClient


import argparse
import datetime
import json
import logging
import os
import startrek_client


ST_DATETIME_DELIMITER = "T"
ROUTINE_DUTY_SLUG = "routineoperations"
DEADLINE_DELTA = 14
SILENCE_DAYS = 2
DEAD_WALLE_URL = 'https://wall-e.yandex-team.ru/projects/hosts?fqdn=%23rtc&status=dead'
SECONDTIME_WALLE_URL = 'https://wall-e.yandex-team.ru/projects/hosts?fqdn=%23rtc&status=report-2nd-time-node'
DEAD_GPU_WALLE_URL = 'https://wall-e.yandex-team.ru/projects/hosts?fqdn=%23rtc.gpu-nvidia&status=dead'
A100_40G_RESERV_SOLOMON_URL = 'https://solomon.yandex-team.ru/?project=yt&cluster=hahn&graph=yt-reserve-gpu&l.tree=gpu_tesla_a100&b=1w&e='
A100_80G_RESERV_SOLOMON_URL = 'https://solomon.yandex-team.ru/?project=yt&cluster=hahn&graph=yt-reserve-gpu&l.tree=gpu_tesla_a100_80g&b=1w&e='



@dataclass
class STStage:
    id: str
    checking_tag: str
    expected_tag: str
    deadline_days: int
    who_pinged: list
    message: str


class SreBotPinger:

    def __init__(self):

        # init logger
        self.logger = logging.getLogger()
        self.logger.setLevel(logging.INFO)
        logger_stream_handler = logging.StreamHandler()
        self.logger.addHandler(logger_stream_handler)

        # init configs and args
        self.args = SreBotPinger.get_args()
        self.logger.info(self.args)

        # init telegram client if enabled
        self.tg = None
        self.duty_tg = ''
        if self.args.tg_notification:
            tg_token=os.getenv('TG_TOKEN', None)
            tg_chat_id=os.getenv('TG_CHATID', None)
            if not tg_token or not tg_chat_id:
                raise EnvironmentError(
                    f"Telegram token: {tg_token if None else '****'} or chatid: {tg_chat_id} vars not set in env")
            self.tg = tg(tg_token, tg_chat_id)


        # init oauth
        self.oauth_token = os.getenv('OAUTH', None)
        self.ua = os.getenv('UA', None)
        if not self.oauth_token or not self.ua:
            raise EnvironmentError(
                f"Oauth token: {self.oauth_token if None else '***'} or user_agent: {self.ua} vars not set in env.")

        # init st client
        self.st_client = startrek_client.Startrek(useragent=self.ua, token=self.oauth_token)

        # init abc client
        self.abc_client = AbcClient(useragent=self.ua, oauth_token=self.oauth_token)

        # get current duty
        self.duty = self.get_sre_duty().get("login")
        if not self.duty:
            raise EnvironmentError("duty not set in SRE duty service")

        # get duty telegram if tg flag set
        if self.tg:
            self.duty_tg = get_duty_telegram(self.duty, self.oauth_token)

        # init walle_cli
        self.walle_cli = WalleAbstract(self.oauth_token)

        # init solomon client
        self.solomon_client = SolomonClient(self.oauth_token)

        # init config
        self.proccessing_config = SreBotPinger.get_config(self.args.config)
        self.logger.info(self.proccessing_config)

        # check dry run
        self.dry_run = self.args.dry_run
        if self.dry_run:
            self.logger.info('Dry run mode enabled. All write operations for stratrek will be replaced by logging')

        # stages init
        self.stage_approve = STStage("approve_check", "scenario_created", "scenario_approved", 2, ["sre", "author"],
                                     "согласования")
        self.stage_prepare = STStage("prepare_check", "scenario_approved", "scenario_processing", 4, ["sre"],
                                     "вывода в maintenance")
        self.stage_process = STStage("process_check", "scenario_processing", "scenario_finished", 3, ["sre"],
                                     "процессинга хостов")
        self.st_stages = (self.stage_approve, self.stage_prepare, self.stage_process)

    @staticmethod
    def get_args():
        """
        :return: rtc sre bot binary cmdline arguments
        """
        parser = argparse.ArgumentParser()
        parser.add_argument("-c", "--config", required=True, help='Path to config in json format.')
        parser.add_argument("-t", "--startrek-period", required=True, help='Startrek retrospective period monthes.')
        parser.add_argument("-p", "--startrek-queue", required=True, help='Startrek queue for proccessing')
        parser.add_argument("-tg", "--tg-notification", required=False, action='store_true',
                            help='Enable telegram notification.')
        parser.add_argument("-dry", "--dry-run", required=False, action='store_true',
                            help='Dry run mode for startrek. Only read operations will be proccesed')
        args = parser.parse_args()
        return args

    @staticmethod
    def get_config(config_path):
        """
        :param config_path: json queues config for rtc sre bot
        :return: check config state
        """
        if not os.path.isfile(config_path):
            raise FileNotFoundError(f"File {config_path}")

        with open(config_path) as config_file:
            try:
                return json.load(config_file)
            except Exception as e:
                raise ValueError(e)

    def get_sre_duty(self):
        service_id = self.abc_client.service_slug_to_id(ROUTINE_DUTY_SLUG)
        return self.abc_client.get_duty(service_id)

    def get_proccessing_queues_ids(self):
        """
        :return: list of queue's ids
        """
        ids = [section['id'] for section in self.proccessing_config]
        self.logger.info(ids)
        if self.args.startrek_queue == 'all':
            return ids
        elif self.args.startrek_queue in ids:
            return [self.args.startrek_queue]
        else:
            raise ValueError('Get incorrect config. No needed queues parsed')

    def get_proccessing_configs(self, queue: str):
        """
        :param queue:
        :return: queue config from config.json
        """
        queue_conf = None
        for section in self.proccessing_config:
            if section['id'] != queue:
                continue
            queue_conf = section
        if not queue_conf:
            raise ValueError("Can't proccess empty config")
        return queue_conf

    def st_set_forgotten_deadline(self, queue: str):
        """
        :param queue: startrek queue"sre and author"
        :return:
        """
        # queue configuration
        queue_conf = self.get_proccessing_configs(queue)

        # checking ignore for queue
        if queue_conf.get("processing_ignore") == "deadline":
            return

        # processing queue
        query = STQuery(queue=queue, tags=queue_conf.get("tags"), extra_filter='Deadline: empty()').build_query()
        issues = self.st_client.issues.find(query)
        self.logger.info(f"Finded  {len(issues)} issues without deadline in query: {queue}")
        for issue in issues:
            self.logger.info(f'Key: {issue.key}')
            self.logger.info(f'Created at: {issue.createdAt}')
            deadline = dateparser.parse(issue.createdAt) + datetime.timedelta(days=DEADLINE_DELTA)
            if self.dry_run:
                self.logger.info(f"Dry run. Dealine in {issue.key} should be set to {deadline}")
            else:
                issue.update(deadline=datetime.datetime.strftime(deadline, "%Y-%m-%d"), params={'notify': False})

    def st_tag_added_date(self, st_ticket_key: str, st_tag: str):
        """
        :param st_ticket_key:
        :param st_tag:
        :return: date when ticket got tag
        """
        temp = []
        issue = self.st_client.issues[st_ticket_key]

        # check weither not a changelog existed
        if not issue or not issue.changelog:
            return None

        # processing changelog
        for history_point in issue.changelog:
            for st_field in history_point.fields:
                if not st_field['field']:
                    return
                if st_field['field'].id == "tags":
                    if st_field['to'] is not None and st_tag in st_field['to']:
                        temp.append(history_point.updatedAt)
                        continue

        # processing results, looking for earlier date
        if not temp:
            return
        temp_date = min(temp)
        raw_date = temp_date.split(ST_DATETIME_DELIMITER)
        if len(raw_date) != 2:
            return None
        logging.debug(raw_date)
        return raw_date[0]

    def st_notify_long_stage(self, queue: str, stage: STStage):
        """
        :param queue: startek queue
        :param stage: scenario stage
        :return:
        """

        # stage configuration
        checking_tag = stage.checking_tag
        expected_tag = stage.expected_tag
        deadline_days = stage.deadline_days
        who_pinged = stage.who_pinged
        message = stage.message

        if any((checking_tag, expected_tag, deadline_days, who_pinged)) is None:
            raise ValueError(f"Stage {stage} is incorrect for queue {queue}")

        # queue configuration
        queue_conf = self.get_proccessing_configs(queue)

        # checking ignore for queue
        if queue_conf.get("processing_ignore") == "tags":
            return

        # set tags for query
        queue_tags = queue_conf.get("tags")
        if not queue_tags:
            raise ValueError("Something going wrong processing startrek queue without tags in config")
        queue_tags.append(checking_tag)

        # processing queue
        query = STQuery(queue=queue, tags=queue_conf.get("tags")).build_query()
        issues = self.st_client.issues.find(query)
        summonees = [self.duty]

        for issue in issues:
            self.logger.info(issue.key)
            if stage.id != "process_check":
                if expected_tag in issue.tags:
                    continue
            from_date = self.st_tag_added_date(issue.key, checking_tag)
            if not from_date:
                continue

            # calculating working days between last stage and current moment
            parsed_from_date_raw = dateparser.parse(from_date)
            parsed_from_date = datetime.datetime.strftime(parsed_from_date_raw, "%Y-%m-%d")
            today = datetime.datetime.strftime(datetime.datetime.now(), "%Y-%m-%d")
            time_diff = get_num_workdays(parsed_from_date, today)

            if time_diff > deadline_days:
                self.logger.info(f'Founded hanged ticket {issue.key} on stage {stage.id} '
                                 f'within {time_diff} working days over deadline')

                # ping issues in ST
                if "author" in who_pinged:
                    summonees.append(issue.createdBy.id)

                ISSUE_URL = f'{ST_URL}/{issue.key}'
                formated_message = f'Превышен таймер SLA в {deadline_days} дней для стадии {message} сценария.' \
                                   f'Обратите внимание на тикет в кратчайшие сроки'
                telegram_message = f'@{bot.duty_tg} : Превышен таймер SLA в {deadline_days} дней для стадии {message} сценария ' \
                                   f'<a href="{short_url(ISSUE_URL)}">{issue.key}</a>.' \
                                   f' Обратите внимание на тикет в кратчайшие сроки'
                if self.dry_run:
                    self.logger.info(f'Dry run. Should ping {issue} due hanged on stage {stage.id} within {time_diff}'
                                     f' working days')
                    self.logger.info(formated_message)
                else:
                    # assign from bot to current duty
                    if not issue.assignee or issue.assignee.id == "robot-rtc-autoadmin":
                        issue.update(assignee = self.duty)

                    # checking last comment date and posting only if inactivity period passed
                    if STQuery.get_last_comment_date(issue) > SILENCE_DAYS:
                        issue.comments.create(text=formated_message, summonees=summonees, params={'notify': False})

                # telegram notification
                if self.tg and not self.dry_run:
                    self.tg.sendMessage(telegram_message)

def make_pretty_hosts_ending(n):
    last_num = n % 10
    if last_num == 1:
        return ''
    elif last_num in (2,3,4):
        return 'а'
    else:
        return 'ов'

if __name__ == "__main__":

    bot = SreBotPinger()

    for st_id in bot.get_proccessing_queues_ids():
        bot.st_set_forgotten_deadline(st_id)
        for stage in bot.st_stages:
            bot.st_notify_long_stage(st_id, stage)

    dead_all = bot.walle_cli.list_dead_rtc()
    dead_2nd_time = bot.walle_cli.list_2ndtime_rtc()
    dead_gpu = bot.walle_cli.list_dead_gpu_rtc()
    gpu_tesla_a100_reserv = bot.solomon_client.get_a100_reserve_size()
    gpu_tesla_a100_80g_reserv = bot.solomon_client.get_a100_80g_reserve_size()

    if bot.dry_run:
        bot.logger.info(f'BURNe: total {dead_all}, 2nd-time {dead_2nd_time}, dead gpu {dead_gpu} hosts. '\
                        f'YT_RESERV: a100-tree {gpu_tesla_a100_reserv}, a100_80g-tree {gpu_tesla_a100_80g_reserv}')

    if bot.tg:
        bot.tg.sendMessage(f'@{bot.duty_tg} Статистика BURNe: всего <a href="{DEAD_WALLE_URL}">{dead_all} хост{make_pretty_hosts_ending(dead_all)}</a>, '
                           f'2nd time node <a href="{SECONDTIME_WALLE_URL}">{dead_2nd_time} хост{make_pretty_hosts_ending(dead_2nd_time)}</a>, '
                           f'dead gpu <a href="{DEAD_GPU_WALLE_URL}">{dead_gpu} хост{make_pretty_hosts_ending(dead_gpu)}</a>.\n'
                           f'Текущий резерв деревьев: <a href="{A100_40G_RESERV_SOLOMON_URL}">gpu_tesla_a100</a> - {gpu_tesla_a100_reserv}, '
                           f'<a href="{A100_80G_RESERV_SOLOMON_URL}">gpu_tesla_a100_80g</a> - {gpu_tesla_a100_80g_reserv}')
