# -*- coding: utf-8 -*-

from dateutil import parser as dateparser

import datetime
import requests

ST_URL = "https://st.yandex-team.ru"


def short_url(url_to_short):
    """
    :return shortened url (uses nda.ya.ru)
    """
    return requests.get("https://nda.ya.ru/--?url={}".format(url_to_short)).text


class STQuery(object):

    @staticmethod
    def query_url(query):
        """
        :param query:
        :return: st filter query
        """
        return "{}/filters/filter?query={}".format(ST_URL, query)

    def __init__(self,
        queue,
        sort=True,
        created='2020-09-01',
        not_closed=True,
        updated=None,
        tags = None,
        extra_filter=None
    ):
        if not queue or not isinstance(queue, str):
            raise ValueError('Unsafe startrek query object with queue: None')
        self.queue = queue
        self.sort = sort
        self.created = created
        self.not_closed = not_closed
        self.extra_filter = extra_filter
        self.updated = updated
        self.tags = tags

    def build_query(self):
        """
        :return: complete query string for startrek
        """
        q = list()

        q.append('Queue: {}'.format(self.queue))
        if self.not_closed:
            q.append('AND "Resolution": empty()')
        if self.extra_filter:
            q.append('AND {}'.format(self.extra_filter))
        if self.tags:
            tags_with_clue = ['"{}"'.format(tag) for tag in self.tags]
            q.append("AND Tags: {}".format(','.join(tags_with_clue)))
        if self.sort:
            q.append('"Sort By": Key DESC AND')
        if self.created:
            q.append('Created: >= {}'.format(self.created))
        if self.updated:
            q.append('Updated: {}'.format(self.updated))
        return ' '.join(q)

    def build_short_query_url(self):
        """
        :return: short nda url for st query filter links
        """
        return short_url(self.query_url(self.build_query()))

    @staticmethod
    def get_last_comment_date(issue):
        """
        :param issue: st issue object
        :return: date of last comment in ticket
        """
        last_comment_activity_date = dateparser.parse(issue.createdAt).replace(tzinfo=None)
        for comment in issue.comments:
            comment_date = dateparser.parse(comment.createdAt).replace(tzinfo=None)
            if comment_date > last_comment_activity_date:
                last_comment_activity_date = comment_date
        return (datetime.datetime.now() - last_comment_activity_date).days
