import argparse
import logging
import yaml

log = logging.getLogger('pkg_tool')
NOTICE = '# Do not edit!!! Please use https://a.yandex-team.ru/arc/trunk/arcadia/infra/rtc/upstream/pkg_tool for updates\n'



def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('-x', '--input-xenial', type=argparse.FileType('r'), help='input-xenial: file with xenial versions', required=True)
    parser.add_argument('-f', '--input-focal', type=argparse.FileType('r'), help='input-focal: file with focal versions', required=True)
    parser.add_argument('-a', '--input-focal-arm', type=argparse.FileType('r'), help='input-focal-arm: file with focal versions for aarch64', required=True)
    parser.add_argument('-o', '--output', type=argparse.FileType('w'), help='output: file with hostctl unit',
                        default='upstream-packages.yaml')
    parser.add_argument('-v', '--version', type=str, help='version: version of hostctl unit', required=True)
    return parser.parse_args()


def main(argv):
    logging.basicConfig(level=logging.DEBUG, format='%(asctime)s - %(name)s - %(levelname)s - %(message)s')
    xenial = yaml.safe_load(argv.input_xenial)
    log.info('Loaded {} xenial packages'.format(len(xenial)))
    focal = yaml.safe_load(argv.input_focal)
    log.info('Loaded {} focal packages'.format(len(focal)))
    focal_arm = yaml.safe_load(argv.input_focal_arm)
    log.info('Loaded {} focal-arm packages'.format(len(focal_arm)))
    all_pkgs = {p for p in list(xenial.keys()) + list(focal.keys()) + list(focal_arm.keys())}
    hostctl_pkgs = []
    selectors = []
    for name in all_pkgs:
        focal_ver = '<skip>' if name not in focal else focal[name]
        focal_arm_ver = '<skip>' if name not in focal_arm else focal_arm[name]
        xenial_ver = '<skip>' if name not in xenial else xenial[name]
        ver = '{}-v'.format(name.replace('.','d').replace('+', 'p'))
        selectors.append({
            'name': ver,
            'match': [
                {'exp': "focal() && arm64()", 'val': focal_arm_ver},
                {'exp': "focal()", 'val': focal_ver},
                {'exp': "xenial()", 'val': xenial_ver},
                {'exp': "default()", 'val': 'non-existing-version'},
            ],
        })
        hostctl_pkgs.append({'name': name, 'version': '{{{}}}'.format(ver)})
    selectors.sort(key=lambda s: s['name'])
    hostctl_pkgs.sort(key=lambda p: p['name'])
    unit_spec = {
        'meta': {
            'kind': 'PackageSet',
            'version': argv.version,
            'name': 'upstream-packages',
            'annotations': {'stage': 'stable'}
        },
        'spec': {'packages': hostctl_pkgs}
    }
    unit_vars = {'vars': selectors}
    unit = [unit_vars, unit_spec]
    argv.output.write(NOTICE)
    yaml.dump_all(unit, argv.output, default_flow_style=False)


if __name__ == '__main__':
    argv = parse_args()
    try:
        main(argv)
    except Exception as e:
        log.exception("Main failed: {}".format(e))
    log.info('Done.')
