import argparse
import collections
import logging
import apt
import os
import yaml
import shutil
import subprocess
import sys


log = logging.getLogger('apt')
Install = collections.namedtuple('Install', 'pkg,ver,aptpkg')

SOURCES_LIST_TEMPLATE_AMD = '''\
# Ubuntu repos
deb http://search-upstream-{distro}.dist.yandex.ru/search-upstream-{distro} unstable/amd64/
deb http://search-upstream-{distro}.dist.yandex.ru/search-upstream-{distro} unstable/all/
deb http://search-upstream-{distro}.dist.yandex.ru/search-upstream-{distro} stable/amd64/
deb http://search-upstream-{distro}.dist.yandex.ru/search-upstream-{distro} stable/all/

# These are Yandex repositories for Ubuntu ({distro}).
deb http://search-{distro}.dist.yandex.ru/search-{distro} stable/all/
deb http://search-{distro}.dist.yandex.ru/search-{distro} unstable/all/

deb http://search-{distro}.dist.yandex.ru/search-{distro} stable/amd64/
deb http://search-{distro}.dist.yandex.ru/search-{distro} unstable/amd64/

# These are Yandex repositories for Ubuntu ({distro}).
deb http://yandex-{distro}.dist.yandex.ru/yandex-{distro} stable/all/
deb http://yandex-{distro}.dist.yandex.ru/yandex-{distro} stable/amd64/
'''

SOURCES_LIST_TEMPLATE_ARM = '''\
# Ubuntu repos
deb http://search-upstream-focal-arm64.dist.yandex.ru/search-upstream-focal-arm64 unstable/arm64/
deb http://search-upstream-focal-arm64.dist.yandex.ru/search-upstream-focal-arm64 unstable/all/
deb http://search-upstream-focal-arm64.dist.yandex.ru/search-upstream-focal-arm64 stable/arm64/
deb http://search-upstream-focal-arm64.dist.yandex.ru/search-upstream-focal-arm64 stable/all/

deb http://common.dist.yandex.ru/common stable/all/
deb http://search.dist.yandex.ru/search stable/all/
deb http://search.dist.yandex.ru/search unstable/all/
deb http://search-arm64.dist.yandex.ru/search-arm64 stable/all/
deb http://search-arm64.dist.yandex.ru/search-arm64 unstable/all/
deb http://search-arm64.dist.yandex.ru/search-arm64 unstable/arm64/
deb http://search-kernel-arm64.dist.yandex.ru/search-kernel-arm64 stable/all/
deb http://search-kernel-arm64.dist.yandex.ru/search-kernel-arm64 unstable/all/
deb http://search-kernel-arm64.dist.yandex.ru/search-kernel-arm64 unstable/arm64/

deb http://cauth-arm64.dist.yandex.ru/cauth-arm64 unstable/arm64/
deb http://cauth-arm64.dist.yandex.ru/cauth-arm64 stable/arm64/
'''
APT_CONF = '''\
APT::Install-Recommends "false";
APT::Install-Suggests "false";
APT::Architecture "{arch}";
'''


class PkgNotFound(Exception):
    pass


class VerNotFound(Exception):
    pass


def pkg_short_name(name):
    idx = name.rfind(':')
    if idx > 0:
        return name[:idx]
    else:
        return name


def sudo_rm_rf(path):
    cmd = 'sudo rm -R -f -v'.split()
    cmd.append(path)
    if subprocess.call(cmd, stdout=sys.stdout, stderr=sys.stderr, stdin=None) != 0:
        raise Exception('Cannot remove path "{}" with command "{}"'.format(path, ' '.join(cmd)))


def configure_apt(path, distro='xenial', arch='amd64'):
    apt_path = os.path.join(path, 'etc', 'apt')
    os.makedirs(apt_path)
    SL_TEMPLATE = str()
    if arch == 'amd64':
        SL_TEMPLATE = SOURCES_LIST_TEMPLATE_AMD
    else:
        SL_TEMPLATE = SOURCES_LIST_TEMPLATE_ARM

    with open(os.path.join(apt_path, 'sources.list'), 'w') as f:
        f.write(SL_TEMPLATE.format(distro=distro))

    with open(os.path.join(apt_path, 'apt.conf'), 'w') as f:
        f.write(APT_CONF.format(arch=arch)

    cmd = 'sudo apt-key --keyring'.split()
    cmd.append(str(os.path.join(apt_path, 'trusted.gpg')))
    cmd.extend('adv --keyserver keyserver.ubuntu.com --recv-keys 7FCD11186050CD1A'.split())
    if subprocess.call(cmd, stdout=sys.stdout, stderr=sys.stderr, stdin=None) != 0:
        raise Exception('Cannot import trusted keyring to root "{}" with command "{}"'.format(path, ' '.join(cmd)))


class Apt(object):
    def __init__(self, root):
        var_path = os.path.join(root, 'var')
        if os.path.exists(var_path):
            shutil.rmtree(var_path)
        self.scheduled = {}
        self.cache = apt.Cache(rootdir=root)
        self.cache.update()
        self.cache.open()

    @staticmethod
    def pkg_actioned(pkg):
        return (pkg.marked_install or pkg.marked_upgrade or
                    pkg.marked_downgrade or pkg.marked_reinstall)

    def get_scheduled_pkgs(self):
        if self.scheduled:
            self.scheduled = {}
        for pkg in self.cache.get_changes():
            if self.pkg_actioned(pkg):
                self.scheduled[pkg_short_name(pkg.name)] = Install(pkg.name, pkg.candidate.version, pkg)
        return self.scheduled

    @staticmethod
    def try_mark_install(p, v, auto=True):
        p.mark_install(False, auto, True)
        p.candidate = v

    @staticmethod
    def mark_install(p, v, auto=True):
        Apt.try_mark_install(p, v, auto)
        if p.marked_keep:
            p.mark_keep()
            Apt.try_mark_install(p, v, auto)

    def try_fix(self, ver):
        for dep in ver.dependencies:
            for bdep in dep:
                if 'Depends' in bdep.rawtype and bdep.relation == '=':
                    dpkg = self.cache[bdep.name]
                    for v in dpkg.versions:
                        if v.version == bdep.version:
                            self.try_mark_install(dpkg, v, False)
                            self.try_fix(v)
                            break
        self.mark_install(ver.package, ver)

    def get_broken(self):
        return [p for p in self.cache if p.is_inst_broken]

    def find_scheduled_by_source(self, source):
        rv = []
        for pkg in self.cache.get_changes():
            if pkg.candidate.record.get('Source') == source:
                rv.append(pkg)
        return rv

    @staticmethod
    def get_strict_deps(p):
        rv = collections.defaultdict(list)
        for v in p.versions:
            for dep in v.dependencies:
                for bdep in dep:
                    if 'Depends' in bdep.rawtype and bdep.relation == '=':
                        rv[bdep.name].append(bdep.version)
        return rv

    def try_fix_broken(self, current):
        broken = self.get_broken()
        if broken:
            for pkg in broken:
                by_src = self.find_scheduled_by_source(pkg.candidate.record.get('Source'))
                if by_src:
                    idx = None
                    for i, v in enumerate(by_src):
                        if v.name == current.package.name:
                            idx = i
                            break
                    sv = None
                    if idx:
                        sv = by_src[idx].candidate.version
                    else:
                        sdeps = Apt.get_strict_deps(pkg)
                        for spkg in by_src:
                            if spkg.name != pkg.name and spkg.name in sdeps and spkg.candidate.version in sdeps[spkg.name]:
                                sv = spkg.candidate.version
                                break
                    for v in pkg.versions:
                        if sv and v.version == sv:
                            self.mark_install(pkg, v)
                            break
                    self.try_fix(pkg.candidate)

    def schedule_install(self, name, ver):
        if name in self.cache:
            pkg = self.cache[name]
            if ver:
                for v in pkg.versions:
                    if v.version == ver:
                        self.mark_install(pkg, v)
                        self.try_fix(v)
                        self.try_fix_broken(v)
                        return
            else:
                pkg.mark_install(False, True, True)
                self.try_fix_broken(pkg.candidate)
                return
            raise VerNotFound('version "{}" not found for package "{}"'.format(ver, name))
        raise PkgNotFound('package "{}" not found'.format(name))

    def apply_selections(self, selected):
        with self.cache.actiongroup():
            for p, v in selected.items():
                try:
                    log.info('Scheduling package "{}={}"'.format(p, v))
                    self.schedule_install(p, v)
                except Exception as e:
                    log.error('Cannot schedule package "{}={}": {}'.format(p, v, e))
            self.enforce_selections(selected)

    def enforce_selections(self, selected):
        scheduled = self.get_scheduled_pkgs()
        for pkg, ver in selected.items():
            if pkg not in scheduled:
                raise PkgNotFound("Package {} is not scheduled to be installed".format(pkg))
            if ver and scheduled[pkg].ver != ver:
                aptpkg = scheduled[pkg].aptpkg
                ok = False
                for v in aptpkg.versions:
                    if v.version == ver:
                        aptpkg.candidate = v
                        self.try_fix_broken(v)
                        ok = self.cache.broken_count == 0
                if not ok:
                    raise VerNotFound("Cannot enforce package {}={}".format(pkg, ver))


def cleanup_packages(pkgs):
    rv = {}
    for p, v in pkgs.items():
        if not isinstance(v, str):
            raise ValueError('Package {} has non-string version "{}"!'.format(p, v))
        rv[pkg_short_name(p)] = v
    return rv


def load_selections(f):
    with f:
        y = yaml.safe_load(f)
        if 'pkgs' in y and isinstance(y['pkgs'], dict):
            selections = cleanup_packages(y['pkgs'])
        else:
            selections = cleanup_packages(y)
        log.info('Loaded {} pkgs from selections.'.format(len(selections)))
        return selections


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('-s', '--selections', type=argparse.FileType('r'), help='selections yaml', required=True)
    parser.add_argument('-d', '--dump', type=argparse.FileType('w'), help='dump computed selections to file', required=True)
    parser.add_argument('-r', '--root', type=str, help='temporary apt root dir', required=True)
    parser.add_argument('--skip-cleanup', action='store_true', help='Do not remove temporary apt root', default=False)
    parser.add_argument('--distro', type=str, help='Distro (xenial, focal, etc...)', default='xenial')
    parser.add_argument('-a', '--arch', type=str, help='temporarily hack for arm64', default='amd64')
    return parser.parse_args()


def main(argv):
    logging.basicConfig(level=logging.DEBUG, format='%(asctime)s - %(name)s - %(levelname)s - %(message)s')
    log.info('Initializing apt.')
    configure_apt(argv.root, distro=argv.distro, arch=argv.arch)
    apt_cache = Apt(argv.root)
    log.info('Initializing apt: done.')
    selected = load_selections(argv.selections)
    apt_cache.apply_selections(selected)
    pkgs = apt_cache.get_scheduled_pkgs()
    yaml.dump({p.pkg: p.ver for p in pkgs.values()}, argv.dump, default_flow_style=False)


if __name__ == '__main__':
    argv = parse_args()
    try:
        main(argv)
    except Exception as e:
        log.exception("Main failed: {}".format(e))
    finally:
        if not argv.skip_cleanup:
            log.info("Cleaning up apt root: {}.".format(argv.root))
            sudo_rm_rf(argv.root)
    log.info('Done.')

