# coding: utf-8
import os
import sys
import json
import logging
import errno

import click
import walle_api
from yaml import safe_dump

from library.python.svn_version import svn_revision
from infra.rtc.walle_validator.lib.setup import SetupClient
from infra.rtc.walle_validator.lib.store import ConfigStore
from infra.rtc.walle_validator.lib.sync import sync_projects, get_remote_projects, sync_setup_configs
from infra.rtc.walle_validator.lib.writer import upload_projects
from infra.rtc.walle_validator.lib.transform import transform_project
from infra.rtc.walle_validator.cli.groupper import group_projects_by_field


class ListOption(click.Option):

    def type_cast_value(self, ctx, value):
        if not value:
            return None
        try:
            return [s.strip() for s in value.split(",")]
        except:
            raise click.BadParameter(value)


class State(object):

    def __init__(self, configs_dir, aux_dir, verbose, setup_token=None, walle_token=None):
        walle_token = walle_token or self._get_walle_token()
        self.client = walle_api.WalleClient(access_token=walle_token)
        setup_token = setup_token or self._get_setup_token()
        self.setup_client = SetupClient(setup_token) if setup_token else None
        self.config_store = ConfigStore(configs_dir, aux_dir)
        self.verbose = verbose

    def _get_token(self, *args):
        try:
            with open(os.path.expanduser(os.path.join(*args))) as stream:
                return stream.read().strip()
        except IOError as exc:
            if exc.errno != errno.ENOENT:
                raise
        return None

    def _get_setup_token(self):
        token = os.environ.get('SETUP_TOKEN')
        if not token:
            token = self._get_token("~", ".setup", "access_token")
        return token

    def _get_walle_token(self):
        token = os.environ.get('WALLE_TOKEN') or os.environ.get("OAUTH")
        if not token:
            token = self._get_token("~", ".wall-e", "access_token")
        return token


pass_state = click.make_pass_decorator(State)


@click.group()
@click.option('--setup-token', envvar='SETUP_OAUTH', default='',
              metavar='OAUTH', help='OAuth token for Setup.')
@click.option('--walle-token', envvar='OAUTH', default='',
              metavar='OAUTH', help='OAuth token for Wall-E.')
@click.option('--configs-dir', envvar='CONFIGS_DIR', default='../projects/configs/',
              metavar='PATH', help='Path to the projects location.')
@click.option('--aux-dir', envvar='AUX_DIR', default='../projects/auxiliaries/',
              metavar='PATH', help='Path to the aux location.')
@click.option('--verbose', '-v', is_flag=True,
              help='Enables verbose mode.')
@click.version_option(str(svn_revision()))
@click.pass_context
def cli(ctx, setup_token, walle_token, configs_dir, aux_dir, verbose):
    ctx.obj = State(os.path.abspath(configs_dir), os.path.abspath(aux_dir), verbose,
                    setup_token=setup_token, walle_token=walle_token)
    level = logging.DEBUG if ctx.obj.verbose else logging.INFO
    logging.basicConfig(level=level, format="%(message)s")


@cli.command(short_help='Update local project configs from Wall-E backend')
@pass_state
def sync(state):
    json.dump(sync_projects(state.client, state.config_store, setup_client=state.setup_client), sys.stdout)


@cli.command('sync_setup', short_help='Update local setup configs from Setup backend')
@pass_state
def sync_setup(state):
    json.dump(sync_setup_configs(config_store=state.config_store, setup_client=state.setup_client), sys.stdout)


@cli.command(short_help='Group projects by specified fields')
@click.argument('field', nargs=-1, required=True)
@click.option('--tag', '-t', cls=ListOption, default=None, help='Filter projects by specified tags')
@pass_state
def groupper(state, field, tag):
    result = group_projects_by_field(state.config_store, field, tag)
    safe_dump(result, sys.stdout, default_flow_style=False)


@cli.command(short_help='Transform local projects')
@pass_state
def transform(state):
    host_count_map = state.config_store.get_host_counts()
    for project in state.config_store.iter_projects():
        logging.info("checking project %s", project.id)
        if transform_project(project, host_count_map):
            logging.info("project %s updated", project.id)
            state.config_store.save_project(project)


@cli.command(short_help='Upload configuration to Wall-E, NOT TESTED YET!')
@click.option('--yes', '-y', is_flag=True, help='Automatically answer "yes."')
@click.option('--push', '-p', is_flag=True, help='Upload configuration to Wall-E, this option will disable dry-run mode.')
@pass_state
def upload(state, yes, push):
    logging.debug("yes: %r, push: %r", yes, push)
    host_counts = state.config_store.get_host_counts()
    local_projects_list = list(state.config_store.iter_projects())
    local_projects = {project.id: project for project in local_projects_list}
    if len(local_projects_list) != len(local_projects):
        raise click.ClickException("there are projects with same id")
    remote_projects = {project.id: project for project in get_remote_projects(state.client)}
    upload_projects(state.client, local_projects, remote_projects, host_counts, push=push, yes=yes)


if __name__ == "__main__":
    cli()
