import hashlib
import json

import click


def _md5(data):
    md5 = hashlib.md5()
    # WARNING: SORTING ONLY LIST
    if isinstance(data, list):
        data = sorted(data)
    md5.update(json.dumps(data, sort_keys=True))
    return md5.hexdigest()


def _is_inner_path(field):
    if '.' in field:
        return True
    else:
        return False


def _get_inner_path_data(data, path):
    path_list = path.split('.')
    curr_dict = None
    for k in path_list:
        if curr_dict is None:
            curr_dict = data
        curr_dict = curr_dict.get(k)
        if curr_dict is None:
            return None
    return curr_dict


def _group_data(projects, fields):
    result = {f: {} for f in fields}
    for project in projects:
        for field in fields:
            if _is_inner_path(field):
                value_field = _get_inner_path_data(vars(project), field)
            else:
                try:
                    value_field = getattr(project, field)
                except AttributeError:
                    raise click.BadParameter(field)

            value_field_hash = _md5(value_field)
            if value_field_hash not in result[field]:
                result[field][value_field_hash] = {
                    'data': value_field,
                    'projects': [project]}
            else:
                result[field][value_field_hash]['projects'].append(project)
    return result


def _collect_tags(tags=None):
    if tags is None:
        tags = ()

    exclude_tags = {tag for tag in tags if tag[0] == '-'}
    include_tags = {tag for tag in tags if tag not in exclude_tags}

    exclude_tags = {tag[1:] for tag in exclude_tags}
    include_tags = {tag[1:] if tag[0] == '+' else tag for tag in include_tags}

    return exclude_tags, include_tags


def group_projects_by_field(config_store, fields, tags=None):
    filtered_project = []
    exclude_tags, include_tags = _collect_tags(tags)

    for project in config_store.iter_projects():
        project_tags = set(project.tags)

        if not include_tags.issubset(project_tags):
            continue

        if exclude_tags & project_tags:
            continue

        filtered_project.append(project)

    gd = _group_data(filtered_project, fields)
    result = {}
    for field, data in gd.items():
        per_field_result = result[field] = {
            "unique values": len(data),
            "variants": []
        }
        variant_list = per_field_result["variants"]
        for hash_data, data in data.items():
            per_value_result = {
                "ids": ", ".join(sorted(prj.id for prj in data["projects"])),
                "value": data["data"]
            }
            variant_list.append(per_value_result)
    return result
