# coding: utf-8
from __future__ import unicode_literals


class ICondition(object):

    def match(self, project):
        raise NotImplementedError("implement me!")


class HasTag(ICondition):

    def __init__(self, tag):
        self.tag = tag

    def match(self, project):
        return self.tag in project.tags

    def __repr__(self):
        return "{} in tags".format(self.tag)


class HasNoTag(ICondition):

    def __init__(self, tag):
        self._tag = tag

    def match(self, project):
        return self._tag not in project.tags

    def __repr__(self):
        return "{} not in tags".format(self._tag)


class HasLabel(ICondition):

    def __init__(self, key, value):
        self._key = key
        self._value = value

    def match(self, project):
        return project.labels.get(self._key) == self._value

    def __repr__(self):
        return "label {!r} = {!r}".format(self._key, self._value)


class HasNoLabel(ICondition):

    def __init__(self, key):
        self._key = key

    def match(self, project):
        return self._key not in project.labels.keys()

    def __repr__(self):
        return "label {!r} != {!r}".format(self._key, self._value)


class And(ICondition):

    def __init__(self, conditions):
        self._conditions = conditions

    def match(self, project):
        return all(cond.match(project) for cond in self._conditions)

    def __repr__(self):
        return " and ".join("({!r})".format(cond) for cond in self._conditions)


class Or(ICondition):

    def __init__(self, conditions):
        self._conditions = conditions

    def match(self, project):
        return any(cond.match(project) for cond in self._conditions)

    def __repr__(self):
        return " or ".join("({!r})".format(cond) for cond in self._conditions)


class Any(ICondition):

    def match(self, project):
        return True

    def __repr__(self):
        return "any"


class Not(ICondition):

    def __init__(self, condition):
        self._condition = condition

    def match(self, project):
        return not self._condition.match(project)

    def __repr__(self):
        return "not {!r}".format(self._condition)
