# coding: utf-8
import os
import errno

from yaml import safe_dump, safe_load, Loader, SafeLoader

from infra.rtc.walle_validator.dto.project import Project
from infra.rtc.walle_validator.dto.setup import SetupConfig
from infra.rtc.walle_validator.dto.maxwell import MaxwellSpec
from infra.rtc.walle_validator.dto.automation_plot import AutomationPlot

CONFIG_SUFFIX = ".yaml"
ORLY_SUFFIX = ".yaml"
MAXWELL_SUFFIX = ".yaml"


def construct_yaml_str(self, node):
    return self.construct_scalar(node)


Loader.add_constructor(u'tag:yaml.org,2002:str', construct_yaml_str)
SafeLoader.add_constructor(u'tag:yaml.org,2002:str', construct_yaml_str)


class ConfigStore(object):

    def __init__(self, configs_dir, aux_dir, orly_rules_dir=None, maxwell_specs_dir=None):
        self._configs_dir = configs_dir
        self._aux_dir = aux_dir
        self._orly_rules_dir = orly_rules_dir
        self._maxwell_specs_dir = maxwell_specs_dir

    def iter_project_ids(self):
        for name in os.listdir(self._configs_dir):
            if name.endswith(CONFIG_SUFFIX):
                project_id = name[:-len(CONFIG_SUFFIX)]
                yield project_id

    def iter_projects(self):
        projects = []
        for project_id in self.iter_project_ids():
            with open(self._get_path_to_project(project_id), "r") as stream:
                project_dict = safe_load(stream)
            projects.append(Project(**project_dict))
        return projects

    def _get_path_to_project(self, project_id):
        return os.path.join(self._configs_dir, "{}{}".format(project_id, CONFIG_SUFFIX))

    def save_project(self, project):
        with open(self._get_path_to_project(project.id), "w") as stream:
            stream.write(safe_dump(project.to_dict(), default_flow_style=False))

    def remove_project(self, project_id):
        os.remove(self._get_path_to_project(project_id))

    def generate_project_files(self, projects):
        return [self._get_path_to_project(project_id) for project_id in projects]

    def _get_path_to_host_counts(self):
        return os.path.join(self._aux_dir, "host_counts.yaml")

    def save_host_counts(self, host_counts):
        with open(self._get_path_to_host_counts(), "w") as stream:
            stream.write(safe_dump(host_counts, default_flow_style=False))

    def get_host_counts(self):
        with open(self._get_path_to_host_counts(), "r") as stream:
            return safe_load(stream)

    def _get_path_to_setup_config(self, name):
        return os.path.join(self._aux_dir, "setup_{}.yaml".format(name))

    def generate_setup_config_files(self, names):
        return [self._get_path_to_setup_config(name) for name in names]

    def save_setup_config(self, name, config):
        with open(self._get_path_to_setup_config(name), "w") as stream:
            stream.write(safe_dump(config, default_flow_style=False))

    def get_setup_config(self, name):
        with open(self._get_path_to_setup_config(name), "r") as stream:
            return safe_load(stream)

    def _get_path_to_automation_plot(self, name):
        return os.path.join(self._aux_dir, "automation_plot_{}.yaml".format(name))

    def save_automation_plot(self, name, config):
        with open(self._get_path_to_automation_plot(name), "w") as stream:
            stream.write(safe_dump(config, default_flow_style=False))

    def get_automation_plot(self, name):
        with open(self._get_path_to_automation_plot(name), "r") as stream:
            return safe_load(stream)

    def iter_setup_configs(self):
        result = []
        deploy_configs = {project.deploy_config for project in self.iter_projects() if project.deploy_config}
        for deploy_config in deploy_configs:
            try:
                result.append(SetupConfig(deploy_config, self.get_setup_config(deploy_config)))
            except IOError as exc:
                if exc.errno != errno.ENOENT:
                    raise
        return result

    def iter_orly_rules(self):
        if self._orly_rules_dir is None:
            raise Exception("no folder for orly rules specified!")
        for name in os.listdir(self._orly_rules_dir):
            if name.endswith(ORLY_SUFFIX):
                path = os.path.join(self._orly_rules_dir, name)
                with open(path, "r") as stream:
                    rule_config = safe_load(stream)
                yield rule_config["meta"]["id"]

    def iter_maxwell_specs(self):
        if self._maxwell_specs_dir is None:
            raise Exception("no folder for maxwell specs specified!")
        for name in os.listdir(self._maxwell_specs_dir):
            if name.endswith(MAXWELL_SUFFIX):
                path = os.path.join(self._maxwell_specs_dir, name)
                with open(path, "r") as stream:
                    maxwell_spec = safe_load(stream)
                yield MaxwellSpec(maxwell_spec["name"], maxwell_spec)

    def iter_automation_plots(self):
        result = []
        automation_plots_ids = {project.automation_plot_id for project in self.iter_projects() if project.automation_plot_id}
        for automation_plot_id in automation_plots_ids:
            result.append(AutomationPlot(automation_plot_id, self.get_automation_plot(automation_plot_id)))
        return result
