# coding: utf-8
from infra.rtc.walle_validator.dto.constants import PROJECT_FIELDS
from infra.rtc.walle_validator.dto.project import Project

RTC_MARKER = "rtc"


def get_remote_projects(client):
    for project_dict in client.get_projects(fields=PROJECT_FIELDS, tags=[RTC_MARKER])["result"]:
        yield Project(**project_dict)


def sync_projects(client, config_store, setup_client=None):
    """
    :type config_store: ConfigStore
    """
    existing_projects = set(config_store.iter_project_ids())
    new_projects = set()
    removed_projects = set()
    setup_configs = set()
    automation_plot_ids = set()
    host_counts = dict()

    for project in get_remote_projects(client):
        project.cleanup()

        if project.id not in existing_projects:
            new_projects.add(project.id)
        else:
            existing_projects.remove(project.id)

        config_store.save_project(project)

        host_counts[project.id] = client.get_hosts(project=project.id, limit=0)["total"]

        if project.deploy_config:
            setup_configs.add(project.deploy_config)
        if project.automation_plot_id:
            automation_plot_ids.add(project.automation_plot_id)

    for project_id in existing_projects:
        removed_projects.add(project_id)
        config_store.remove_project(project_id)

    for automation_plot_id in automation_plot_ids:
        automation_plot_config = client.get_automation_plot(
            automation_plot_id,
            fields=["name", "owners", "checks"]
        )
        for check_config in automation_plot_config["checks"]:
            check_config.pop("start_time", None)
        config_store.save_automation_plot(automation_plot_id, automation_plot_config)

    config_store.save_host_counts(host_counts)

    result = {
        "added": config_store.generate_project_files(new_projects),
        "removed": config_store.generate_project_files(removed_projects)
    }
    if setup_client is not None:
        setup_result = sync_setup_configs(config_store, setup_client, setup_configs)
        result["added"].extend(setup_result["added"])
        result["removed"].extend(setup_result["removed"])

    return result


def sync_setup_configs(config_store, setup_client=None, setup_configs=None):
    if not setup_configs:
        setup_configs = set([project.deploy_config for project in config_store.iter_projects()])

    current_setup_configs = set([setup_config.name for setup_config in config_store.iter_setup_configs()])
    new_setup_configs = set()

    for setup_config in setup_configs:
        if setup_config not in current_setup_configs:
            new_setup_configs.add(setup_config)
        config_store.save_setup_config(setup_config, setup_client.get_config(setup_config))

    removed_setup_configs = current_setup_configs - setup_configs

    return {
        "added": config_store.generate_setup_config_files(new_setup_configs),
        "removed": config_store.generate_setup_config_files(removed_setup_configs)
    }
