# coding: utf-8

import re
from copy import deepcopy

from .utils import ProjectTransformer
from .constants import (
    DOOR_TO_RTC_GROUP,
    RTCSRE_ABC_DEVOPS_GROUP,
    RTCSUPPORT_GROUP,
    YT_ABC_DEV_GROUP,
    LOGBROKER_ABC_DEVOPS_GROUP,
    DUTYSEARCH_ABC_DEVOPS_GROUP,
    YA_AGENT_ABC_DEVOPS_GROUP,
    YT_TAG,
    YP_TAG,
    YP_MASTERS_TAG,
    YP_MASTER_TESTING_TAG,
    YP_MASTER_PRESTABLE_TAG,
    SPECIAL_REBOOT_TAG,
    SOLOMON_ABC_DEVOPS_GROUP,
    RTC_BAD_OWNERS,
    ALLOWED_RECEPIENTS
)

EXPECTED_ROLES = {
    "noc_access": [
        DOOR_TO_RTC_GROUP
    ],
    "user": [
        RTCSUPPORT_GROUP,
        SOLOMON_ABC_DEVOPS_GROUP,
    ],
    "superuser": [
        RTCSRE_ABC_DEVOPS_GROUP
    ]
}

YP_MASTER_TAGS = {
    YP_MASTERS_TAG,
    YP_MASTER_TESTING_TAG,
    YP_MASTER_PRESTABLE_TAG
}

YP_MASTER_NONSTABLE_TAGS = {
    YP_MASTER_TESTING_TAG,
    YP_MASTER_PRESTABLE_TAG
}


def is_yp_master(project):
    for tag in project.tags:
        if tag in YP_MASTER_TAGS:
            return True
    return False


def is_yp_nonprod_master(project):
    for tag in project.tags:
        if tag in YP_MASTER_NONSTABLE_TAGS:
            return True
    return False


def normalize_roles(roles):
    return {role: set(members) for role, members in roles.items()}


def transform_roles(project):
    def append_to_list(l, value):
        if value not in l:
            l.append(value)

    modified_expected_roles = deepcopy(EXPECTED_ROLES)
    for role, members in project.roles.items():
        if role not in modified_expected_roles:
            modified_expected_roles[role] = []
        target = modified_expected_roles[role]
        for member in members:
            append_to_list(target, member)

    if YT_TAG in project.tags:
        append_to_list(modified_expected_roles["user"], YT_ABC_DEV_GROUP)

    if project.labels.get("scheduler") in ("gencfg", "yp"):
        append_to_list(modified_expected_roles["user"], LOGBROKER_ABC_DEVOPS_GROUP)
        append_to_list(modified_expected_roles["user"], DUTYSEARCH_ABC_DEVOPS_GROUP)
        append_to_list(modified_expected_roles["user"], YA_AGENT_ABC_DEVOPS_GROUP)

    if normalize_roles(project.roles) != normalize_roles(modified_expected_roles):
        project.roles = modified_expected_roles
        return True

    return False


def transform_owners(project):
    touched = False
    if RTCSUPPORT_GROUP in project.owners:
        project.owners.remove(RTCSUPPORT_GROUP)
        touched = True
    if RTCSRE_ABC_DEVOPS_GROUP in project.owners:
        project.owners.remove(RTCSRE_ABC_DEVOPS_GROUP)
        touched = True
    if YT_ABC_DEV_GROUP in project.owners:
        project.owners.remove(YT_ABC_DEV_GROUP)
        touched = True
    bad_owners = set(RTC_BAD_OWNERS) & set(project.owners)
    for owner in bad_owners:
        project.owners.remove(owner)
        touched = True
    return touched


def get_yt_cluster_name(project):
    cluster_name = None
    if YP_TAG in project.tags:
        m = re.match("^yp-iss-([a-z]+)-yt-([a-z]+)(-[-0-9a-z]+)?$", project.id)
        assert m is not None, "wrong project name"
        cluster_name = m.group(2)
        if cluster_name == "seneca":
            cluster_name = "{}_{}".format(cluster_name, m.group(1))
    elif project.id in ("rtc-yt-mtn-amd", "rtc-yt-mtn", "search-yt-testing"):
        cluster_name = "arnold"
    elif project.id == "rtc-bert-mtn" or project.id == "rtc-bert-infiniband-mtn":
        cluster_name = "hahn"
    else:
        m = re.match("^rtc-yt-([a-z]+)-?([a-z]+)?(-[-a-z]+)?$", project.id)
        assert m is not None, "wrong project name"
        cluster_name = m.group(1)
        if cluster_name == "seneca":
            cluster_name = "{}_{}".format(cluster_name, m.group(2))
    return cluster_name


def transform_tags(project):
    transform = ProjectTransformer(project)
    if is_yp_master(project) and 'rtc.stage-experiment' not in project.tags:
        if YP_MASTERS_TAG in project.tags:
            reboot_tag = "rtc.reboot_segment-yp_masters"
        else:
            reboot_tag = "rtc.reboot_segment-yp_devmasters"
        transform.append_tag(reboot_tag)
        transform.remove_tag(SPECIAL_REBOOT_TAG)

    if YT_TAG in project.tags:
        if project.id != "rtc-yt-inbox" and not is_yp_master(project):
            tag = "rtc.yt_cluster-{}".format(get_yt_cluster_name(project))
            if tag not in project.tags:
                transform.append_tag(tag)

        if 'masters' in project.id.split('-'):
            reboot_tag = "rtc.reboot_segment-yt_masters"
            transform.append_tag(reboot_tag)
            transform.remove_tag(SPECIAL_REBOOT_TAG)

    if 'rtc.stage-experiment' in project.tags:
        if "yp_master_testing" in project.tags:
            reboot_tag = 'rtc.reboot_segment-yp_devmasters'
        else:
            reboot_tag = 'rtc.reboot_segment-experiment'
        protected_tags = [reboot_tag]
        suggest_by_pref = lambda x: x.startswith('rtc.reboot_segment') and x not in protected_tags
        transform.remove_multiple_tags_by_predicate(suggest_by_pref)
        transform.append_tag(reboot_tag)

    return transform.is_touched()


def transform_profile_tags(project):
    touched = False
    if 'rtc' in project.tags and project.profile_tags:
        if 'disk' in project.profile_tags:
            project.profile_tags.append('rtc')
            project.profile_tags.remove('disk')
            touched = True
    return touched


def get_max_fixes(host_count):
    assert host_count > 0
    quorum_limit = host_count // 2
    percent_limit = int(host_count * 0.1)
    if percent_limit > 0:
        return min(quorum_limit, percent_limit)
    elif quorum_limit > 0:
        return quorum_limit
    else:
        return 1


def transform_limits(project, host_count_map):
    if project.id not in host_count_map or project.id in ["rtc-preorders-lake", "rtc-yt-inbox", "search-delete"]:
        return False

    total_hosts = host_count_map[project.id]
    max_fixes = get_max_fixes(total_hosts or 1)

    dns_thresholds = project.automation_limits["max_dns_fixes"]
    if len(dns_thresholds) == 1:
        max_fixes = min(max_fixes, dns_thresholds[0]["limit"])

    limit_spec = {
        "period": "1d",
        "limit": max_fixes
    }
    if len(dns_thresholds) != 1 or limit_spec != dns_thresholds[0]:
        project.automation_limits["max_dns_fixes"] = [limit_spec]
        return True

    return False


def _remove_recepient(recipients, item):
    touched = False
    l = recipients.get(item, [])
    to_remove = set(l) - set(ALLOWED_RECEPIENTS)
    for x in to_remove:
        l.remove(x)
        touched = True
    return touched


def transform_notifications(project):
    notifications = project.notifications or {}
    recipients = notifications.get("recipients", {})
    touched = False
    if _remove_recepient(recipients, "audit"):
        touched = True
    if _remove_recepient(recipients, "bot"):
        touched = True
    if _remove_recepient(recipients, "critical"):
        touched = True
    if _remove_recepient(recipients, "error"):
        touched = True
    if _remove_recepient(recipients, "info"):
        touched = True
    if _remove_recepient(recipients, "warning"):
        touched = True
    return touched


def transform_project(project, host_count_map):
    touched = False
    if transform_roles(project):
        touched = True
    if transform_owners(project):
        touched = True
    if transform_tags(project):
        touched = True
    if transform_limits(project, host_count_map):
        touched = True
    if transform_notifications(project):
        touched = True
    if transform_profile_tags(project):
        touched = True
    return touched
