# coding: utf-8
from __future__ import print_function

import socket
import mock
import click
import logging

from deepdiff import DeepDiff

from .constants import RTC_TAG
from .errors import ValidationError


def make_diff(a, b):
    return DeepDiff(a, b)


def generate_reason():
    return "walle_validator from {}".format(socket.getfqdn())


class LazyClient(object):

    def __init__(self, client, project_id, remote_project=None):
        self._client = client
        self._project_id = project_id
        self._remote_project = remote_project

    def modify_project_automation_limits(self, project_id, **automation_limits):
        assert self._project_id == project_id
        if self._remote_project is None or self._remote_project.automation_limits != automation_limits:
            self._client.modify_project_automation_limits(id=project_id, reason=generate_reason(), **automation_limits)

    def set_provisioner_config(self, project_id, provisioner, deploy_config, deploy_tags, deploy_network, deploy_config_policy):
        assert self._project_id == project_id
        if (
            self._remote_project is None
            or self._remote_project.provisioner != provisioner
            or self._remote_project.deploy_config != deploy_config
            or self._remote_project.deploy_tags != deploy_tags
            or self._remote_project.deploy_network != deploy_network
            or self._remote_project.deploy_config_policy != deploy_config_policy
        ):
            self._client.set_provisioner_config(
                id=project_id,
                reason=generate_reason(),
                provisioner=provisioner,
                deploy_config=deploy_config,
                deploy_tags=deploy_tags,
                deploy_network=deploy_network,
                deploy_config_policy=deploy_config_policy
            )

    def set_project_dns_domain(self, project_id, dns_domain):
        assert self._project_id == project_id
        if self._remote_project is None or self._remote_project.dns_domain != dns_domain:
            self._client.set_project_dns_domain(id=project_id, reason=generate_reason(), dns_domain=dns_domain)

    def unset_project_dns_domain(self, project_id):
        assert self._project_id == project_id
        if self._remote_project is None:
            return
        if self._remote_project.dns_domain is not None:
            self._client.unset_project_dns_domain(id=project_id, reason=generate_reason())

    def set_project_vlan_scheme(self, project_id, vlan_scheme, native_vlan, extra_vlans):
        assert self._project_id == project_id
        if (
            self._remote_project is None
            or self._remote_project.vlan_scheme != vlan_scheme
            or self._remote_project.native_vlan != native_vlan
            or self._remote_project.extra_vlans != extra_vlans
        ):
            self._client.set_project_vlan_scheme(
                id=project_id,
                reason=generate_reason(),
                scheme=vlan_scheme,
                native_vlan=native_vlan,
                extra_vlans=extra_vlans
            )

    def reset_project_vlan_scheme(self, project_id):
        assert self._project_id == project_id
        if self._remote_project is None:
            return
        if self._remote_project.vlan_scheme is not None:
            self._client.reset_project_vlan_scheme(id=project_id, reason=generate_reason())

    def modify_project_notification_recipients(self, project_id, recipients):
        assert self._project_id == project_id

        remote_recipients = (
            (self._remote_project.notifications or {}).get("recipients", {})
            if self._remote_project is not None
            else {}
        )
        if (
            self._remote_project is None
            or remote_recipients != recipients
        ):
            logging.debug("method: modify_project_notification_recipients, project_id: %s, delta: %r", project_id, make_diff(remote_recipients, recipients))
            self._client.modify_project_notification_recipients(id=project_id, action="set", reason=generate_reason(), **recipients)

    def modify_project_owned_vlans(self, project_id, owned_vlans):
        assert self._project_id == project_id

        if self._remote_project is None:
            self._client.modify_project_owned_vlans(id=project_id, action="set", vlans=owned_vlans, reason=generate_reason())

        elif sorted(self._remote_project.owned_vlans) != sorted(owned_vlans):
            remote_vlans = set(self._remote_project.owned_vlans)
            local_vlans = set(owned_vlans)

            vlans_to_add = sorted(local_vlans - remote_vlans)
            if vlans_to_add:
                self._client.modify_project_owned_vlans(id=project_id, action="add", vlans=vlans_to_add, reason=generate_reason())

            vlans_to_delete = sorted(remote_vlans - local_vlans)
            if vlans_to_delete:
                self._client.modify_project_owned_vlans(id=project_id, action="remove", vlans=vlans_to_delete, reason=generate_reason())

    def modify_project_owners(self, project_id, owners):
        assert self._project_id == project_id

        if self._remote_project is None:
            self._client.modify_project_owners(id=project_id, action="set", owners=owners, reason=generate_reason())

        elif sorted(self._remote_project.owners) != sorted(owners):
            remote_owners = set(self._remote_project.owners)
            local_owners = set(owners)

            owners_to_add = sorted(local_owners - remote_owners)
            if owners_to_add:
                self._client.modify_project_owners(id=project_id, action="add", owners=owners_to_add, reason=generate_reason())

            owners_to_delete = sorted(remote_owners - local_owners)
            if owners_to_delete:
                self._client.modify_project_owners(id=project_id, action="remove", owners=owners_to_delete, reason=generate_reason())

    def set_profiling_config(self, project_id, profile, profile_tags):
        assert self._project_id == project_id
        if (
            self._remote_project is None
            or self._remote_project.profile != profile
            or self._remote_project.profile_tags != profile_tags
        ):
            self._client.set_profiling_config(id=project_id, profile=profile, profile_tags=profile_tags, reason=generate_reason())

    def set_failure_report_parameters(self, project_id, **reports):
        assert self._project_id == project_id
        if self._remote_project is None:
            self._client.set_failure_report_parameters(id=project_id, reason=generate_reason(), **reports)
        elif self._remote_project.reports != reports:
            logging.debug("method: set_failure_report_parameters, project_id: %s, delta: %r", project_id, make_diff(self._remote_project.reports, reports))
            self._client.set_failure_report_parameters(id=project_id, reason=generate_reason(), **reports)

    def remove_failure_report_parameters(self, project_id):
        assert self._project_id == project_id
        if self._remote_project is None:
            return
        if self._remote_project.reports:
            self._client.remove_failure_report_parameters(id=project_id, reason=generate_reason())

    def set_project_automation_plot(self, project_id, automation_plot_id):
        assert self._project_id == project_id
        if self._remote_project is None or self._remote_project.automation_plot_id != automation_plot_id:
            self._client.set_project_automation_plot(id=project_id, automation_plot_id=automation_plot_id, reason=generate_reason())

    def unset_project_automation_plot(self, project_id):
        assert self._project_id == project_id
        if self._remote_project is None:
            return
        if self._remote_project.automation_plot_id is not None:
            self._client.unset_project_automation_plot(id=project_id, reason=generate_reason())

    def modify_project_tags(self, project_id, tags):
        assert self._project_id == project_id

        if self._remote_project is None:
            self._client.modify_project_tags(id=project_id, action="set", tags=tags, reason=generate_reason())

        elif sorted(self._remote_project.tags) != sorted(tags):
            remote_tags = set(self._remote_project.tags)
            local_tags = set(tags)

            tags_to_add = sorted(local_tags - remote_tags)
            if tags_to_add:
                self._client.modify_project_tags(id=project_id, action="add", tags=tags_to_add, reason=generate_reason())

            tags_to_delete = sorted(remote_tags - local_tags)
            if tags_to_delete:
                self._client.modify_project_tags(id=project_id, action="remove", tags=tags_to_delete, reason=generate_reason())

    def modify_project_enable_reboot_via_ssh(self, project_id):
        if self._remote_project and self._remote_project.reboot_via_ssh:
            return
        self._client.modify_project_enable_reboot_via_ssh(id=project_id, reason=generate_reason())

    def modify_project_disable_reboot_via_ssh(self, project_id):
        if self._remote_project and not self._remote_project.reboot_via_ssh:
            return
        self._client.modify_project_disable_reboot_via_ssh(id=project_id, reason=generate_reason())

    def modify_project(self, project_id, **project_kwargs):
        assert self._project_id == project_id

        if self._remote_project is not None:
            cms_settings = {"cms": self._remote_project.cms_settings.get('cms', None)}
            if cms_settings['cms'] == "default":
                cms_settings["cms_max_busy_hosts"] = self._remote_project.cms_settings.get('cms_max_busy_hosts', None)
            else:
                cms_settings["cms_api_version"] = self._remote_project.cms_settings.get('cms_api_version')
                if self._remote_project.cms_settings.get('cms_tvm_app_id'):
                    cms_settings["cms_tvm_app_id"] = self._remote_project.cms_settings.get('cms_tvm_app_id')
            if self._remote_project.cms_settings.get('temporary_unreachable_enabled'):
                cms_settings['temporary_unreachable_enabled'] = self._remote_project.cms_settings.get('temporary_unreachable_enabled')
            remote_project_kwargs = {
                "cms_settings": cms_settings,
                "bot_project_id": self._remote_project.bot_project_id,
                "certificate_deploy": self._remote_project.certificate_deploy,
                "name": self._remote_project.name
            }

            if self._remote_project.hbf_project_id is not None:
                remote_project_kwargs["hbf_project_id"] = self._remote_project.hbf_project_id
            if self._remote_project.default_host_restrictions is not None:
                remote_project_kwargs["default_host_restrictions"] = self._remote_project.default_host_restrictions

        else:
            remote_project_kwargs = {}

        if project_kwargs != remote_project_kwargs:
            logging.debug("method: modify_project, project_id: %s, delta: %r", project_id, make_diff(remote_project_kwargs, project_kwargs))
            self._client.modify_project(id=project_id, reason=generate_reason(), **project_kwargs)

    def sync_roles(self, project_id, roles):
        assert self._project_id == project_id

        remote_roles = self._remote_project.roles if self._remote_project is not None else {}

        for role_name, members in roles.items():
            if role_name == "owner":
                continue
            elif role_name not in remote_roles:
                for member in members:
                    self._client.request_add_project_role_member(project_id=project_id, role=role_name, member=member)

        for role_name, members in remote_roles.items():
            if role_name == "owner":
                continue
            elif role_name not in roles:
                for member in members:
                    self._client.request_remove_project_role_member(project_id=project_id, role=role_name, member=member)
            else:
                remote_members = set(members)
                local_members = set(roles[role_name])
                for member in local_members - remote_members:
                    self._client.request_add_project_role_member(project_id=project_id, role=role_name, member=member)
                for member in remote_members - local_members:
                    self._client.request_remove_project_role_member(project_id=project_id, role=role_name, member=member)


def upload_one_project(client, local_project, remote_project=None):
    client = LazyClient(client, local_project.id, remote_project)

    project_kwargs = local_project.to_dict()
    project_kwargs.pop("dns_automation")
    project_kwargs.pop("healing_automation")

    project_id = project_kwargs.pop("id")

    roles = project_kwargs.pop("roles", {})
    client.sync_roles(project_id, roles)

    automation_limits = project_kwargs.pop("automation_limits", {})
    client.modify_project_automation_limits(project_id, **automation_limits)

    provisioner = project_kwargs.pop("provisioner", "lui")
    deploy_config = project_kwargs.pop("deploy_config", None)
    deploy_tags = project_kwargs.pop("deploy_tags", None)
    deploy_network = project_kwargs.pop("deploy_network", None)
    deploy_config_policy = project_kwargs.pop("deploy_config_policy", None)
    client.set_provisioner_config(project_id, provisioner, deploy_config, deploy_tags, deploy_network, deploy_config_policy)

    dns_domain = project_kwargs.pop("dns_domain", None)
    if dns_domain:
        client.set_project_dns_domain(project_id, dns_domain)
    else:
        client.unset_project_dns_domain(project_id)

    vlan_scheme = project_kwargs.pop("vlan_scheme", None)
    native_vlan = project_kwargs.pop("native_vlan", None)
    extra_vlans = project_kwargs.pop("extra_vlans", None)
    if not project_kwargs.get("hbf_project_id"):
        if vlan_scheme:
            client.set_project_vlan_scheme(project_id, vlan_scheme, native_vlan, extra_vlans)
        else:
            client.reset_project_vlan_scheme(project_id)

    notifications = project_kwargs.pop("notifications", {})
    recipients = notifications.pop("recipients", {})
    if recipients:
        client.modify_project_notification_recipients(project_id, recipients)
    else:
        client.modify_project_notification_recipients(project_id, {})

    owned_vlans = project_kwargs.pop("owned_vlans", None)
    if owned_vlans:
        client.modify_project_owned_vlans(project_id, owned_vlans)
    else:
        client.modify_project_owned_vlans(project_id, [])

    owners = project_kwargs.pop("owners", None)
    if owners:
        client.modify_project_owners(project_id, owners)
    else:
        client.modify_project_owners(project_id, [])

    profile = project_kwargs.pop("profile", None)
    profile_tags = project_kwargs.pop("profile_tags", None)
    client.set_profiling_config(project_id, profile, profile_tags)

    reports = project_kwargs.pop("reports", None)
    if reports:
        client.set_failure_report_parameters(project_id, **reports)
    else:
        client.remove_failure_report_parameters(project_id)

    automation_plot_id = project_kwargs.pop("automation_plot_id", None)
    if automation_plot_id:
        client.set_project_automation_plot(project_id, automation_plot_id)
    else:
        client.unset_project_automation_plot(project_id)

    tags = project_kwargs.pop("tags", None)
    if tags:
        client.modify_project_tags(project_id, tags)
    else:
        client.modify_project_tags(project_id, [])

    reboot_via_ssh = project_kwargs.pop("reboot_via_ssh", None)
    if reboot_via_ssh:
        client.modify_project_enable_reboot_via_ssh(project_id)
    else:
        client.modify_project_disable_reboot_via_ssh(project_id)

    cms_settings = project_kwargs.pop("cms_settings", None)
    _cms_settings = dict(cms=cms_settings.get("cms", None),
                         cms_api_version=cms_settings.get("cms_api_version", None),
                         cms_max_busy_hosts=cms_settings.get("cms_max_busy_hosts", None),
                         cms_tvm_app_id=cms_settings.get("cms_tvm_app_id", None),
                         temporary_unreachable_enabled=cms_settings.get("temporary_unreachable_enabled", None))

    cms_settings = {"cms": _cms_settings['cms']}
    if _cms_settings.get('cms') == "default":
        cms_settings["cms_max_busy_hosts"] = _cms_settings.get('cms_max_busy_hosts')
    else:
        cms_settings['cms_api_version'] = _cms_settings.get('cms_api_version')
        if _cms_settings.get('cms_tvm_app_id'):
            cms_settings["cms_tvm_app_id"] = _cms_settings.get('cms_tvm_app_id')
    if _cms_settings['temporary_unreachable_enabled']:
        cms_settings["temporary_unreachable_enabled"] = _cms_settings["temporary_unreachable_enabled"]

    project_kwargs["cms_settings"] = cms_settings

    client.modify_project(project_id, **project_kwargs)


def upload_projects(client, local_projects, remote_projects, host_counts, push=False, yes=False):
    changed_projects = {}

    created_projects = sorted(local_projects.viewkeys() - remote_projects.viewkeys())
    for created_project_id in created_projects:
        logging.info("[%s] project should be created", created_project_id)

    deleted_projects = sorted(remote_projects.viewkeys() - local_projects.viewkeys())
    for deleted_project_id in deleted_projects:
        if host_counts.get(deleted_project_id, 0) > 0:
            raise ValidationError("Project {} still have hosts, can't delete it".format(deleted_project_id))
        logging.info("[%s] project should be deleted", deleted_project_id)

    for project_id in created_projects + sorted(local_projects.viewkeys() & remote_projects.viewkeys()):
        mocked_client = mock.Mock()
        logging.debug("Checking project %s", project_id)
        upload_one_project(mocked_client, local_projects[project_id], remote_projects.get(project_id))
        if mocked_client.mock_calls:
            changed_projects[project_id] = mocked_client.mock_calls
            logging.info("[%s] project changed:", project_id)
            for call in mocked_client.mock_calls:
                logging.info("[%s]    %r", project_id, call)
            logging.info("[%s]", project_id)

    if created_projects:
        logging.warning("Created projects: %s", " ".join(created_projects))
    if deleted_projects:
        logging.warning("Deleted projects: %s", " ".join(deleted_projects))
    if changed_projects:
        logging.warning("Changed projects: %s", " ".join(sorted(changed_projects)))

    touched = bool(changed_projects or deleted_projects or created_projects)
    if push and touched:
        for created_project_id in created_projects:
            if yes or click.confirm('Do you want to create project {}?'.format(created_project_id)):
                client.add_project(
                    id=created_project_id,
                    name=created_project_id.replace("_", " ").replace("-", " "),
                    provisioner="lui",
                    deploy_config="web",
                    bot_project_id=100001955,
                    tags=[RTC_TAG],
                    reason=generate_reason()
                )
            else:
                logging.warning("[%s] creation declined", created_project_id)

        for deleted_project_id in deleted_projects:
            if yes or click.confirm('Do you want to delete project {}?'.format(deleted_project_id)):
                client.remove_project(
                    id=deleted_project_id,
                    reason=generate_reason()
                )
            else:
                logging.warning("[%s] deletion declined", deleted_project_id)

        for project_id, mock_calls in sorted(changed_projects.items()):
            logging.info("[%s] project changed:", project_id)
            for call in mock_calls:
                logging.info("[%s]     %r", project_id, call)
            if yes or click.confirm('Do you want to update project {}?'.format(project_id)):
                upload_one_project(client, local_projects[project_id], remote_projects.get(project_id))
                logging.info("[%s] project updated", project_id)
            else:
                logging.warning("[%s] update declined", project_id)
            logging.info("[%s]", project_id)

    elif touched:
        logging.warning("Ignore changes because of dry-run mode")
    else:
        logging.warning("Nothing to do")

    return touched
