import argparse
import logging
import yaml
import os

log = logging.getLogger('pkg_tool')

UNIT_HEADER = '# !!!DO NOT EDIT THIS FILE MANUALLY!!! Please use: https://a.yandex-team.ru/arc/trunk/arcadia/infra/rtc/yandex-packages\n'

def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('-i', '--input', type=str, help='input: directory with packages specs', required=True)
    return parser.parse_args()


def remove_prefix(text, prefix):
    return text[text.startswith(prefix) and len(prefix):]


def format_pkg(root_dir, name, location):
    var_name = name.replace('-', '').replace('_', '')
    # trying to cut var name
    var_name = remove_prefix(var_name, 'yandex')
    var_name = remove_prefix(var_name, 'search')
    p = os.path.join(root_dir, 'specs', name + '.yaml')
    log.info('reading {} in {}'.format(name, p))
    with open(p) as f:
        pkg_yaml = yaml.safe_load(f)
    matches = []
    l = pkg_yaml[location]
    los_keys = {'update_focal', 'update_xenial'} & {k for k in l.keys()}
    if len(los_keys) == 1:
        raise RuntimeError('Package {} should contain both update_xenial and update_focal keys or contain none of them, but contains only {} in section {}'.format(name, los_keys, location))
    dos_keys = {'update_focal', 'update_xenial'} & {k for k in pkg_yaml['default'].keys()}
    if len(dos_keys) == 1:
        raise RuntimeError('Package {} should contain both update_xenial and update_focal keys or contain none of them, but contains only {} in default section'.format(name, dos_keys))
    if len(los_keys & dos_keys) == 0 and (los_keys or dos_keys):
        raise RuntimeError('Package {} should contain both update_xenial and update_focal keys in sections {}'.format(name, (location, 'default')))
    if los_keys:
        share = l['share']
        if share == 100:
            matches.append({
                'exp': 'xenial()',
                'val': l['update_xenial'],
            })
            matches.append({
                'exp': 'focal()',
                'val': l['update_focal'],
            })
        else:
            matches.append({
                'exp': 'xenial() && perc({})'.format(l['share']),
                'val': l['update_xenial'],
            })
            matches.append({
                'exp': 'focal() && perc({})'.format(l['share']),
                'val': l['update_focal'],
            })
            default = pkg_yaml['default']
            matches.append({
                'exp': 'xenial()',
                'val': default['update_xenial'],
            })
            matches.append({
                'exp': 'focal()',
                'val': default['update_focal'],
            })
    else:
        share = l['share']
        if share == 100:
            matches.append({
                'exp': 'default()',
                'val': l['update']
            })
        else:
            matches.append({
                'exp': "perc({})".format(l['share']),
                'val': l['update']
            })
            default = pkg_yaml['default']
            matches.append({
                'exp': 'default()',
                'val': default['update']
            })
    expr = {
        'name': var_name,
        'match': matches
    }
    pkg = {
        'name': name,
        'version': '{' + var_name + '}'
    }
    return expr, pkg


def format_pkgs(root_dir, names, loc):
    exprs = []
    pkgs = []
    error = False
    for name in names:
        try:
            expr, pkg = format_pkg(root_dir, name, loc)
            exprs.append(expr)
            pkgs.append(pkg)
        except Exception as e:
            log.error(e)
            error = True
    if error:
        log.fatal('Errors validating specs... exiting...')
        exit(1)
    return exprs, pkgs


def main(argv):
    logging.basicConfig(level=logging.DEBUG, format='%(asctime)s - %(name)s - %(levelname)s - %(message)s')
    root_dir = argv.input
    main_spec_file = os.path.join(root_dir, "unit.yaml")
    specs_dir = os.path.join(root_dir, "specs")
    # check dir structure
    if not os.path.isdir(root_dir):
        raise "'{}' not a dir".format(root_dir)
    if not os.path.isfile(main_spec_file):
        raise "'{}' not a file".format(main_spec_file)
    if not os.path.isdir(specs_dir):
        raise "'{}' not a dir".format(specs_dir)
    with open(main_spec_file) as f:
        spec = yaml.safe_load(f)
    names = spec["pkgs"]
    log.info('[{}] defined on top level'.format(', '.join(names)))
    for l in ['prestable', 'sas', 'vla', 'man', 'msk']:
        exprs, pkgs = format_pkgs(root_dir, names, l)
        result_unit = {
            'meta': {
                'name': spec["name"],
                'kind': 'PackageSet',
                'version': str(spec["version"]),
                'annotations': {'stage': spec["name"]}
            },
            'spec': {'packages': pkgs}
        }
        vars = {'vars': exprs}
        with open(os.path.join(root_dir, "{}-{}.yaml".format(l, spec["name"])), 'w') as f:
            docs = [vars, result_unit]
            dump = yaml.dump_all(docs, default_flow_style=False)
            f.write(UNIT_HEADER)
            f.write(dump)


if __name__ == '__main__':
    argv = parse_args()
    try:
        main(argv)
    except Exception as e:
        log.exception("Main failed: {}".format(e))
    log.info('Done.')
