import argparse
import sys

from infra.rtc_sla_tentacles.backend.lib.clickhouse import manage_columns
from infra.rtc_sla_tentacles.backend.lib.clickhouse import database as clickhouse_database
from infra.rtc_sla_tentacles.backend.lib.clickhouse.client import ClickhouseClient
from infra.rtc_sla_tentacles.backend.lib.config.interface import ConfigInterface


def display_columns(client, table_name, action):
    if action == "config":
        print("Columns for configured database")
        for name, type_ in manage_columns.get_configured_columns(table_name, with_types=True):
            print('\t', name, "\033[31m" + type_ + "\033[0m")
    else:
        print("Columns for actual database")
        for name, type_ in manage_columns.get_database_columns(client, table_name, with_types=True):
            print('\t', name, "\033[31m" + type_ + "\033[0m")


def create_replicated_table(config_interface, table_name):
    table_object = manage_columns.TABLE_OBJECT_MAPPING[table_name]

    for node in config_interface.get_clickhouse_nodes():
        manage_columns.create_table(
            ClickhouseClient(config_interface, force_node=node),
            table_name,
            replica_name="{replica}",
            replica_table_path=f"/table_{table_name}",
            partition_key=(f"toYYYYMMDDhhmmss(toStartOfHour({table_object.date_col}))", ),
        )


def resolve_diff(client, table_name, action="show"):
    appended, removed, modified = manage_columns.get_diff_between_tables(client, table_name)
    if action == "show":
        print("Difference between actual database columns and configured")
        for i in appended:
            print('\t\033[32m+\033[0m', i[0])
        for i in removed:
            print('\t\033[31m-\033[0m', i[0])
        for i in modified:
            print('\t\033[34m*\033[0m', i[0], "\033[31m" + i[1] + "\033[0m", "->", "\033[31m" + i[2] + "\033[0m")
    if action == "add_new":
        for name, type in appended:
            client.raw(f"ALTER TABLE {table_name} ADD COLUMN {name} {type}")
    if action == "drop_old":
        for name, _ in removed:
            client.raw(f"ALTER TABLE {table_name} DROP COLUMN {name}")
    if action == "modify":
        for name, _, new_type in modified:
            client.raw(f"ALTER TABLE {table_name} MODIFY COLUMN {name} {type}")


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("-c",
                        "--conf",
                        required=True,
                        dest="CONFIG_FILE",
                        help="Path to main configuration file.")
    parser.add_argument("--host",
                        required=False,
                        dest="CH_HOST",
                        help="ClickHouse host name.")
    parser.add_argument("--port",
                        required=False,
                        dest="CH_PORT",
                        default=8443,
                        help="ClickHouse port number.")
    parser.add_argument("-t",
                        "--table",
                        dest="TABLE",
                        default=None,
                        help="ClickHouse table name")
    command_group = parser.add_mutually_exclusive_group()
    command_group.add_argument("--list",
                               help="Lists fields of database.",
                               choices=["config", "real"])
    command_group.add_argument("--diff",
                               help="Shows/resolves difference between database and config.",
                               choices=["show", "add_new", "drop_old", "modify"])
    command_group.add_argument("--create", help="Create table", choices=["create"])
    args = vars(parser.parse_args())
    sys.argv = [sys.argv[0], "-c", args["CONFIG_FILE"]]
    list_command = args.pop("list")
    diff_command = args.pop("diff")
    create_command = args.pop("create")
    config_interface = ConfigInterface()
    clickhouse_params = config_interface.get_api_config()["clickhouse_parameters"]

    # Use host and port provided from CLI.
    if args["CH_HOST"]:
        manual_host = {
            "fqdn": args["CH_HOST"],
            "port": args["CH_PORT"],
            "datacenter_name": "test-run-no-dc"
        }
        clickhouse_params["hosts"]["in_my_dc"] = [manual_host]
        clickhouse_params["hosts"]["in_other_dcs"] = []

    client = ClickhouseClient(config_interface)
    table_name = args["TABLE"]

    if not table_name:
        table_name = clickhouse_database.Tentacle.table_name()

    if list_command:
        display_columns(client, table_name, list_command)
    elif diff_command:
        resolve_diff(client, table_name, diff_command)
    elif create_command:
        create_replicated_table(config_interface, table_name)


if __name__ == '__main__':
    main()
