import argparse
import sys
import time

from infra.rtc_sla_tentacles.backend.lib.util import read_yaml_file
from infra.rtc_sla_tentacles.backend.lib.config.interface import ConfigInterface
from infra.rtc_sla_tentacles.backend.lib.config.raw_config_storage import RawConfigStorage
from infra.rtc_sla_tentacles.backend.lib.harvesters.manager import HarvestersManager
from infra.rtc_sla_tentacles.backend.lib.harvesters_snapshots.manager import HarvesterSnapshotManager
from infra.rtc_sla_tentacles.backend.lib.util import SentrySender
from infra.rtc_sla_tentacles.backend.lib.mongo.client import MongoClient


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('-c', '--config', required=True, help='Path to YAML config file', metavar='path/to.yaml')
    parser.add_argument('-l',
                        '--load-all-harvesters',
                        action="store_true",
                        required=False,
                        default=False,
                        help="Create all harvesters' instances")
    parser.add_argument('harvester_type', help='Type of the harvester')
    parser.add_argument('harvester_name', help='Name of the harvester instance')
    return parser.parse_args()


def prepare_config(args):
    config = read_yaml_file(args.config)

    # fix logging
    config['logging']['formatters']['simple'] = {
        'format': '%(asctime)s.%(msecs)03d [%(levelname)-1s] [%(name)-10s]   %(message)s',
        'datefmt': '%Y-%m-%d %H:%M:%S',
    }
    config['logging']['handlers'].clear()
    config['logging']['handlers']['common'] = {
        'class': 'logging.StreamHandler',
        'formatter': 'simple',
    }
    config['logging']['loggers'].clear()
    config['logging']['loggers'][''] = {
        'level': 'DEBUG',
        'handlers': ['common'],
    }
    config['logging']['loggers']['YpClient'] = {
        'level': 'DEBUG',
        'handlers': ['common'],
        'propagate': False,
    }

    # Optionally prune unused harvesters to speedup init
    # FIXME we should not use knowledge about config inner structure
    #       however we currently cannot create just a single instance of
    #       a single type, and any other harvesters may require additional
    #       unneeded secrets or something else
    harvester_type = args.harvester_type
    harvester_groups = config['harvesters']['harvesters']
    if harvester_type not in harvester_groups:
        print(
            f"Harvester group {harvester_type!r} is not known. Candidates are:\n" +
            ",\n".join(f" - {name}" for name in harvester_groups.keys()),
            file=sys.stderr,
        )
        raise SystemExit(1)

    if not args.load_all_harvesters:
        for group in list(harvester_groups.keys()):
            if group != harvester_type:
                del harvester_groups[group]

    harvester_name = args.harvester_name
    instances = harvester_groups[harvester_type]['arguments']
    if harvester_name not in instances:
        print(
            f"Harvester name {harvester_name!r} is not known. Candidates are:\n" +
            ",\n".join(f" - {name}" for name in instances.keys()),
            file=sys.stderr,
        )
        raise SystemExit(1)

    for instance in list(instances.keys()):
        if instance != harvester_name:
            del instances[instance]

    storage = RawConfigStorage(
        cli_args={
            'SECRETS_DIRECTORY': None,
            'MY_DATACENTER_NAME': None,
            'MONGO_DB_NAME': None,
            'CH_DB_NAME': None,
        },
        file_cfg=config,
    )
    return storage


def main():
    args = parse_args()

    storage = prepare_config(args)
    ci = ConfigInterface(storage, sentry_sender=SentrySender(manual_init=True))

    mongo_client = MongoClient(ci)
    snapshot_manager = HarvesterSnapshotManager(mongo_client)
    manager = HarvestersManager(ci, snapshot_manager)

    # FIXME do not access private property
    group = manager._harvester_groups[args.harvester_type]
    instance = group.harvesters[args.harvester_name]

    ts = int(time.time())
    instance.run(ts)
